/*
 This file is part of GNU Taler
 (C) 2021-2023 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 * @author Nic Eigel
 */

export interface WithId {
  id: string;
}

export namespace AuditorBackend {
  interface ErrorDetail {
    // Numeric error code unique to the condition.
    // The other arguments are specific to the error value reported here.
    code: number;

    // Human-readable description of the error, i.e. "missing parameter", "commitment violation", ...
    // Should give a human-readable hint about the error's nature. Optional, may change without notice!
    hint?: string;

    // Optional detail about the specific input value that failed. May change without notice!
    detail?: string;

    // Name of the parameter that was bogus (if applicable).
    parameter?: string;

    // Path to the argument that was bogus (if applicable).
    path?: string;

    // Offset of the argument that was bogus (if applicable).
    offset?: string;

    // Index of the argument that was bogus (if applicable).
    index?: string;

    // Name of the object that was bogus (if applicable).
    object?: string;

    // Name of the currency than was problematic (if applicable).
    currency?: string;

    // Expected type (if applicable).
    type_expected?: string;

    // Type that was provided instead (if applicable).
    type_actual?: string;
  }

  interface VersionResponse {
    // libtool-style representation of the Merchant protocol version, see
    // https://www.gnu.org/software/libtool/manual/html_node/Versioning.html#Versioning
    // The format is "current:revision:age".
    // Name of the protocol.

    name: "taler-auditor";
    version: string;

    // Default (!) currency supported by this backend.
    // This is the currency that the backend should
    // suggest by default to the user when entering
    // amounts. See currencies for a list of
    // supported currencies and how to render them.
    implementation: string;
    currency: string;
    auditor_public_key: string;
    exchange_master_public_key: string;

    // How services should render currencies supported
    // by this backend.  Maps
    // currency codes (e.g. "EUR" or "KUDOS") to
    // the respective currency specification.
    // All currencies in this map are supported by
    // the backend.  Note that the actual currency
    // specifications are a *hint* for applications
    // that would like *advice* on how to render amounts.
    // Applications *may* ignore the currency specification
    // if they know how to render currencies that they are
    // used with.
    //currencies: { currency: CurrencySpecification };

    // Array of exchanges trusted by the merchant.
    // Since protocol v6.
    // exchanges: ExchangeConfigInfo[];
  }

  namespace AmountArithmeticInconsistency {
    class ClassAmountArithmeticInconsistency {
      data: AmountArithmeticInconsistencyDetail[];
    }

    interface SummaryResponse {
      amount_arithmetic_inconsistency: AmountArithmeticInconsistencyDetail[];
    }

    interface AmountArithmeticInconsistencyDetail {
      row_id: number;
      operation: string;
      exchange_amount: string;
      auditor_amount: string;
      profitable: boolean;
      suppressed: boolean;
    }
  }

  namespace WireFormatInconsistency {
    class ClassWireFormatInconsistency {
      data: WireFormatInconsistencyDetail[];
    }

    interface SummaryResponse {
      responseData: WireFormatInconsistencyDetail[];
    }

    interface WireFormatInconsistencyDetail {
      row_id: number;
      amount: string;
      wire_offset: string;
      diagnostic: string;
      suppressed: boolean;
    }
  }
}
