/* rgxg - ReGular eXpression Generator
 *
 * Copyright (c) 2019 Hannes von Haugwitz
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute
 * it freely, subject to the following restrictions:
 *
 *     1. The origin of this software must not be misrepresented; you
 *     must not claim that you wrote the original software. If you use
 *     this software in a product, an acknowledgment in the product
 *     documentation would be appreciated but is not required.
 *
 *     2. Altered source versions must be plainly marked as such, and
 *     must not be misrepresented as being the original software.
 *
 *     3. This notice may not be removed or altered from any source
 *     distribution.
 */

#include <check.h>

/* needed for malloc, free */
#include <stdlib.h>

#include "rgxg/net.h"

typedef struct {
    const char *cidr;
    const char *expected_regex;
    rgxg_options_t options;
} cidr_test_t;

static cidr_test_t cidr_regex[] = {
    { "0.0.0.0/0",  "(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3}", 0 },
    { "0.0.0.0/1",  "(12[0-7]|1[01][0-9]|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3}", 0 },
    { "0.0.0.0/2",  "(6[0-3]|[1-5]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3}", 0 },
    { "0.0.0.0/3",  "(3[01]|[12]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3}", 0 },
    { "0.0.0.0/4",  "(1[0-5]|[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3}", 0 },
    { "0.0.0.0/5",  "[0-7](\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3}", 0 },
    { "0.0.0.0/6",  "[0-3](\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3}", 0 },
    { "0.0.0.0/7",  "[01](\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3}", 0 },

    { "0.0.0.0/8",  "0(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3}", 0 },
    { "0.0.0.0/9",  "0\\.(12[0-7]|1[01][0-9]|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2}", 0 },
    { "0.0.0.0/10", "0\\.(6[0-3]|[1-5]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2}", 0 },
    { "0.0.0.0/11", "0\\.(3[01]|[12]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2}", 0 },
    { "0.0.0.0/12", "0\\.(1[0-5]|[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2}", 0 },
    { "0.0.0.0/13", "0\\.[0-7](\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2}", 0 },
    { "0.0.0.0/14", "0\\.[0-3](\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2}", 0 },
    { "0.0.0.0/15", "0\\.[01](\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2}", 0 },

    { "0.0.0.0/16", "0\\.0(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2}", 0 },
    { "0.0.0.0/18", "0\\.0\\.(6[0-3]|[1-5]?[0-9])\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])", 0 },
    { "0.0.0.0/19", "0\\.0\\.(3[01]|[12]?[0-9])\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])", 0 },
    { "0.0.0.0/20", "0\\.0\\.(1[0-5]|[0-9])\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])", 0 },
    { "0.0.0.0/21", "0\\.0\\.[0-7]\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])", 0 },
    { "0.0.0.0/22", "0\\.0\\.[0-3]\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])", 0 },
    { "0.0.0.0/23", "0\\.0\\.[01]\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])", 0 },
    { "0.0.0.0/24", "0\\.0\\.0\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])", 0 },

    { "0.0.0.0/25", "0\\.0\\.0\\.(12[0-7]|1[01][0-9]|[1-9]?[0-9])", 0 },
    { "0.0.0.0/26", "0\\.0\\.0\\.(6[0-3]|[1-5]?[0-9])", 0 },
    { "0.0.0.0/27", "0\\.0\\.0\\.(3[01]|[12]?[0-9])", 0 },
    { "0.0.0.0/28", "0\\.0\\.0\\.(1[0-5]|[0-9])", 0 },
    { "0.0.0.0/29", "0\\.0\\.0\\.[0-7]", 0 },
    { "0.0.0.0/30", "0\\.0\\.0\\.[0-3]", 0 },
    { "0.0.0.0/31", "0\\.0\\.0\\.[01]", 0 },
    { "0.0.0.0/32", "0\\.0\\.0\\.0", 0 },

    { "192.0.2.232/0", "(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3}", 0 },
    { "192.0.2.232/1", "(25[0-5]|2[0-4][0-9]|1[3-9][0-9]|12[89])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3}", 0 },
    { "192.0.2.232/2", "(25[0-5]|2[0-4][0-9]|19[2-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3}", 0 },
    { "192.0.2.232/3", "(22[0-3]|2[01][0-9]|19[2-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3}", 0 },
    { "192.0.2.232/4", "(20[0-7]|19[2-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3}", 0 },
    { "192.0.2.232/5", "19[2-9](\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3}", 0 },
    { "192.0.2.232/6", "19[2-5](\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3}", 0 },
    { "192.0.2.232/7", "19[23](\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3}", 0 },

    { "192.0.2.232/8",  "192(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3}", 0 },
    { "192.0.2.232/9",  "192\\.(12[0-7]|1[01][0-9]|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2}", 0 },
    { "192.0.2.232/10", "192\\.(6[0-3]|[1-5]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2}", 0 },
    { "192.0.2.232/11", "192\\.(3[01]|[12]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2}", 0 },
    { "192.0.2.232/12", "192\\.(1[0-5]|[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2}", 0 },
    { "192.0.2.232/13", "192\\.[0-7](\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2}", 0 },
    { "192.0.2.232/14", "192\\.[0-3](\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2}", 0 },
    { "192.0.2.232/15", "192\\.[01](\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2}", 0 },

    { "192.0.2.232/16", "192\\.0(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2}", 0 },
    { "192.0.2.232/18", "192\\.0\\.(6[0-3]|[1-5]?[0-9])\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])", 0 },
    { "192.0.2.232/19", "192\\.0\\.(3[01]|[12]?[0-9])\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])", 0 },
    { "192.0.2.232/20", "192\\.0\\.(1[0-5]|[0-9])\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])", 0 },
    { "192.0.2.232/21", "192\\.0\\.[0-7]\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])", 0 },
    { "192.0.2.232/22", "192\\.0\\.[0-3]\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])", 0 },
    { "192.0.2.232/23", "192\\.0\\.[23]\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])", 0 },
    { "192.0.2.232/24", "192\\.0\\.2\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])", 0 },

    { "192.0.2.232/25", "192\\.0\\.2\\.(25[0-5]|2[0-4][0-9]|1[3-9][0-9]|12[89])", 0 },
    { "192.0.2.232/26", "192\\.0\\.2\\.(25[0-5]|2[0-4][0-9]|19[2-9])", 0 },
    { "192.0.2.232/27", "192\\.0\\.2\\.(25[0-5]|2[34][0-9]|22[4-9])", 0 },
    { "192.0.2.232/28", "192\\.0\\.2\\.(23[0-9]|22[4-9])", 0 },
    { "192.0.2.232/29", "192\\.0\\.2\\.23[2-9]", 0 },
    { "192.0.2.232/30", "192\\.0\\.2\\.23[2-5]", 0 },
    { "192.0.2.232/31", "192\\.0\\.2\\.23[23]", 0 },
    { "192.0.2.232/32", "192\\.0\\.2\\.232", 0 },

    { "0::/128", "0?0?0?0(:0?0?0?0){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "0:0::/128", "0?0?0?0(:0?0?0?0){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "0:0:0::/128", "0?0?0?0(:0?0?0?0){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "0:0:0:0::/128", "0?0?0?0(:0?0?0?0){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "0:0:0:0:0::/128", "0?0?0?0(:0?0?0?0){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "0:0:0:0:0:0::/128", "0?0?0?0(:0?0?0?0){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "0:0:0:0:0:0:0::/128", "0?0?0?0(:0?0?0?0){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "0:0:0:0:0:0::0/128", "0?0?0?0(:0?0?0?0){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "0:0:0:0:0::0:0/128", "0?0?0?0(:0?0?0?0){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "0:0:0:0::0:0:0/128", "0?0?0?0(:0?0?0?0){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "0:0:0:0:0:0:0:0/128", "0?0?0?0(:0?0?0?0){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "0:0:0::0:0:0:0/128", "0?0?0?0(:0?0?0?0){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "0:0::0:0:0:0:0/128", "0?0?0?0(:0?0?0?0){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "0::0:0:0:0:0:0/128", "0?0?0?0(:0?0?0?0){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0:0:0:0:0:0:0/128", "0?0?0?0(:0?0?0?0){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0:0:0:0:0:0/128", "0?0?0?0(:0?0?0?0){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0:0:0:0:0/128", "0?0?0?0(:0?0?0?0){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0:0:0:0/128", "0?0?0?0(:0?0?0?0){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0:0:0/128", "0?0?0?0(:0?0?0?0){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0:0/128", "0?0?0?0(:0?0?0?0){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/128", "0?0?0?0(:0?0?0?0){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/128", "0?0?0?0(:0?0?0?0){5}((:0?0?0?0){2}|:0\\.0\\.0\\.0)", RGXG_NOZEROCOMPRESSION },
    { "::0/128", "(:(:0?0?0?0){1,7}|::|0?0?0?0(:(:0?0?0?0){1,6}|::|:0?0?0?0(:(:0?0?0?0){1,5}|::|:0?0?0?0(:(:0?0?0?0){1,4}|::|:0?0?0?0(:(:0?0?0?0){1,3}|::|:0?0?0?0(:(:0?0?0?0){1,2}|::|:0?0?0?0(::0?0?0?0|::|:0?0?0?0(::|:0?0?0?0))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/128", "((:(:0?0?0?0){1,7}|::|0?0?0?0(:(:0?0?0?0){1,6}|::|:0?0?0?0(:(:0?0?0?0){1,5}|::|:0?0?0?0(:(:0?0?0?0){1,4}|::|:0?0?0?0(:(:0?0?0?0){1,3}|::|:0?0?0?0(:(:0?0?0?0){1,2}|::|:0?0?0?0(::0?0?0?0|::|:0?0?0?0(::|:0?0?0?0))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):0\\.0\\.0\\.0)", 0 },
    { "::0/127", "0?0?0?0(:0?0?0?0){6}:0?0?0?[01]", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/127", "0?0?0?0(:0?0?0?0){5}(:0?0?0?0:0?0?0?[01]|:0\\.0\\.0\\.[01])", RGXG_NOZEROCOMPRESSION },
    { "::0/127", "(:(:0?0?0?0){0,6}:0?0?0?[01]|::|0?0?0?0(:(:0?0?0?0){0,5}:0?0?0?[01]|::|:0?0?0?0(:(:0?0?0?0){0,4}:0?0?0?[01]|::|:0?0?0?0(:(:0?0?0?0){0,3}:0?0?0?[01]|::|:0?0?0?0(:(:0?0?0?0){0,2}:0?0?0?[01]|::|:0?0?0?0(:(:0?0?0?0)?:0?0?0?[01]|::|:0?0?0?0(::0?0?0?[01]|::|:0?0?0?0(::|:0?0?0?[01]))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/127", "((:(:0?0?0?0){0,6}:0?0?0?[01]|::|0?0?0?0(:(:0?0?0?0){0,5}:0?0?0?[01]|::|:0?0?0?0(:(:0?0?0?0){0,4}:0?0?0?[01]|::|:0?0?0?0(:(:0?0?0?0){0,3}:0?0?0?[01]|::|:0?0?0?0(:(:0?0?0?0){0,2}:0?0?0?[01]|::|:0?0?0?0(:(:0?0?0?0)?:0?0?0?[01]|::|:0?0?0?0(::0?0?0?[01]|::|:0?0?0?0(::|:0?0?0?[01]))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):0\\.0\\.0\\.[01])", 0 },
    { "::0/126", "0?0?0?0(:0?0?0?0){6}:0?0?0?[0-3]", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/126", "0?0?0?0(:0?0?0?0){5}(:0?0?0?0:0?0?0?[0-3]|:0\\.0\\.0\\.[0-3])", RGXG_NOZEROCOMPRESSION },
    { "::0/126", "(:(:0?0?0?0){0,6}:0?0?0?[0-3]|::|0?0?0?0(:(:0?0?0?0){0,5}:0?0?0?[0-3]|::|:0?0?0?0(:(:0?0?0?0){0,4}:0?0?0?[0-3]|::|:0?0?0?0(:(:0?0?0?0){0,3}:0?0?0?[0-3]|::|:0?0?0?0(:(:0?0?0?0){0,2}:0?0?0?[0-3]|::|:0?0?0?0(:(:0?0?0?0)?:0?0?0?[0-3]|::|:0?0?0?0(::0?0?0?[0-3]|::|:0?0?0?0(::|:0?0?0?[0-3]))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/126", "((:(:0?0?0?0){0,6}:0?0?0?[0-3]|::|0?0?0?0(:(:0?0?0?0){0,5}:0?0?0?[0-3]|::|:0?0?0?0(:(:0?0?0?0){0,4}:0?0?0?[0-3]|::|:0?0?0?0(:(:0?0?0?0){0,3}:0?0?0?[0-3]|::|:0?0?0?0(:(:0?0?0?0){0,2}:0?0?0?[0-3]|::|:0?0?0?0(:(:0?0?0?0)?:0?0?0?[0-3]|::|:0?0?0?0(::0?0?0?[0-3]|::|:0?0?0?0(::|:0?0?0?[0-3]))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):0\\.0\\.0\\.[0-3])", 0 },
    { "::0/125", "0?0?0?0(:0?0?0?0){6}:0?0?0?[0-7]", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/125", "0?0?0?0(:0?0?0?0){5}(:0?0?0?0:0?0?0?[0-7]|:0\\.0\\.0\\.[0-7])", RGXG_NOZEROCOMPRESSION },
    { "::0/125", "(:(:0?0?0?0){0,6}:0?0?0?[0-7]|::|0?0?0?0(:(:0?0?0?0){0,5}:0?0?0?[0-7]|::|:0?0?0?0(:(:0?0?0?0){0,4}:0?0?0?[0-7]|::|:0?0?0?0(:(:0?0?0?0){0,3}:0?0?0?[0-7]|::|:0?0?0?0(:(:0?0?0?0){0,2}:0?0?0?[0-7]|::|:0?0?0?0(:(:0?0?0?0)?:0?0?0?[0-7]|::|:0?0?0?0(::0?0?0?[0-7]|::|:0?0?0?0(::|:0?0?0?[0-7]))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/125", "((:(:0?0?0?0){0,6}:0?0?0?[0-7]|::|0?0?0?0(:(:0?0?0?0){0,5}:0?0?0?[0-7]|::|:0?0?0?0(:(:0?0?0?0){0,4}:0?0?0?[0-7]|::|:0?0?0?0(:(:0?0?0?0){0,3}:0?0?0?[0-7]|::|:0?0?0?0(:(:0?0?0?0){0,2}:0?0?0?[0-7]|::|:0?0?0?0(:(:0?0?0?0)?:0?0?0?[0-7]|::|:0?0?0?0(::0?0?0?[0-7]|::|:0?0?0?0(::|:0?0?0?[0-7]))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):0\\.0\\.0\\.[0-7])", 0 },
    { "::0/124", "0?0?0?0(:0?0?0?0){6}:0?0?0?[0-9A-Fa-f]", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/124", "0?0?0?0(:0?0?0?0){5}(:0?0?0?0:0?0?0?[0-9A-Fa-f]|:0\\.0\\.0\\.(1[0-5]|[0-9]))", RGXG_NOZEROCOMPRESSION },
    { "::0/124", "(:(:0?0?0?0){0,6}:0?0?0?[0-9A-Fa-f]|::|0?0?0?0(:(:0?0?0?0){0,5}:0?0?0?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0){0,4}:0?0?0?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0){0,3}:0?0?0?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0){0,2}:0?0?0?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0)?:0?0?0?[0-9A-Fa-f]|::|:0?0?0?0(::0?0?0?[0-9A-Fa-f]|::|:0?0?0?0(::|:0?0?0?[0-9A-Fa-f]))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/124", "((:(:0?0?0?0){0,6}:0?0?0?[0-9A-Fa-f]|::|0?0?0?0(:(:0?0?0?0){0,5}:0?0?0?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0){0,4}:0?0?0?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0){0,3}:0?0?0?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0){0,2}:0?0?0?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0)?:0?0?0?[0-9A-Fa-f]|::|:0?0?0?0(::0?0?0?[0-9A-Fa-f]|::|:0?0?0?0(::|:0?0?0?[0-9A-Fa-f]))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):0\\.0\\.0\\.(1[0-5]|[0-9]))", 0 },
    { "::0/123", "0?0?0?0(:0?0?0?0){6}:0?0?[01]?[0-9A-Fa-f]", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/123", "0?0?0?0(:0?0?0?0){5}(:0?0?0?0:0?0?[01]?[0-9A-Fa-f]|:0\\.0\\.0\\.(3[01]|[12]?[0-9]))", RGXG_NOZEROCOMPRESSION },
    { "::0/123", "(:(:0?0?0?0){0,6}:0?0?[01]?[0-9A-Fa-f]|::|0?0?0?0(:(:0?0?0?0){0,5}:0?0?[01]?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0){0,4}:0?0?[01]?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0){0,3}:0?0?[01]?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0){0,2}:0?0?[01]?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0)?:0?0?[01]?[0-9A-Fa-f]|::|:0?0?0?0(::0?0?[01]?[0-9A-Fa-f]|::|:0?0?0?0(::|:0?0?[01]?[0-9A-Fa-f]))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/123", "((:(:0?0?0?0){0,6}:0?0?[01]?[0-9A-Fa-f]|::|0?0?0?0(:(:0?0?0?0){0,5}:0?0?[01]?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0){0,4}:0?0?[01]?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0){0,3}:0?0?[01]?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0){0,2}:0?0?[01]?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0)?:0?0?[01]?[0-9A-Fa-f]|::|:0?0?0?0(::0?0?[01]?[0-9A-Fa-f]|::|:0?0?0?0(::|:0?0?[01]?[0-9A-Fa-f]))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):0\\.0\\.0\\.(3[01]|[12]?[0-9]))", 0 },
    { "::0/122", "0?0?0?0(:0?0?0?0){6}:0?0?[0-3]?[0-9A-Fa-f]", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/122", "0?0?0?0(:0?0?0?0){5}(:0?0?0?0:0?0?[0-3]?[0-9A-Fa-f]|:0\\.0\\.0\\.(6[0-3]|[1-5]?[0-9]))", RGXG_NOZEROCOMPRESSION },
    { "::0/122", "(:(:0?0?0?0){0,6}:0?0?[0-3]?[0-9A-Fa-f]|::|0?0?0?0(:(:0?0?0?0){0,5}:0?0?[0-3]?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0){0,4}:0?0?[0-3]?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0){0,3}:0?0?[0-3]?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0){0,2}:0?0?[0-3]?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0)?:0?0?[0-3]?[0-9A-Fa-f]|::|:0?0?0?0(::0?0?[0-3]?[0-9A-Fa-f]|::|:0?0?0?0(::|:0?0?[0-3]?[0-9A-Fa-f]))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/122", "((:(:0?0?0?0){0,6}:0?0?[0-3]?[0-9A-Fa-f]|::|0?0?0?0(:(:0?0?0?0){0,5}:0?0?[0-3]?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0){0,4}:0?0?[0-3]?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0){0,3}:0?0?[0-3]?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0){0,2}:0?0?[0-3]?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0)?:0?0?[0-3]?[0-9A-Fa-f]|::|:0?0?0?0(::0?0?[0-3]?[0-9A-Fa-f]|::|:0?0?0?0(::|:0?0?[0-3]?[0-9A-Fa-f]))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):0\\.0\\.0\\.(6[0-3]|[1-5]?[0-9]))", 0 },
    { "::0/121", "0?0?0?0(:0?0?0?0){6}:0?0?[0-7]?[0-9A-Fa-f]", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/121", "0?0?0?0(:0?0?0?0){5}(:0?0?0?0:0?0?[0-7]?[0-9A-Fa-f]|:0\\.0\\.0\\.(12[0-7]|1[01][0-9]|[1-9]?[0-9]))", RGXG_NOZEROCOMPRESSION },
    { "::0/121", "(:(:0?0?0?0){0,6}:0?0?[0-7]?[0-9A-Fa-f]|::|0?0?0?0(:(:0?0?0?0){0,5}:0?0?[0-7]?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0){0,4}:0?0?[0-7]?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0){0,3}:0?0?[0-7]?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0){0,2}:0?0?[0-7]?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0)?:0?0?[0-7]?[0-9A-Fa-f]|::|:0?0?0?0(::0?0?[0-7]?[0-9A-Fa-f]|::|:0?0?0?0(::|:0?0?[0-7]?[0-9A-Fa-f]))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/121", "((:(:0?0?0?0){0,6}:0?0?[0-7]?[0-9A-Fa-f]|::|0?0?0?0(:(:0?0?0?0){0,5}:0?0?[0-7]?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0){0,4}:0?0?[0-7]?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0){0,3}:0?0?[0-7]?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0){0,2}:0?0?[0-7]?[0-9A-Fa-f]|::|:0?0?0?0(:(:0?0?0?0)?:0?0?[0-7]?[0-9A-Fa-f]|::|:0?0?0?0(::0?0?[0-7]?[0-9A-Fa-f]|::|:0?0?0?0(::|:0?0?[0-7]?[0-9A-Fa-f]))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):0\\.0\\.0\\.(12[0-7]|1[01][0-9]|[1-9]?[0-9]))", 0 },
    { "::0/120", "0?0?0?0(:0?0?0?0){6}:0?0?[0-9A-Fa-f]{1,2}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/120", "0?0?0?0(:0?0?0?0){5}(:0?0?0?0:0?0?[0-9A-Fa-f]{1,2}|:0\\.0\\.0\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9]))", RGXG_NOZEROCOMPRESSION },
    { "::0/120", "(:(:0?0?0?0){0,6}:0?0?[0-9A-Fa-f]{1,2}|::|0?0?0?0(:(:0?0?0?0){0,5}:0?0?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0){0,4}:0?0?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0){0,3}:0?0?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}:0?0?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0)?:0?0?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(::0?0?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(::|:0?0?[0-9A-Fa-f]{1,2}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/120", "((:(:0?0?0?0){0,6}:0?0?[0-9A-Fa-f]{1,2}|::|0?0?0?0(:(:0?0?0?0){0,5}:0?0?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0){0,4}:0?0?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0){0,3}:0?0?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}:0?0?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0)?:0?0?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(::0?0?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(::|:0?0?[0-9A-Fa-f]{1,2}))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):0\\.0\\.0\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9]))", 0 },
    { "::0/119", "0?0?0?0(:0?0?0?0){6}:0?[01]?[0-9A-Fa-f]{1,2}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/119", "0?0?0?0(:0?0?0?0){5}(:0?0?0?0:0?[01]?[0-9A-Fa-f]{1,2}|:0\\.0\\.[01]\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9]))", RGXG_NOZEROCOMPRESSION },
    { "::0/119", "(:(:0?0?0?0){0,6}:0?[01]?[0-9A-Fa-f]{1,2}|::|0?0?0?0(:(:0?0?0?0){0,5}:0?[01]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0){0,4}:0?[01]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0){0,3}:0?[01]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}:0?[01]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0)?:0?[01]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(::0?[01]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(::|:0?[01]?[0-9A-Fa-f]{1,2}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/119", "((:(:0?0?0?0){0,6}:0?[01]?[0-9A-Fa-f]{1,2}|::|0?0?0?0(:(:0?0?0?0){0,5}:0?[01]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0){0,4}:0?[01]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0){0,3}:0?[01]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}:0?[01]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0)?:0?[01]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(::0?[01]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(::|:0?[01]?[0-9A-Fa-f]{1,2}))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):0\\.0\\.[01]\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9]))", 0 },
    { "::0/118", "0?0?0?0(:0?0?0?0){6}:0?[0-3]?[0-9A-Fa-f]{1,2}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/118", "0?0?0?0(:0?0?0?0){5}(:0?0?0?0:0?[0-3]?[0-9A-Fa-f]{1,2}|:0\\.0\\.[0-3]\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9]))", RGXG_NOZEROCOMPRESSION },
    { "::0/118", "(:(:0?0?0?0){0,6}:0?[0-3]?[0-9A-Fa-f]{1,2}|::|0?0?0?0(:(:0?0?0?0){0,5}:0?[0-3]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0){0,4}:0?[0-3]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0){0,3}:0?[0-3]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}:0?[0-3]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0)?:0?[0-3]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(::0?[0-3]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(::|:0?[0-3]?[0-9A-Fa-f]{1,2}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/118", "((:(:0?0?0?0){0,6}:0?[0-3]?[0-9A-Fa-f]{1,2}|::|0?0?0?0(:(:0?0?0?0){0,5}:0?[0-3]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0){0,4}:0?[0-3]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0){0,3}:0?[0-3]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}:0?[0-3]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0)?:0?[0-3]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(::0?[0-3]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(::|:0?[0-3]?[0-9A-Fa-f]{1,2}))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):0\\.0\\.[0-3]\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9]))", 0 },
    { "::0/117", "0?0?0?0(:0?0?0?0){6}:0?[0-7]?[0-9A-Fa-f]{1,2}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/117", "0?0?0?0(:0?0?0?0){5}(:0?0?0?0:0?[0-7]?[0-9A-Fa-f]{1,2}|:0\\.0\\.[0-7]\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9]))", RGXG_NOZEROCOMPRESSION },
    { "::0/117", "(:(:0?0?0?0){0,6}:0?[0-7]?[0-9A-Fa-f]{1,2}|::|0?0?0?0(:(:0?0?0?0){0,5}:0?[0-7]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0){0,4}:0?[0-7]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0){0,3}:0?[0-7]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}:0?[0-7]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0)?:0?[0-7]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(::0?[0-7]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(::|:0?[0-7]?[0-9A-Fa-f]{1,2}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/117", "((:(:0?0?0?0){0,6}:0?[0-7]?[0-9A-Fa-f]{1,2}|::|0?0?0?0(:(:0?0?0?0){0,5}:0?[0-7]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0){0,4}:0?[0-7]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0){0,3}:0?[0-7]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}:0?[0-7]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(:(:0?0?0?0)?:0?[0-7]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(::0?[0-7]?[0-9A-Fa-f]{1,2}|::|:0?0?0?0(::|:0?[0-7]?[0-9A-Fa-f]{1,2}))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):0\\.0\\.[0-7]\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9]))", 0 },
    { "::0/116", "0?0?0?0(:0?0?0?0){6}:0?[0-9A-Fa-f]{1,3}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/116", "0?0?0?0(:0?0?0?0){5}(:0?0?0?0:0?[0-9A-Fa-f]{1,3}|:0\\.0\\.(1[0-5]|[0-9])\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9]))", RGXG_NOZEROCOMPRESSION },
    { "::0/116", "(:(:0?0?0?0){0,6}:0?[0-9A-Fa-f]{1,3}|::|0?0?0?0(:(:0?0?0?0){0,5}:0?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0){0,4}:0?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0){0,3}:0?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0){0,2}:0?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0)?:0?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(::0?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(::|:0?[0-9A-Fa-f]{1,3}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/116", "((:(:0?0?0?0){0,6}:0?[0-9A-Fa-f]{1,3}|::|0?0?0?0(:(:0?0?0?0){0,5}:0?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0){0,4}:0?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0){0,3}:0?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0){0,2}:0?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0)?:0?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(::0?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(::|:0?[0-9A-Fa-f]{1,3}))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):0\\.0\\.(1[0-5]|[0-9])\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9]))", 0 },
    { "::0/115", "0?0?0?0(:0?0?0?0){6}:[01]?[0-9A-Fa-f]{1,3}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/115", "0?0?0?0(:0?0?0?0){5}(:0?0?0?0:[01]?[0-9A-Fa-f]{1,3}|:0\\.0\\.(3[01]|[12]?[0-9])\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9]))", RGXG_NOZEROCOMPRESSION },
    { "::0/115", "(:(:0?0?0?0){0,6}:[01]?[0-9A-Fa-f]{1,3}|::|0?0?0?0(:(:0?0?0?0){0,5}:[01]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0){0,4}:[01]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0){0,3}:[01]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0){0,2}:[01]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0)?:[01]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(::[01]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(::|:[01]?[0-9A-Fa-f]{1,3}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/115", "((:(:0?0?0?0){0,6}:[01]?[0-9A-Fa-f]{1,3}|::|0?0?0?0(:(:0?0?0?0){0,5}:[01]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0){0,4}:[01]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0){0,3}:[01]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0){0,2}:[01]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0)?:[01]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(::[01]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(::|:[01]?[0-9A-Fa-f]{1,3}))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):0\\.0\\.(3[01]|[12]?[0-9])\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9]))", 0 },
    { "::0/114", "0?0?0?0(:0?0?0?0){6}:[0-3]?[0-9A-Fa-f]{1,3}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/114", "0?0?0?0(:0?0?0?0){5}(:0?0?0?0:[0-3]?[0-9A-Fa-f]{1,3}|:0\\.0\\.(6[0-3]|[1-5]?[0-9])\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9]))", RGXG_NOZEROCOMPRESSION },
    { "::0/114", "(:(:0?0?0?0){0,6}:[0-3]?[0-9A-Fa-f]{1,3}|::|0?0?0?0(:(:0?0?0?0){0,5}:[0-3]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0){0,4}:[0-3]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0){0,3}:[0-3]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0){0,2}:[0-3]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0)?:[0-3]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(::[0-3]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(::|:[0-3]?[0-9A-Fa-f]{1,3}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/114", "((:(:0?0?0?0){0,6}:[0-3]?[0-9A-Fa-f]{1,3}|::|0?0?0?0(:(:0?0?0?0){0,5}:[0-3]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0){0,4}:[0-3]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0){0,3}:[0-3]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0){0,2}:[0-3]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0)?:[0-3]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(::[0-3]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(::|:[0-3]?[0-9A-Fa-f]{1,3}))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):0\\.0\\.(6[0-3]|[1-5]?[0-9])\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9]))", 0 },
    { "::0/113", "0?0?0?0(:0?0?0?0){6}:[0-7]?[0-9A-Fa-f]{1,3}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/113", "0?0?0?0(:0?0?0?0){5}(:0?0?0?0:[0-7]?[0-9A-Fa-f]{1,3}|:0\\.0\\.(12[0-7]|1[01][0-9]|[1-9]?[0-9])\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9]))", RGXG_NOZEROCOMPRESSION },
    { "::0/113", "(:(:0?0?0?0){0,6}:[0-7]?[0-9A-Fa-f]{1,3}|::|0?0?0?0(:(:0?0?0?0){0,5}:[0-7]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0){0,4}:[0-7]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0){0,3}:[0-7]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0){0,2}:[0-7]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0)?:[0-7]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(::[0-7]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(::|:[0-7]?[0-9A-Fa-f]{1,3}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/113", "((:(:0?0?0?0){0,6}:[0-7]?[0-9A-Fa-f]{1,3}|::|0?0?0?0(:(:0?0?0?0){0,5}:[0-7]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0){0,4}:[0-7]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0){0,3}:[0-7]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0){0,2}:[0-7]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(:(:0?0?0?0)?:[0-7]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(::[0-7]?[0-9A-Fa-f]{1,3}|::|:0?0?0?0(::|:[0-7]?[0-9A-Fa-f]{1,3}))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):0\\.0\\.(12[0-7]|1[01][0-9]|[1-9]?[0-9])\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9]))", 0 },
    { "::0/112", "0?0?0?0(:0?0?0?0){6}:[0-9A-Fa-f]{1,4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/112", "0?0?0?0(:0?0?0?0){5}(:0?0?0?0:[0-9A-Fa-f]{1,4}|:0\\.0(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2})", RGXG_NOZEROCOMPRESSION },
    { "::0/112", "(:(:0?0?0?0){0,6}:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,5}:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,4}:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/112", "((:(:0?0?0?0){0,6}:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,5}:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,4}:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):0\\.0(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2})", 0 },
    { "::0/111", "0?0?0?0(:0?0?0?0){5}:0?0?0?[01]:[0-9A-Fa-f]{1,4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/111", "0?0?0?0(:0?0?0?0){5}(:0?0?0?[01]:[0-9A-Fa-f]{1,4}|:0\\.[01](\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2})", RGXG_NOZEROCOMPRESSION },
    { "::0/111", "(:(:0?0?0?0){0,5}(:0?0?0?[01])?:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,4}(:0?0?0?[01])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:0?0?0?[01])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[01])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[01])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?[01])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:0?0?0?[01](::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/111", "((:(:0?0?0?0){0,5}(:0?0?0?[01])?:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,4}(:0?0?0?[01])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:0?0?0?[01])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[01])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[01])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?[01])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:0?0?0?[01](::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):0\\.[01](\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2})", 0 },
    { "::0/110", "0?0?0?0(:0?0?0?0){5}:0?0?0?[0-3]:[0-9A-Fa-f]{1,4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/110", "0?0?0?0(:0?0?0?0){5}(:0?0?0?[0-3]:[0-9A-Fa-f]{1,4}|:0\\.[0-3](\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2})", RGXG_NOZEROCOMPRESSION },
    { "::0/110", "(:(:0?0?0?0){0,5}(:0?0?0?[0-3])?:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,4}(:0?0?0?[0-3])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:0?0?0?[0-3])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[0-3])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-3])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?[0-3])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:0?0?0?[0-3](::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/110", "((:(:0?0?0?0){0,5}(:0?0?0?[0-3])?:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,4}(:0?0?0?[0-3])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:0?0?0?[0-3])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[0-3])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-3])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?[0-3])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:0?0?0?[0-3](::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):0\\.[0-3](\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2})", 0 },
    { "::0/109", "0?0?0?0(:0?0?0?0){5}:0?0?0?[0-7]:[0-9A-Fa-f]{1,4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/109", "0?0?0?0(:0?0?0?0){5}(:0?0?0?[0-7]:[0-9A-Fa-f]{1,4}|:0\\.[0-7](\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2})", RGXG_NOZEROCOMPRESSION },
    { "::0/109", "(:(:0?0?0?0){0,5}(:0?0?0?[0-7])?:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,4}(:0?0?0?[0-7])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:0?0?0?[0-7])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[0-7])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-7])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?[0-7])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:0?0?0?[0-7](::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/109", "((:(:0?0?0?0){0,5}(:0?0?0?[0-7])?:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,4}(:0?0?0?[0-7])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:0?0?0?[0-7])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[0-7])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-7])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?[0-7])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:0?0?0?[0-7](::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):0\\.[0-7](\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2})", 0 },
    { "::0/108", "0?0?0?0(:0?0?0?0){5}:0?0?0?[0-9A-Fa-f]:[0-9A-Fa-f]{1,4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/108", "0?0?0?0(:0?0?0?0){5}(:0?0?0?[0-9A-Fa-f]:[0-9A-Fa-f]{1,4}|:0\\.(1[0-5]|[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2})", RGXG_NOZEROCOMPRESSION },
    { "::0/108", "(:(:0?0?0?0){0,5}(:0?0?0?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,4}(:0?0?0?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:0?0?0?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:0?0?0?[0-9A-Fa-f](::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/108", "((:(:0?0?0?0){0,5}(:0?0?0?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,4}(:0?0?0?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:0?0?0?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:0?0?0?[0-9A-Fa-f](::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):0\\.(1[0-5]|[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2})", 0 },
    { "::0/107", "0?0?0?0(:0?0?0?0){5}:0?0?[01]?[0-9A-Fa-f]:[0-9A-Fa-f]{1,4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/107", "0?0?0?0(:0?0?0?0){5}(:0?0?[01]?[0-9A-Fa-f]:[0-9A-Fa-f]{1,4}|:0\\.(3[01]|[12]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2})", RGXG_NOZEROCOMPRESSION },
    { "::0/107", "(:(:0?0?0?0){0,5}(:0?0?[01]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,4}(:0?0?[01]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:0?0?[01]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[01]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?[01]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?[01]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:0?0?[01]?[0-9A-Fa-f](::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/107", "((:(:0?0?0?0){0,5}(:0?0?[01]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,4}(:0?0?[01]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:0?0?[01]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[01]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?[01]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?[01]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:0?0?[01]?[0-9A-Fa-f](::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):0\\.(3[01]|[12]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2})", 0 },
    { "::0/106", "0?0?0?0(:0?0?0?0){5}:0?0?[0-3]?[0-9A-Fa-f]:[0-9A-Fa-f]{1,4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/106", "0?0?0?0(:0?0?0?0){5}(:0?0?[0-3]?[0-9A-Fa-f]:[0-9A-Fa-f]{1,4}|:0\\.(6[0-3]|[1-5]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2})", RGXG_NOZEROCOMPRESSION },
    { "::0/106", "(:(:0?0?0?0){0,5}(:0?0?[0-3]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,4}(:0?0?[0-3]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:0?0?[0-3]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[0-3]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?[0-3]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?[0-3]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:0?0?[0-3]?[0-9A-Fa-f](::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/106", "((:(:0?0?0?0){0,5}(:0?0?[0-3]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,4}(:0?0?[0-3]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:0?0?[0-3]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[0-3]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?[0-3]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?[0-3]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:0?0?[0-3]?[0-9A-Fa-f](::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):0\\.(6[0-3]|[1-5]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2})", 0 },
    { "::0/105", "0?0?0?0(:0?0?0?0){5}:0?0?[0-7]?[0-9A-Fa-f]:[0-9A-Fa-f]{1,4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/105", "0?0?0?0(:0?0?0?0){5}(:0?0?[0-7]?[0-9A-Fa-f]:[0-9A-Fa-f]{1,4}|:0\\.(12[0-7]|1[01][0-9]|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2})", RGXG_NOZEROCOMPRESSION },
    { "::0/105", "(:(:0?0?0?0){0,5}(:0?0?[0-7]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,4}(:0?0?[0-7]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:0?0?[0-7]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[0-7]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?[0-7]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?[0-7]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:0?0?[0-7]?[0-9A-Fa-f](::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/105", "((:(:0?0?0?0){0,5}(:0?0?[0-7]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,4}(:0?0?[0-7]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:0?0?[0-7]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[0-7]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?[0-7]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?[0-7]?[0-9A-Fa-f])?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:0?0?[0-7]?[0-9A-Fa-f](::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):0\\.(12[0-7]|1[01][0-9]|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){2})", 0 },
    { "::0/104", "0?0?0?0(:0?0?0?0){5}:0?0?[0-9A-Fa-f]{1,2}:[0-9A-Fa-f]{1,4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/104", "0?0?0?0(:0?0?0?0){5}(:0?0?[0-9A-Fa-f]{1,2}:[0-9A-Fa-f]{1,4}|:0(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/104", "(:(:0?0?0?0){0,5}(:0?0?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,4}(:0?0?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:0?0?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:0?0?[0-9A-Fa-f]{1,2}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/104", "((:(:0?0?0?0){0,5}(:0?0?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,4}(:0?0?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:0?0?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:0?0?[0-9A-Fa-f]{1,2}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):0(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/103", "0?0?0?0(:0?0?0?0){5}:0?[01]?[0-9A-Fa-f]{1,2}:[0-9A-Fa-f]{1,4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/103", "0?0?0?0(:0?0?0?0){5}(:0?[01]?[0-9A-Fa-f]{1,2}:[0-9A-Fa-f]{1,4}|:[01](\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/103", "(:(:0?0?0?0){0,5}(:0?[01]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,4}(:0?[01]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:0?[01]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?[01]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:0?[01]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?[01]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:0?[01]?[0-9A-Fa-f]{1,2}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/103", "((:(:0?0?0?0){0,5}(:0?[01]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,4}(:0?[01]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:0?[01]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?[01]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:0?[01]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?[01]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:0?[01]?[0-9A-Fa-f]{1,2}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):[01](\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/102", "0?0?0?0(:0?0?0?0){5}:0?[0-3]?[0-9A-Fa-f]{1,2}:[0-9A-Fa-f]{1,4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/102", "0?0?0?0(:0?0?0?0){5}(:0?[0-3]?[0-9A-Fa-f]{1,2}:[0-9A-Fa-f]{1,4}|:[0-3](\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/102", "(:(:0?0?0?0){0,5}(:0?[0-3]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,4}(:0?[0-3]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:0?[0-3]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?[0-3]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:0?[0-3]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?[0-3]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:0?[0-3]?[0-9A-Fa-f]{1,2}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/102", "((:(:0?0?0?0){0,5}(:0?[0-3]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,4}(:0?[0-3]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:0?[0-3]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?[0-3]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:0?[0-3]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?[0-3]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:0?[0-3]?[0-9A-Fa-f]{1,2}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):[0-3](\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/101", "0?0?0?0(:0?0?0?0){5}:0?[0-7]?[0-9A-Fa-f]{1,2}:[0-9A-Fa-f]{1,4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/101", "0?0?0?0(:0?0?0?0){5}(:0?[0-7]?[0-9A-Fa-f]{1,2}:[0-9A-Fa-f]{1,4}|:[0-7](\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/101", "(:(:0?0?0?0){0,5}(:0?[0-7]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,4}(:0?[0-7]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:0?[0-7]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?[0-7]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:0?[0-7]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?[0-7]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:0?[0-7]?[0-9A-Fa-f]{1,2}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/101", "((:(:0?0?0?0){0,5}(:0?[0-7]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,4}(:0?[0-7]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:0?[0-7]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?[0-7]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:0?[0-7]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?[0-7]?[0-9A-Fa-f]{1,2})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:0?[0-7]?[0-9A-Fa-f]{1,2}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):[0-7](\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/100", "0?0?0?0(:0?0?0?0){5}:0?[0-9A-Fa-f]{1,3}:[0-9A-Fa-f]{1,4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/100", "0?0?0?0(:0?0?0?0){5}(:0?[0-9A-Fa-f]{1,3}:[0-9A-Fa-f]{1,4}|:(1[0-5]|[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/100", "(:(:0?0?0?0){0,5}(:0?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,4}(:0?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:0?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:0?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:0?[0-9A-Fa-f]{1,3}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/100", "((:(:0?0?0?0){0,5}(:0?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,4}(:0?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:0?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:0?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:0?[0-9A-Fa-f]{1,3}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):(1[0-5]|[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/99", "0?0?0?0(:0?0?0?0){5}:[01]?[0-9A-Fa-f]{1,3}:[0-9A-Fa-f]{1,4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/99", "0?0?0?0(:0?0?0?0){5}(:[01]?[0-9A-Fa-f]{1,3}:[0-9A-Fa-f]{1,4}|:(3[01]|[12]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/99", "(:(:0?0?0?0){0,5}(:[01]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,4}(:[01]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:[01]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:[01]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:[01]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:[01]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:[01]?[0-9A-Fa-f]{1,3}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/99", "((:(:0?0?0?0){0,5}(:[01]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,4}(:[01]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:[01]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:[01]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:[01]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:[01]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:[01]?[0-9A-Fa-f]{1,3}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):(3[01]|[12]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/98", "0?0?0?0(:0?0?0?0){5}:[0-3]?[0-9A-Fa-f]{1,3}:[0-9A-Fa-f]{1,4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/98", "0?0?0?0(:0?0?0?0){5}(:[0-3]?[0-9A-Fa-f]{1,3}:[0-9A-Fa-f]{1,4}|:(6[0-3]|[1-5]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/98", "(:(:0?0?0?0){0,5}(:[0-3]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,4}(:[0-3]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:[0-3]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:[0-3]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:[0-3]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:[0-3]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:[0-3]?[0-9A-Fa-f]{1,3}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/98", "((:(:0?0?0?0){0,5}(:[0-3]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,4}(:[0-3]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:[0-3]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:[0-3]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:[0-3]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:[0-3]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:[0-3]?[0-9A-Fa-f]{1,3}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):(6[0-3]|[1-5]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/97", "0?0?0?0(:0?0?0?0){5}:[0-7]?[0-9A-Fa-f]{1,3}:[0-9A-Fa-f]{1,4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/97", "0?0?0?0(:0?0?0?0){5}(:[0-7]?[0-9A-Fa-f]{1,3}:[0-9A-Fa-f]{1,4}|:(12[0-7]|1[01][0-9]|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/97", "(:(:0?0?0?0){0,5}(:[0-7]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,4}(:[0-7]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:[0-7]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:[0-7]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:[0-7]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:[0-7]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:[0-7]?[0-9A-Fa-f]{1,3}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/97", "((:(:0?0?0?0){0,5}(:[0-7]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|0?0?0?0(:(:0?0?0?0){0,4}(:[0-7]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:[0-7]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:[0-7]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:[0-7]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(:(:[0-7]?[0-9A-Fa-f]{1,3})?:[0-9A-Fa-f]{1,4}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:[0-7]?[0-9A-Fa-f]{1,3}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):(12[0-7]|1[01][0-9]|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/96", "0?0?0?0(:0?0?0?0){5}(:[0-9A-Fa-f]{1,4}){2}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/96", "0?0?0?0(:0?0?0?0){5}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/96", "(:(:0?0?0?0){0,5}(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,4}(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/96", "((:(:0?0?0?0){0,5}(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,4}(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,3}(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,5}|0?0?0?0(:(:0?0?0?0){0,4}|:0?0?0?0(:(:0?0?0?0){0,3}|:0?0?0?0(:(:0?0?0?0){0,2}|:0?0?0?0(:(:0?0?0?0)?|:0?0?0?0(:|:0?0?0?0)))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/95", "0?0?0?0(:0?0?0?0){4}:0?0?0?[01](:[0-9A-Fa-f]{1,4}){2}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/95", "0?0?0?0(:0?0?0?0){4}:0?0?0?[01]((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/95", "(:(:0?0?0?0){0,4}(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,3}(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?[01](::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/95", "((:(:0?0?0?0){0,4}(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,3}(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?[01](::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,4}(:0?0?0?[01])?|0?0?0?0(:(:0?0?0?0){0,3}(:0?0?0?[01])?|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[01])?|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[01])?|:0?0?0?0(:(:0?0?0?[01])?|:0?0?0?0(:|:0?0?0?[01])))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/94", "0?0?0?0(:0?0?0?0){4}:0?0?0?[0-3](:[0-9A-Fa-f]{1,4}){2}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/94", "0?0?0?0(:0?0?0?0){4}:0?0?0?[0-3]((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/94", "(:(:0?0?0?0){0,4}(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,3}(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?[0-3](::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/94", "((:(:0?0?0?0){0,4}(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,3}(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?[0-3](::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,4}(:0?0?0?[0-3])?|0?0?0?0(:(:0?0?0?0){0,3}(:0?0?0?[0-3])?|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[0-3])?|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-3])?|:0?0?0?0(:(:0?0?0?[0-3])?|:0?0?0?0(:|:0?0?0?[0-3])))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/93", "0?0?0?0(:0?0?0?0){4}:0?0?0?[0-7](:[0-9A-Fa-f]{1,4}){2}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/93", "0?0?0?0(:0?0?0?0){4}:0?0?0?[0-7]((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/93", "(:(:0?0?0?0){0,4}(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,3}(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?[0-7](::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/93", "((:(:0?0?0?0){0,4}(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,3}(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?[0-7](::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,4}(:0?0?0?[0-7])?|0?0?0?0(:(:0?0?0?0){0,3}(:0?0?0?[0-7])?|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[0-7])?|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-7])?|:0?0?0?0(:(:0?0?0?[0-7])?|:0?0?0?0(:|:0?0?0?[0-7])))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/92", "0?0?0?0(:0?0?0?0){4}:0?0?0?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){2}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/92", "0?0?0?0(:0?0?0?0){4}:0?0?0?[0-9A-Fa-f]((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/92", "(:(:0?0?0?0){0,4}(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,3}(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?[0-9A-Fa-f](::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/92", "((:(:0?0?0?0){0,4}(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,3}(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?[0-9A-Fa-f](::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,4}(:0?0?0?[0-9A-Fa-f])?|0?0?0?0(:(:0?0?0?0){0,3}(:0?0?0?[0-9A-Fa-f])?|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[0-9A-Fa-f])?|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-9A-Fa-f])?|:0?0?0?0(:(:0?0?0?[0-9A-Fa-f])?|:0?0?0?0(:|:0?0?0?[0-9A-Fa-f])))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/91", "0?0?0?0(:0?0?0?0){4}:0?0?[01]?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){2}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/91", "0?0?0?0(:0?0?0?0){4}:0?0?[01]?[0-9A-Fa-f]((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/91", "(:(:0?0?0?0){0,4}(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,3}(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?[01]?[0-9A-Fa-f](::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/91", "((:(:0?0?0?0){0,4}(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,3}(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?[01]?[0-9A-Fa-f](::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,4}(:0?0?[01]?[0-9A-Fa-f])?|0?0?0?0(:(:0?0?0?0){0,3}(:0?0?[01]?[0-9A-Fa-f])?|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[01]?[0-9A-Fa-f])?|:0?0?0?0(:(:0?0?0?0)?(:0?0?[01]?[0-9A-Fa-f])?|:0?0?0?0(:(:0?0?[01]?[0-9A-Fa-f])?|:0?0?0?0(:|:0?0?[01]?[0-9A-Fa-f])))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/90", "0?0?0?0(:0?0?0?0){4}:0?0?[0-3]?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){2}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/90", "0?0?0?0(:0?0?0?0){4}:0?0?[0-3]?[0-9A-Fa-f]((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/90", "(:(:0?0?0?0){0,4}(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,3}(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?[0-3]?[0-9A-Fa-f](::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/90", "((:(:0?0?0?0){0,4}(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,3}(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?[0-3]?[0-9A-Fa-f](::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,4}(:0?0?[0-3]?[0-9A-Fa-f])?|0?0?0?0(:(:0?0?0?0){0,3}(:0?0?[0-3]?[0-9A-Fa-f])?|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[0-3]?[0-9A-Fa-f])?|:0?0?0?0(:(:0?0?0?0)?(:0?0?[0-3]?[0-9A-Fa-f])?|:0?0?0?0(:(:0?0?[0-3]?[0-9A-Fa-f])?|:0?0?0?0(:|:0?0?[0-3]?[0-9A-Fa-f])))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/89", "0?0?0?0(:0?0?0?0){4}:0?0?[0-7]?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){2}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/89", "0?0?0?0(:0?0?0?0){4}:0?0?[0-7]?[0-9A-Fa-f]((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/89", "(:(:0?0?0?0){0,4}(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,3}(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?[0-7]?[0-9A-Fa-f](::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/89", "((:(:0?0?0?0){0,4}(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,3}(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?[0-7]?[0-9A-Fa-f](::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,4}(:0?0?[0-7]?[0-9A-Fa-f])?|0?0?0?0(:(:0?0?0?0){0,3}(:0?0?[0-7]?[0-9A-Fa-f])?|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[0-7]?[0-9A-Fa-f])?|:0?0?0?0(:(:0?0?0?0)?(:0?0?[0-7]?[0-9A-Fa-f])?|:0?0?0?0(:(:0?0?[0-7]?[0-9A-Fa-f])?|:0?0?0?0(:|:0?0?[0-7]?[0-9A-Fa-f])))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/88", "0?0?0?0(:0?0?0?0){4}:0?0?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){2}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/88", "0?0?0?0(:0?0?0?0){4}:0?0?[0-9A-Fa-f]{1,2}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/88", "(:(:0?0?0?0){0,4}(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,3}(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?[0-9A-Fa-f]{1,2}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/88", "((:(:0?0?0?0){0,4}(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,3}(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?[0-9A-Fa-f]{1,2}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,4}(:0?0?[0-9A-Fa-f]{1,2})?|0?0?0?0(:(:0?0?0?0){0,3}(:0?0?[0-9A-Fa-f]{1,2})?|:0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[0-9A-Fa-f]{1,2})?|:0?0?0?0(:(:0?0?0?0)?(:0?0?[0-9A-Fa-f]{1,2})?|:0?0?0?0(:(:0?0?[0-9A-Fa-f]{1,2})?|:0?0?0?0(:|:0?0?[0-9A-Fa-f]{1,2})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/87", "0?0?0?0(:0?0?0?0){4}:0?[01]?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){2}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/87", "0?0?0?0(:0?0?0?0){4}:0?[01]?[0-9A-Fa-f]{1,2}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/87", "(:(:0?0?0?0){0,4}(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,3}(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?[01]?[0-9A-Fa-f]{1,2}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/87", "((:(:0?0?0?0){0,4}(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,3}(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?[01]?[0-9A-Fa-f]{1,2}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,4}(:0?[01]?[0-9A-Fa-f]{1,2})?|0?0?0?0(:(:0?0?0?0){0,3}(:0?[01]?[0-9A-Fa-f]{1,2})?|:0?0?0?0(:(:0?0?0?0){0,2}(:0?[01]?[0-9A-Fa-f]{1,2})?|:0?0?0?0(:(:0?0?0?0)?(:0?[01]?[0-9A-Fa-f]{1,2})?|:0?0?0?0(:(:0?[01]?[0-9A-Fa-f]{1,2})?|:0?0?0?0(:|:0?[01]?[0-9A-Fa-f]{1,2})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/86", "0?0?0?0(:0?0?0?0){4}:0?[0-3]?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){2}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/86", "0?0?0?0(:0?0?0?0){4}:0?[0-3]?[0-9A-Fa-f]{1,2}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/86", "(:(:0?0?0?0){0,4}(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,3}(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?[0-3]?[0-9A-Fa-f]{1,2}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/86", "((:(:0?0?0?0){0,4}(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,3}(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?[0-3]?[0-9A-Fa-f]{1,2}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,4}(:0?[0-3]?[0-9A-Fa-f]{1,2})?|0?0?0?0(:(:0?0?0?0){0,3}(:0?[0-3]?[0-9A-Fa-f]{1,2})?|:0?0?0?0(:(:0?0?0?0){0,2}(:0?[0-3]?[0-9A-Fa-f]{1,2})?|:0?0?0?0(:(:0?0?0?0)?(:0?[0-3]?[0-9A-Fa-f]{1,2})?|:0?0?0?0(:(:0?[0-3]?[0-9A-Fa-f]{1,2})?|:0?0?0?0(:|:0?[0-3]?[0-9A-Fa-f]{1,2})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/85", "0?0?0?0(:0?0?0?0){4}:0?[0-7]?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){2}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/85", "0?0?0?0(:0?0?0?0){4}:0?[0-7]?[0-9A-Fa-f]{1,2}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/85", "(:(:0?0?0?0){0,4}(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,3}(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?[0-7]?[0-9A-Fa-f]{1,2}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/85", "((:(:0?0?0?0){0,4}(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,3}(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?[0-7]?[0-9A-Fa-f]{1,2}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,4}(:0?[0-7]?[0-9A-Fa-f]{1,2})?|0?0?0?0(:(:0?0?0?0){0,3}(:0?[0-7]?[0-9A-Fa-f]{1,2})?|:0?0?0?0(:(:0?0?0?0){0,2}(:0?[0-7]?[0-9A-Fa-f]{1,2})?|:0?0?0?0(:(:0?0?0?0)?(:0?[0-7]?[0-9A-Fa-f]{1,2})?|:0?0?0?0(:(:0?[0-7]?[0-9A-Fa-f]{1,2})?|:0?0?0?0(:|:0?[0-7]?[0-9A-Fa-f]{1,2})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/84", "0?0?0?0(:0?0?0?0){4}:0?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){2}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/84", "0?0?0?0(:0?0?0?0){4}:0?[0-9A-Fa-f]{1,3}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/84", "(:(:0?0?0?0){0,4}(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,3}(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?[0-9A-Fa-f]{1,3}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/84", "((:(:0?0?0?0){0,4}(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,3}(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?[0-9A-Fa-f]{1,3}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,4}(:0?[0-9A-Fa-f]{1,3})?|0?0?0?0(:(:0?0?0?0){0,3}(:0?[0-9A-Fa-f]{1,3})?|:0?0?0?0(:(:0?0?0?0){0,2}(:0?[0-9A-Fa-f]{1,3})?|:0?0?0?0(:(:0?0?0?0)?(:0?[0-9A-Fa-f]{1,3})?|:0?0?0?0(:(:0?[0-9A-Fa-f]{1,3})?|:0?0?0?0(:|:0?[0-9A-Fa-f]{1,3})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/83", "0?0?0?0(:0?0?0?0){4}:[01]?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){2}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/83", "0?0?0?0(:0?0?0?0){4}:[01]?[0-9A-Fa-f]{1,3}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/83", "(:(:0?0?0?0){0,4}(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,3}(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[01]?[0-9A-Fa-f]{1,3}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/83", "((:(:0?0?0?0){0,4}(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,3}(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[01]?[0-9A-Fa-f]{1,3}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,4}(:[01]?[0-9A-Fa-f]{1,3})?|0?0?0?0(:(:0?0?0?0){0,3}(:[01]?[0-9A-Fa-f]{1,3})?|:0?0?0?0(:(:0?0?0?0){0,2}(:[01]?[0-9A-Fa-f]{1,3})?|:0?0?0?0(:(:0?0?0?0)?(:[01]?[0-9A-Fa-f]{1,3})?|:0?0?0?0(:(:[01]?[0-9A-Fa-f]{1,3})?|:0?0?0?0(:|:[01]?[0-9A-Fa-f]{1,3})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/82", "0?0?0?0(:0?0?0?0){4}:[0-3]?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){2}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/82", "0?0?0?0(:0?0?0?0){4}:[0-3]?[0-9A-Fa-f]{1,3}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/82", "(:(:0?0?0?0){0,4}(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,3}(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-3]?[0-9A-Fa-f]{1,3}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/82", "((:(:0?0?0?0){0,4}(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,3}(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-3]?[0-9A-Fa-f]{1,3}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,4}(:[0-3]?[0-9A-Fa-f]{1,3})?|0?0?0?0(:(:0?0?0?0){0,3}(:[0-3]?[0-9A-Fa-f]{1,3})?|:0?0?0?0(:(:0?0?0?0){0,2}(:[0-3]?[0-9A-Fa-f]{1,3})?|:0?0?0?0(:(:0?0?0?0)?(:[0-3]?[0-9A-Fa-f]{1,3})?|:0?0?0?0(:(:[0-3]?[0-9A-Fa-f]{1,3})?|:0?0?0?0(:|:[0-3]?[0-9A-Fa-f]{1,3})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/81", "0?0?0?0(:0?0?0?0){4}:[0-7]?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){2}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/81", "0?0?0?0(:0?0?0?0){4}:[0-7]?[0-9A-Fa-f]{1,3}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/81", "(:(:0?0?0?0){0,4}(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,3}(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-7]?[0-9A-Fa-f]{1,3}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/81", "((:(:0?0?0?0){0,4}(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|0?0?0?0(:(:0?0?0?0){0,3}(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:0?0?0?0)?(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,2}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-7]?[0-9A-Fa-f]{1,3}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,4}(:[0-7]?[0-9A-Fa-f]{1,3})?|0?0?0?0(:(:0?0?0?0){0,3}(:[0-7]?[0-9A-Fa-f]{1,3})?|:0?0?0?0(:(:0?0?0?0){0,2}(:[0-7]?[0-9A-Fa-f]{1,3})?|:0?0?0?0(:(:0?0?0?0)?(:[0-7]?[0-9A-Fa-f]{1,3})?|:0?0?0?0(:(:[0-7]?[0-9A-Fa-f]{1,3})?|:0?0?0?0(:|:[0-7]?[0-9A-Fa-f]{1,3})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/80", "0?0?0?0(:0?0?0?0){4}(:[0-9A-Fa-f]{1,4}){3}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/80", "0?0?0?0(:0?0?0?0){4}:[0-9A-Fa-f]{1,4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/80", "(:(:0?0?0?0){0,4}(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,3}(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/80", "((:(:0?0?0?0){0,4}(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,3}(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0){0,2}(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,4}(:[0-9A-Fa-f]{1,4})?|0?0?0?0(:(:0?0?0?0){0,3}(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:0?0?0?0){0,2}(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:0?0?0?0)?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/79", "0?0?0?0(:0?0?0?0){3}:0?0?0?[01](:[0-9A-Fa-f]{1,4}){3}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/79", "0?0?0?0(:0?0?0?0){3}:0?0?0?[01]:[0-9A-Fa-f]{1,4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/79", "(:(:0?0?0?0){0,3}(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?[01](:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/79", "((:(:0?0?0?0){0,3}(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?[01](:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,3}(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4})?|0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:[0-9A-Fa-f]{1,4})?|:0?0?0?[01](:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/78", "0?0?0?0(:0?0?0?0){3}:0?0?0?[0-3](:[0-9A-Fa-f]{1,4}){3}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/78", "0?0?0?0(:0?0?0?0){3}:0?0?0?[0-3]:[0-9A-Fa-f]{1,4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/78", "(:(:0?0?0?0){0,3}(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?[0-3](:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/78", "((:(:0?0?0?0){0,3}(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?[0-3](:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,3}(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4})?|0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:[0-9A-Fa-f]{1,4})?|:0?0?0?[0-3](:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/77", "0?0?0?0(:0?0?0?0){3}:0?0?0?[0-7](:[0-9A-Fa-f]{1,4}){3}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/77", "0?0?0?0(:0?0?0?0){3}:0?0?0?[0-7]:[0-9A-Fa-f]{1,4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/77", "(:(:0?0?0?0){0,3}(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?[0-7](:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/77", "((:(:0?0?0?0){0,3}(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?[0-7](:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,3}(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4})?|0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:[0-9A-Fa-f]{1,4})?|:0?0?0?[0-7](:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/76", "0?0?0?0(:0?0?0?0){3}:0?0?0?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){3}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/76", "0?0?0?0(:0?0?0?0){3}:0?0?0?[0-9A-Fa-f]:[0-9A-Fa-f]{1,4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/76", "(:(:0?0?0?0){0,3}(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/76", "((:(:0?0?0?0){0,3}(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,3}(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4})?|0?0?0?0(:(:0?0?0?0){0,2}(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:[0-9A-Fa-f]{1,4})?|:0?0?0?[0-9A-Fa-f](:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/75", "0?0?0?0(:0?0?0?0){3}:0?0?[01]?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){3}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/75", "0?0?0?0(:0?0?0?0){3}:0?0?[01]?[0-9A-Fa-f]:[0-9A-Fa-f]{1,4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/75", "(:(:0?0?0?0){0,3}(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?[01]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/75", "((:(:0?0?0?0){0,3}(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?[01]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,3}(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4})?|0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:0?0?0?0)?(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:[0-9A-Fa-f]{1,4})?|:0?0?[01]?[0-9A-Fa-f](:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/74", "0?0?0?0(:0?0?0?0){3}:0?0?[0-3]?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){3}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/74", "0?0?0?0(:0?0?0?0){3}:0?0?[0-3]?[0-9A-Fa-f]:[0-9A-Fa-f]{1,4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/74", "(:(:0?0?0?0){0,3}(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?[0-3]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/74", "((:(:0?0?0?0){0,3}(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?[0-3]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,3}(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4})?|0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:0?0?0?0)?(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:[0-9A-Fa-f]{1,4})?|:0?0?[0-3]?[0-9A-Fa-f](:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/73", "0?0?0?0(:0?0?0?0){3}:0?0?[0-7]?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){3}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/73", "0?0?0?0(:0?0?0?0){3}:0?0?[0-7]?[0-9A-Fa-f]:[0-9A-Fa-f]{1,4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/73", "(:(:0?0?0?0){0,3}(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?[0-7]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/73", "((:(:0?0?0?0){0,3}(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?[0-7]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,3}(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4})?|0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:0?0?0?0)?(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:[0-9A-Fa-f]{1,4})?|:0?0?[0-7]?[0-9A-Fa-f](:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/72", "0?0?0?0(:0?0?0?0){3}:0?0?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){3}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/72", "0?0?0?0(:0?0?0?0){3}:0?0?[0-9A-Fa-f]{1,2}:[0-9A-Fa-f]{1,4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/72", "(:(:0?0?0?0){0,3}(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/72", "((:(:0?0?0?0){0,3}(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,3}(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4})?|0?0?0?0(:(:0?0?0?0){0,2}(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:0?0?0?0)?(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:[0-9A-Fa-f]{1,4})?|:0?0?[0-9A-Fa-f]{1,2}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/71", "0?0?0?0(:0?0?0?0){3}:0?[01]?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){3}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/71", "0?0?0?0(:0?0?0?0){3}:0?[01]?[0-9A-Fa-f]{1,2}:[0-9A-Fa-f]{1,4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/71", "(:(:0?0?0?0){0,3}(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,2}(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?[01]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/71", "((:(:0?0?0?0){0,3}(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,2}(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?[01]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,3}(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4})?|0?0?0?0(:(:0?0?0?0){0,2}(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:0?0?0?0)?(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:[0-9A-Fa-f]{1,4})?|:0?[01]?[0-9A-Fa-f]{1,2}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/70", "0?0?0?0(:0?0?0?0){3}:0?[0-3]?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){3}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/70", "0?0?0?0(:0?0?0?0){3}:0?[0-3]?[0-9A-Fa-f]{1,2}:[0-9A-Fa-f]{1,4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/70", "(:(:0?0?0?0){0,3}(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,2}(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?[0-3]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/70", "((:(:0?0?0?0){0,3}(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,2}(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?[0-3]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,3}(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4})?|0?0?0?0(:(:0?0?0?0){0,2}(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:0?0?0?0)?(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:[0-9A-Fa-f]{1,4})?|:0?[0-3]?[0-9A-Fa-f]{1,2}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/69", "0?0?0?0(:0?0?0?0){3}:0?[0-7]?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){3}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/69", "0?0?0?0(:0?0?0?0){3}:0?[0-7]?[0-9A-Fa-f]{1,2}:[0-9A-Fa-f]{1,4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/69", "(:(:0?0?0?0){0,3}(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,2}(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?[0-7]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/69", "((:(:0?0?0?0){0,3}(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,2}(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?[0-7]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,3}(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4})?|0?0?0?0(:(:0?0?0?0){0,2}(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:0?0?0?0)?(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:[0-9A-Fa-f]{1,4})?|:0?[0-7]?[0-9A-Fa-f]{1,2}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/68", "0?0?0?0(:0?0?0?0){3}:0?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){3}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/68", "0?0?0?0(:0?0?0?0){3}:0?[0-9A-Fa-f]{1,3}:[0-9A-Fa-f]{1,4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/68", "(:(:0?0?0?0){0,3}(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,2}(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/68", "((:(:0?0?0?0){0,3}(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,2}(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,3}(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4})?|0?0?0?0(:(:0?0?0?0){0,2}(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:0?0?0?0)?(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:[0-9A-Fa-f]{1,4})?|:0?[0-9A-Fa-f]{1,3}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/67", "0?0?0?0(:0?0?0?0){3}:[01]?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){3}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/67", "0?0?0?0(:0?0?0?0){3}:[01]?[0-9A-Fa-f]{1,3}:[0-9A-Fa-f]{1,4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/67", "(:(:0?0?0?0){0,3}(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,2}(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[01]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/67", "((:(:0?0?0?0){0,3}(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,2}(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[01]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,3}(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4})?|0?0?0?0(:(:0?0?0?0){0,2}(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:0?0?0?0)?(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:[0-9A-Fa-f]{1,4})?|:[01]?[0-9A-Fa-f]{1,3}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/66", "0?0?0?0(:0?0?0?0){3}:[0-3]?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){3}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/66", "0?0?0?0(:0?0?0?0){3}:[0-3]?[0-9A-Fa-f]{1,3}:[0-9A-Fa-f]{1,4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/66", "(:(:0?0?0?0){0,3}(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,2}(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-3]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/66", "((:(:0?0?0?0){0,3}(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,2}(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-3]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,3}(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4})?|0?0?0?0(:(:0?0?0?0){0,2}(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:0?0?0?0)?(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:[0-9A-Fa-f]{1,4})?|:[0-3]?[0-9A-Fa-f]{1,3}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/65", "0?0?0?0(:0?0?0?0){3}:[0-7]?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){3}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/65", "0?0?0?0(:0?0?0?0){3}:[0-7]?[0-9A-Fa-f]{1,3}:[0-9A-Fa-f]{1,4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/65", "(:(:0?0?0?0){0,3}(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,2}(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-7]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/65", "((:(:0?0?0?0){0,3}(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|0?0?0?0(:(:0?0?0?0){0,2}(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:0?0?0?0)?(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,3}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-7]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,3}(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4})?|0?0?0?0(:(:0?0?0?0){0,2}(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:0?0?0?0)?(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4})?|:0?0?0?0(:(:[0-9A-Fa-f]{1,4})?|:[0-7]?[0-9A-Fa-f]{1,3}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/64", "0?0?0?0(:0?0?0?0){3}(:[0-9A-Fa-f]{1,4}){4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/64", "0?0?0?0(:0?0?0?0){3}(:[0-9A-Fa-f]{1,4}){2}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/64", "(:(:0?0?0?0){0,3}(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0){0,2}(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/64", "((:(:0?0?0?0){0,3}(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0){0,2}(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:0?0?0?0)?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,3}(:[0-9A-Fa-f]{1,4}){0,2}|0?0?0?0(:(:0?0?0?0){0,2}(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:0?0?0?0)?(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/63", "0?0?0?0(:0?0?0?0){2}:0?0?0?[01](:[0-9A-Fa-f]{1,4}){4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/63", "0?0?0?0(:0?0?0?0){2}:0?0?0?[01](:[0-9A-Fa-f]{1,4}){2}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/63", "(:(:0?0?0?0){0,2}(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0)?(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?[01](:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/63", "((:(:0?0?0?0){0,2}(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0)?(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?[01](:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,2}(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){0,2}|0?0?0?0(:(:0?0?0?0)?(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?[01](:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/62", "0?0?0?0(:0?0?0?0){2}:0?0?0?[0-3](:[0-9A-Fa-f]{1,4}){4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/62", "0?0?0?0(:0?0?0?0){2}:0?0?0?[0-3](:[0-9A-Fa-f]{1,4}){2}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/62", "(:(:0?0?0?0){0,2}(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?[0-3](:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/62", "((:(:0?0?0?0){0,2}(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?[0-3](:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,2}(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){0,2}|0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?[0-3](:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/61", "0?0?0?0(:0?0?0?0){2}:0?0?0?[0-7](:[0-9A-Fa-f]{1,4}){4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/61", "0?0?0?0(:0?0?0?0){2}:0?0?0?[0-7](:[0-9A-Fa-f]{1,4}){2}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/61", "(:(:0?0?0?0){0,2}(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?[0-7](:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/61", "((:(:0?0?0?0){0,2}(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?[0-7](:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,2}(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){0,2}|0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?[0-7](:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/60", "0?0?0?0(:0?0?0?0){2}:0?0?0?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/60", "0?0?0?0(:0?0?0?0){2}:0?0?0?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){2}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/60", "(:(:0?0?0?0){0,2}(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/60", "((:(:0?0?0?0){0,2}(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,2}(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){0,2}|0?0?0?0(:(:0?0?0?0)?(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/59", "0?0?0?0(:0?0?0?0){2}:0?0?[01]?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/59", "0?0?0?0(:0?0?0?0){2}:0?0?[01]?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){2}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/59", "(:(:0?0?0?0){0,2}(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0)?(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?[01]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/59", "((:(:0?0?0?0){0,2}(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0)?(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?[01]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,2}(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){0,2}|0?0?0?0(:(:0?0?0?0)?(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?[01]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/58", "0?0?0?0(:0?0?0?0){2}:0?0?[0-3]?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/58", "0?0?0?0(:0?0?0?0){2}:0?0?[0-3]?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){2}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/58", "(:(:0?0?0?0){0,2}(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0)?(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?[0-3]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/58", "((:(:0?0?0?0){0,2}(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0)?(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?[0-3]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,2}(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){0,2}|0?0?0?0(:(:0?0?0?0)?(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?[0-3]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/57", "0?0?0?0(:0?0?0?0){2}:0?0?[0-7]?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/57", "0?0?0?0(:0?0?0?0){2}:0?0?[0-7]?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){2}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/57", "(:(:0?0?0?0){0,2}(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0)?(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?[0-7]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/57", "((:(:0?0?0?0){0,2}(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0)?(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?[0-7]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,2}(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){0,2}|0?0?0?0(:(:0?0?0?0)?(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?[0-7]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/56", "0?0?0?0(:0?0?0?0){2}:0?0?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/56", "0?0?0?0(:0?0?0?0){2}:0?0?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){2}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/56", "(:(:0?0?0?0){0,2}(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0)?(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/56", "((:(:0?0?0?0){0,2}(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0)?(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,2}(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){0,2}|0?0?0?0(:(:0?0?0?0)?(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/55", "0?0?0?0(:0?0?0?0){2}:0?[01]?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/55", "0?0?0?0(:0?0?0?0){2}:0?[01]?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){2}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/55", "(:(:0?0?0?0){0,2}(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0)?(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?[01]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/55", "((:(:0?0?0?0){0,2}(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0)?(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?[01]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,2}(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){0,2}|0?0?0?0(:(:0?0?0?0)?(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,2}|:0?[01]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/54", "0?0?0?0(:0?0?0?0){2}:0?[0-3]?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/54", "0?0?0?0(:0?0?0?0){2}:0?[0-3]?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){2}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/54", "(:(:0?0?0?0){0,2}(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0)?(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?[0-3]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/54", "((:(:0?0?0?0){0,2}(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0)?(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?[0-3]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,2}(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){0,2}|0?0?0?0(:(:0?0?0?0)?(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,2}|:0?[0-3]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/53", "0?0?0?0(:0?0?0?0){2}:0?[0-7]?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/53", "0?0?0?0(:0?0?0?0){2}:0?[0-7]?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){2}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/53", "(:(:0?0?0?0){0,2}(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0)?(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?[0-7]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/53", "((:(:0?0?0?0){0,2}(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0)?(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?[0-7]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,2}(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){0,2}|0?0?0?0(:(:0?0?0?0)?(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,2}|:0?[0-7]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/52", "0?0?0?0(:0?0?0?0){2}:0?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/52", "0?0?0?0(:0?0?0?0){2}:0?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){2}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/52", "(:(:0?0?0?0){0,2}(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0)?(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/52", "((:(:0?0?0?0){0,2}(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0)?(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,2}(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){0,2}|0?0?0?0(:(:0?0?0?0)?(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,2}|:0?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/51", "0?0?0?0(:0?0?0?0){2}:[01]?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/51", "0?0?0?0(:0?0?0?0){2}:[01]?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){2}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/51", "(:(:0?0?0?0){0,2}(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0)?(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[01]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/51", "((:(:0?0?0?0){0,2}(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0)?(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[01]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,2}(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){0,2}|0?0?0?0(:(:0?0?0?0)?(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,2}|:[01]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/50", "0?0?0?0(:0?0?0?0){2}:[0-3]?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/50", "0?0?0?0(:0?0?0?0){2}:[0-3]?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){2}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/50", "(:(:0?0?0?0){0,2}(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0)?(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-3]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/50", "((:(:0?0?0?0){0,2}(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0)?(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-3]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,2}(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){0,2}|0?0?0?0(:(:0?0?0?0)?(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-3]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/49", "0?0?0?0(:0?0?0?0){2}:[0-7]?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){4}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/49", "0?0?0?0(:0?0?0?0){2}:[0-7]?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){2}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/49", "(:(:0?0?0?0){0,2}(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0)?(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-7]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/49", "((:(:0?0?0?0){0,2}(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,4}|::|0?0?0?0(:(:0?0?0?0)?(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,4}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-7]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,2}(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){0,2}|0?0?0?0(:(:0?0?0?0)?(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){0,2}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-7]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/48", "0?0?0?0(:0?0?0?0){2}(:[0-9A-Fa-f]{1,4}){5}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/48", "0?0?0?0(:0?0?0?0){2}(:[0-9A-Fa-f]{1,4}){3}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/48", "(:(:0?0?0?0){0,2}(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:0?0?0?0)?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/48", "((:(:0?0?0?0){0,2}(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:0?0?0?0)?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0){0,2}(:[0-9A-Fa-f]{1,4}){0,3}|0?0?0?0(:(:0?0?0?0)?(:[0-9A-Fa-f]{1,4}){0,3}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,3}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/47", "0?0?0?0:0?0?0?0:0?0?0?[01](:[0-9A-Fa-f]{1,4}){5}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/47", "0?0?0?0:0?0?0?0:0?0?0?[01](:[0-9A-Fa-f]{1,4}){3}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/47", "(:(:0?0?0?0)?(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?[01](:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/47", "((:(:0?0?0?0)?(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?[01](:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0)?(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){0,3}|0?0?0?0(:(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){0,3}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,3}|:0?0?0?[01](:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/46", "0?0?0?0:0?0?0?0:0?0?0?[0-3](:[0-9A-Fa-f]{1,4}){5}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/46", "0?0?0?0:0?0?0?0:0?0?0?[0-3](:[0-9A-Fa-f]{1,4}){3}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/46", "(:(:0?0?0?0)?(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?[0-3](:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/46", "((:(:0?0?0?0)?(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?[0-3](:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0)?(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){0,3}|0?0?0?0(:(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){0,3}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,3}|:0?0?0?[0-3](:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/45", "0?0?0?0:0?0?0?0:0?0?0?[0-7](:[0-9A-Fa-f]{1,4}){5}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/45", "0?0?0?0:0?0?0?0:0?0?0?[0-7](:[0-9A-Fa-f]{1,4}){3}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/45", "(:(:0?0?0?0)?(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?[0-7](:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/45", "((:(:0?0?0?0)?(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?[0-7](:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0)?(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){0,3}|0?0?0?0(:(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){0,3}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,3}|:0?0?0?[0-7](:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/44", "0?0?0?0:0?0?0?0:0?0?0?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){5}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/44", "0?0?0?0:0?0?0?0:0?0?0?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){3}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/44", "(:(:0?0?0?0)?(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/44", "((:(:0?0?0?0)?(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0)?(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){0,3}|0?0?0?0(:(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){0,3}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,3}|:0?0?0?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/43", "0?0?0?0:0?0?0?0:0?0?[01]?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){5}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/43", "0?0?0?0:0?0?0?0:0?0?[01]?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){3}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/43", "(:(:0?0?0?0)?(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?[01]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/43", "((:(:0?0?0?0)?(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?[01]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0)?(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){0,3}|0?0?0?0(:(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){0,3}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,3}|:0?0?[01]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/42", "0?0?0?0:0?0?0?0:0?0?[0-3]?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){5}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/42", "0?0?0?0:0?0?0?0:0?0?[0-3]?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){3}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/42", "(:(:0?0?0?0)?(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?[0-3]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/42", "((:(:0?0?0?0)?(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?[0-3]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0)?(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){0,3}|0?0?0?0(:(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){0,3}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,3}|:0?0?[0-3]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/41", "0?0?0?0:0?0?0?0:0?0?[0-7]?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){5}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/41", "0?0?0?0:0?0?0?0:0?0?[0-7]?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){3}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/41", "(:(:0?0?0?0)?(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?[0-7]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/41", "((:(:0?0?0?0)?(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?[0-7]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0)?(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){0,3}|0?0?0?0(:(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){0,3}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,3}|:0?0?[0-7]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/40", "0?0?0?0:0?0?0?0:0?0?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){5}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/40", "0?0?0?0:0?0?0?0:0?0?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){3}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/40", "(:(:0?0?0?0)?(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/40", "((:(:0?0?0?0)?(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0)?(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){0,3}|0?0?0?0(:(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){0,3}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,3}|:0?0?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/39", "0?0?0?0:0?0?0?0:0?[01]?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){5}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/39", "0?0?0?0:0?0?0?0:0?[01]?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){3}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/39", "(:(:0?0?0?0)?(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?[01]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/39", "((:(:0?0?0?0)?(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?[01]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0)?(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){0,3}|0?0?0?0(:(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){0,3}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,3}|:0?[01]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/38", "0?0?0?0:0?0?0?0:0?[0-3]?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){5}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/38", "0?0?0?0:0?0?0?0:0?[0-3]?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){3}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/38", "(:(:0?0?0?0)?(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?[0-3]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/38", "((:(:0?0?0?0)?(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?[0-3]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0)?(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){0,3}|0?0?0?0(:(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){0,3}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,3}|:0?[0-3]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/37", "0?0?0?0:0?0?0?0:0?[0-7]?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){5}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/37", "0?0?0?0:0?0?0?0:0?[0-7]?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){3}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/37", "(:(:0?0?0?0)?(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?[0-7]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/37", "((:(:0?0?0?0)?(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?[0-7]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0)?(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){0,3}|0?0?0?0(:(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){0,3}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,3}|:0?[0-7]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/36", "0?0?0?0:0?0?0?0:0?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){5}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/36", "0?0?0?0:0?0?0?0:0?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){3}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/36", "(:(:0?0?0?0)?(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/36", "((:(:0?0?0?0)?(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0)?(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){0,3}|0?0?0?0(:(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){0,3}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,3}|:0?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/35", "0?0?0?0:0?0?0?0:[01]?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){5}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/35", "0?0?0?0:0?0?0?0:[01]?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){3}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/35", "(:(:0?0?0?0)?(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[01]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/35", "((:(:0?0?0?0)?(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[01]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0)?(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){0,3}|0?0?0?0(:(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){0,3}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,3}|:[01]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/34", "0?0?0?0:0?0?0?0:[0-3]?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){5}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/34", "0?0?0?0:0?0?0?0:[0-3]?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){3}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/34", "(:(:0?0?0?0)?(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-3]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/34", "((:(:0?0?0?0)?(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-3]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0)?(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){0,3}|0?0?0?0(:(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){0,3}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-3]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/33", "0?0?0?0:0?0?0?0:[0-7]?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){5}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/33", "0?0?0?0:0?0?0?0:[0-7]?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){3}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/33", "(:(:0?0?0?0)?(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-7]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/33", "((:(:0?0?0?0)?(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,5}|::|0?0?0?0(:(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,5}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-7]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0)?(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){0,3}|0?0?0?0(:(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){0,3}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-7]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/32", "0?0?0?0:0?0?0?0(:[0-9A-Fa-f]{1,4}){6}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/32", "0?0?0?0:0?0?0?0(:[0-9A-Fa-f]{1,4}){4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/32", "(:(:0?0?0?0)?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/32", "((:(:0?0?0?0)?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?0)?(:[0-9A-Fa-f]{1,4}){0,4}|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,4}|:0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/31", "0?0?0?0:0?0?0?[01](:[0-9A-Fa-f]{1,4}){6}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/31", "0?0?0?0:0?0?0?[01](:[0-9A-Fa-f]{1,4}){4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/31", "(:(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:0?0?0?[01](:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/31", "((:(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:0?0?0?[01](:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?[01])?(:[0-9A-Fa-f]{1,4}){0,4}|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,4}|:0?0?0?[01](:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/30", "0?0?0?0:0?0?0?[0-3](:[0-9A-Fa-f]{1,4}){6}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/30", "0?0?0?0:0?0?0?[0-3](:[0-9A-Fa-f]{1,4}){4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/30", "(:(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:0?0?0?[0-3](:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/30", "((:(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:0?0?0?[0-3](:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?[0-3])?(:[0-9A-Fa-f]{1,4}){0,4}|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,4}|:0?0?0?[0-3](:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/29", "0?0?0?0:0?0?0?[0-7](:[0-9A-Fa-f]{1,4}){6}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/29", "0?0?0?0:0?0?0?[0-7](:[0-9A-Fa-f]{1,4}){4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/29", "(:(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:0?0?0?[0-7](:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/29", "((:(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:0?0?0?[0-7](:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?[0-7])?(:[0-9A-Fa-f]{1,4}){0,4}|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,4}|:0?0?0?[0-7](:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/28", "0?0?0?0:0?0?0?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){6}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/28", "0?0?0?0:0?0?0?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/28", "(:(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:0?0?0?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/28", "((:(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:0?0?0?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?0?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){0,4}|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,4}|:0?0?0?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/27", "0?0?0?0:0?0?[01]?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){6}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/27", "0?0?0?0:0?0?[01]?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/27", "(:(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:0?0?[01]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/27", "((:(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:0?0?[01]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?[01]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){0,4}|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,4}|:0?0?[01]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/26", "0?0?0?0:0?0?[0-3]?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){6}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/26", "0?0?0?0:0?0?[0-3]?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/26", "(:(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:0?0?[0-3]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/26", "((:(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:0?0?[0-3]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?[0-3]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){0,4}|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,4}|:0?0?[0-3]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/25", "0?0?0?0:0?0?[0-7]?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){6}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/25", "0?0?0?0:0?0?[0-7]?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/25", "(:(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:0?0?[0-7]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/25", "((:(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:0?0?[0-7]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?[0-7]?[0-9A-Fa-f])?(:[0-9A-Fa-f]{1,4}){0,4}|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,4}|:0?0?[0-7]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/24", "0?0?0?0:0?0?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){6}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/24", "0?0?0?0:0?0?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/24", "(:(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:0?0?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/24", "((:(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:0?0?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?0?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){0,4}|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,4}|:0?0?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/23", "0?0?0?0:0?[01]?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){6}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/23", "0?0?0?0:0?[01]?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/23", "(:(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:0?[01]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/23", "((:(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:0?[01]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?[01]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){0,4}|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,4}|:0?[01]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/22", "0?0?0?0:0?[0-3]?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){6}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/22", "0?0?0?0:0?[0-3]?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/22", "(:(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:0?[0-3]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/22", "((:(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:0?[0-3]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?[0-3]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){0,4}|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,4}|:0?[0-3]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/21", "0?0?0?0:0?[0-7]?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){6}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/21", "0?0?0?0:0?[0-7]?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/21", "(:(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:0?[0-7]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/21", "((:(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:0?[0-7]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?[0-7]?[0-9A-Fa-f]{1,2})?(:[0-9A-Fa-f]{1,4}){0,4}|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,4}|:0?[0-7]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/20", "0?0?0?0:0?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){6}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/20", "0?0?0?0:0?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/20", "(:(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:0?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/20", "((:(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:0?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:0?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){0,4}|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,4}|:0?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/19", "0?0?0?0:[01]?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){6}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/19", "0?0?0?0:[01]?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/19", "(:(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[01]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/19", "((:(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[01]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:[01]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){0,4}|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,4}|:[01]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/18", "0?0?0?0:[0-3]?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){6}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/18", "0?0?0?0:[0-3]?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/18", "(:(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-3]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/18", "((:(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-3]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:[0-3]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){0,4}|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,4}|:[0-3]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/17", "0?0?0?0:[0-7]?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){6}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/17", "0?0?0?0:[0-7]?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){4}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/17", "(:(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-7]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/17", "((:(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){1,6}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-7]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:[0-7]?[0-9A-Fa-f]{1,3})?(:[0-9A-Fa-f]{1,4}){0,4}|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,4}|:[0-7]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/16", "0?0?0?0(:[0-9A-Fa-f]{1,4}){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/16", "0?0?0?0(:[0-9A-Fa-f]{1,4}){5}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/16", "(:(:[0-9A-Fa-f]{1,4}){1,7}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/16", "((:(:[0-9A-Fa-f]{1,4}){1,7}|::|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:[0-9A-Fa-f]{1,4}){0,5}|0?0?0?0(:(:[0-9A-Fa-f]{1,4}){0,4}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/15", "0?0?0?[01](:[0-9A-Fa-f]{1,4}){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/15", "0?0?0?[01](:[0-9A-Fa-f]{1,4}){5}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/15", "(:(:[0-9A-Fa-f]{1,4}){1,7}|::|0?0?0?[01](:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/15", "((:(:[0-9A-Fa-f]{1,4}){1,7}|::|0?0?0?[01](:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:[0-9A-Fa-f]{1,4}){0,5}|0?0?0?[01](:(:[0-9A-Fa-f]{1,4}){0,4}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/14", "0?0?0?[0-3](:[0-9A-Fa-f]{1,4}){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/14", "0?0?0?[0-3](:[0-9A-Fa-f]{1,4}){5}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/14", "(:(:[0-9A-Fa-f]{1,4}){1,7}|::|0?0?0?[0-3](:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/14", "((:(:[0-9A-Fa-f]{1,4}){1,7}|::|0?0?0?[0-3](:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:[0-9A-Fa-f]{1,4}){0,5}|0?0?0?[0-3](:(:[0-9A-Fa-f]{1,4}){0,4}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/13", "0?0?0?[0-7](:[0-9A-Fa-f]{1,4}){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/13", "0?0?0?[0-7](:[0-9A-Fa-f]{1,4}){5}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/13", "(:(:[0-9A-Fa-f]{1,4}){1,7}|::|0?0?0?[0-7](:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/13", "((:(:[0-9A-Fa-f]{1,4}){1,7}|::|0?0?0?[0-7](:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:[0-9A-Fa-f]{1,4}){0,5}|0?0?0?[0-7](:(:[0-9A-Fa-f]{1,4}){0,4}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/12", "0?0?0?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/12", "0?0?0?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){5}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/12", "(:(:[0-9A-Fa-f]{1,4}){1,7}|::|0?0?0?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/12", "((:(:[0-9A-Fa-f]{1,4}){1,7}|::|0?0?0?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:[0-9A-Fa-f]{1,4}){0,5}|0?0?0?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){0,4}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/11", "0?0?[01]?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/11", "0?0?[01]?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){5}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/11", "(:(:[0-9A-Fa-f]{1,4}){1,7}|::|0?0?[01]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/11", "((:(:[0-9A-Fa-f]{1,4}){1,7}|::|0?0?[01]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:[0-9A-Fa-f]{1,4}){0,5}|0?0?[01]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){0,4}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/10", "0?0?[0-3]?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/10", "0?0?[0-3]?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){5}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/10", "(:(:[0-9A-Fa-f]{1,4}){1,7}|::|0?0?[0-3]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/10", "((:(:[0-9A-Fa-f]{1,4}){1,7}|::|0?0?[0-3]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:[0-9A-Fa-f]{1,4}){0,5}|0?0?[0-3]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){0,4}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/9", "0?0?[0-7]?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/9", "0?0?[0-7]?[0-9A-Fa-f](:[0-9A-Fa-f]{1,4}){5}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/9", "(:(:[0-9A-Fa-f]{1,4}){1,7}|::|0?0?[0-7]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/9", "((:(:[0-9A-Fa-f]{1,4}){1,7}|::|0?0?[0-7]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:[0-9A-Fa-f]{1,4}){0,5}|0?0?[0-7]?[0-9A-Fa-f](:(:[0-9A-Fa-f]{1,4}){0,4}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/8", "0?0?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/8", "0?0?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){5}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/8", "(:(:[0-9A-Fa-f]{1,4}){1,7}|::|0?0?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/8", "((:(:[0-9A-Fa-f]{1,4}){1,7}|::|0?0?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:[0-9A-Fa-f]{1,4}){0,5}|0?0?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){0,4}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/7", "0?[01]?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/7", "0?[01]?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){5}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/7", "(:(:[0-9A-Fa-f]{1,4}){1,7}|::|0?[01]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/7", "((:(:[0-9A-Fa-f]{1,4}){1,7}|::|0?[01]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:[0-9A-Fa-f]{1,4}){0,5}|0?[01]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){0,4}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/6", "0?[0-3]?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/6", "0?[0-3]?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){5}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/6", "(:(:[0-9A-Fa-f]{1,4}){1,7}|::|0?[0-3]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/6", "((:(:[0-9A-Fa-f]{1,4}){1,7}|::|0?[0-3]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:[0-9A-Fa-f]{1,4}){0,5}|0?[0-3]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){0,4}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/5", "0?[0-7]?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/5", "0?[0-7]?[0-9A-Fa-f]{1,2}(:[0-9A-Fa-f]{1,4}){5}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/5", "(:(:[0-9A-Fa-f]{1,4}){1,7}|::|0?[0-7]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/5", "((:(:[0-9A-Fa-f]{1,4}){1,7}|::|0?[0-7]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:[0-9A-Fa-f]{1,4}){0,5}|0?[0-7]?[0-9A-Fa-f]{1,2}(:(:[0-9A-Fa-f]{1,4}){0,4}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/4", "0?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/4", "0?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){5}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/4", "(:(:[0-9A-Fa-f]{1,4}){1,7}|::|0?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/4", "((:(:[0-9A-Fa-f]{1,4}){1,7}|::|0?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:[0-9A-Fa-f]{1,4}){0,5}|0?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){0,4}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/3", "[01]?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/3", "[01]?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){5}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/3", "(:(:[0-9A-Fa-f]{1,4}){1,7}|::|[01]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/3", "((:(:[0-9A-Fa-f]{1,4}){1,7}|::|[01]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:[0-9A-Fa-f]{1,4}){0,5}|[01]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){0,4}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/2", "[0-3]?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/2", "[0-3]?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){5}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/2", "(:(:[0-9A-Fa-f]{1,4}){1,7}|::|[0-3]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/2", "((:(:[0-9A-Fa-f]{1,4}){1,7}|::|[0-3]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:[0-9A-Fa-f]{1,4}){0,5}|[0-3]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){0,4}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/1", "[0-7]?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/1", "[0-7]?[0-9A-Fa-f]{1,3}(:[0-9A-Fa-f]{1,4}){5}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/1", "(:(:[0-9A-Fa-f]{1,4}){1,7}|::|[0-7]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/1", "((:(:[0-9A-Fa-f]{1,4}){1,7}|::|[0-7]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:[0-9A-Fa-f]{1,4}){0,5}|[0-7]?[0-9A-Fa-f]{1,3}(:(:[0-9A-Fa-f]{1,4}){0,4}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
    { "::0/0", "[0-9A-Fa-f]{1,4}(:[0-9A-Fa-f]{1,4}){7}", RGXG_NOZEROCOMPRESSION|RGXG_NOMIXEDNOTATION },
    { "::0/0", "[0-9A-Fa-f]{1,4}(:[0-9A-Fa-f]{1,4}){5}((:[0-9A-Fa-f]{1,4}){2}|:(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", RGXG_NOZEROCOMPRESSION },
    { "::0/0", "(:(:[0-9A-Fa-f]{1,4}){1,7}|::|[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))", RGXG_NOMIXEDNOTATION },
    { "::0/0", "((:(:[0-9A-Fa-f]{1,4}){1,7}|::|[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,6}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,5}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,4}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,3}|::|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){1,2}|::|:[0-9A-Fa-f]{1,4}(::[0-9A-Fa-f]{1,4}|::|:[0-9A-Fa-f]{1,4}(::|:[0-9A-Fa-f]{1,4}))))))))|(:(:[0-9A-Fa-f]{1,4}){0,5}|[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,4}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,3}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4}){0,2}|:[0-9A-Fa-f]{1,4}(:(:[0-9A-Fa-f]{1,4})?|:[0-9A-Fa-f]{1,4}(:|:[0-9A-Fa-f]{1,4})))))):(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])(\\.(25[0-5]|2[0-4][0-9]|1[0-9]{2}|[1-9]?[0-9])){3})", 0 },
};

static int num_cidr_regex = sizeof cidr_regex / sizeof(cidr_test_t);

START_TEST (test_cidr_regex) {
    char *regex = NULL;
    int n = rgxg_net_cidr_string(cidr_regex[_i].cidr, NULL, regex, cidr_regex[_i].options);
    if (n > 0) {
        regex = malloc(sizeof(char) * (n+1)); /* freed below */
        rgxg_net_cidr_string(cidr_regex[_i].cidr, NULL, regex, cidr_regex[_i].options);
        regex[n] = '\0';
        ck_assert(regex != NULL);
        ck_assert_msg(strcmp(cidr_regex[_i].expected_regex, regex) == 0, "rgxg_net_cidr_string: '%s' (options: %d): regex returned '%s' != '%s'", cidr_regex[_i].cidr, cidr_regex[_i].options, regex, cidr_regex[_i].expected_regex);
        free(regex);
    } else {
        ck_abort_msg("rgxg_net_cidr_string: unexpected return value %d", n);
    }
}
END_TEST

Suite *make_rgxg_net_suite(void) {

    Suite *s = suite_create ("rgxg/net");

    TCase *tc_rgxg_net_cidr_string = tcase_create ("rgxg_net_cidr_string");

    tcase_add_loop_test (tc_rgxg_net_cidr_string, test_cidr_regex, 0, num_cidr_regex);

    suite_add_tcase (s, tc_rgxg_net_cidr_string);

    return s;
}
