% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cSpline.R
\name{cSpline}
\alias{cSpline}
\title{C-Spline Basis for Polynomial Splines}
\usage{
cSpline(
  x,
  df = NULL,
  knots = NULL,
  degree = 3L,
  intercept = TRUE,
  Boundary.knots = NULL,
  derivs = 0L,
  scale = TRUE,
  ...
)
}
\arguments{
\item{x}{The predictor variable.  Missing values are allowed and will be
returned as they are.}

\item{df}{Degree of freedom that equals to the column number of the returned
matrix.  One can specify \code{df} rather than \code{knots}, then the
function chooses \code{df - degree - as.integer(intercept)} internal
knots at suitable quantiles of \code{x} ignoring missing values and
those \code{x} outside of the boundary.  If internal knots are specified
via \code{knots}, the specified \code{df} will be ignored.}

\item{knots}{The internal breakpoints that define the splines.  The default
is \code{NULL}, which results in a basis for ordinary polynomial
regression.  Typical values are the mean or median for one knot,
quantiles for more knots.}

\item{degree}{The degree of C-spline defined to be the degree of the
associated M-spline instead of actual polynomial degree. For example,
C-spline basis of degree 2 is defined as the scaled double integral of
associated M-spline basis of degree 2.}

\item{intercept}{If \code{TRUE} by default, all spline bases are included.
Notice that when using C-Spline for shape-restricted regression,
\code{intercept = TRUE} should be set even when an intercept term is
considered additional to the spline bases in the model.}

\item{Boundary.knots}{Boundary points at which to anchor the splines.  By
default, they are the range of \code{x} excluding \code{NA}.  If both
\code{knots} and \code{Boundary.knots} are supplied, the basis
parameters do not depend on \code{x}. Data can extend beyond
\code{Boundary.knots}.}

\item{derivs}{A non-negative integer specifying the order of derivatives of
C-splines. The default value is \code{0L} for C-spline bases.}

\item{scale}{A logical value indicating if scaling C-splines is required. If
\code{TRUE} by default, each C-spline basis is scaled to have unit
height at right boundary knot; the corresponding I-spline and M-spline
basis matrices shipped in attributes are also scaled to the same extent.}

\item{...}{Optional arguments that are not used.}
}
\value{
A numeric matrix of \code{length(x)} rows and \code{df} columns if
    \code{df} is specified or \code{length(knots) + degree +
    as.integer(intercept)} columns if \code{knots} are specified instead.
    Attributes that correspond to the arguments specified are returned
    mainly for other functions in this package.
}
\description{
Generates the convex regression spline (called C-spline) basis matrix by
integrating I-spline basis for a polynomial spline or the corresponding
derivatives.
}
\details{
It is an implementation of the close form C-spline basis derived from the
recursion formula of I-splines and M-splines.
}
\examples{
library(splines2)

x <- seq.int(0, 1, 0.01)
knots <- c(0.3, 0.5, 0.6)

### when 'scale = TRUE' (by default)
csMat <- cSpline(x, knots = knots, degree = 2)

op <- par(mar = c(2.5, 2.5, 0.2, 0.1), mgp = c(1.5, 0.5, 0))
matplot(x, csMat, type = "l", ylab = "C-spline basis")
abline(v = knots, lty = 2, col = "gray")
isMat <- deriv(csMat)
msMat <- deriv(csMat, derivs = 2)
matplot(x, isMat, type = "l", ylab = "scaled I-spline basis")
matplot(x, msMat, type = "l", ylab = "scaled M-spline basis")

## reset to previous plotting settings
par(op)

### when 'scale = FALSE'
csMat <- cSpline(x, knots = knots, degree = 2, scale = FALSE)

## the corresponding I-splines and M-splines (with same arguments)
isMat <- iSpline(x, knots = knots, degree = 2)
msMat <- mSpline(x, knots = knots, degree = 2, intercept = TRUE)

## or using deriv methods (more efficient)
isMat1 <- deriv(csMat)
msMat1 <- deriv(csMat, derivs = 2)

## equivalent
stopifnot(all.equal(isMat, isMat1, check.attributes = FALSE))
stopifnot(all.equal(msMat, msMat1, check.attributes = FALSE))
}
\references{
Meyer, M. C. (2008). Inference using shape-restricted regression splines.
\emph{The Annals of Applied Statistics}, 2(3), 1013--1033.
}
\seealso{
\code{\link{iSpline}} for I-splines;
\code{\link{mSpline}} for M-splines.
}
