// This file is part of AsmJit project <https://asmjit.com>
//
// See asmjit.h or LICENSE.md for license and copyright information
// SPDX-License-Identifier: Zlib

#include <asmjit/core.h>
#if !defined(ASMJIT_NO_X86)

#include <asmjit/x86.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "asmjit_test_assembler.h"
#include "cmdline.h"

using namespace asmjit;

#define TEST_INSTRUCTION(OPCODE, ...) \
  tester.testValidInstruction(#__VA_ARGS__, OPCODE, tester.assembler.__VA_ARGS__)

#define FAIL_INSTRUCTION(ExpectedError, ...) \
  tester.testInvalidInstruction(#__VA_ARGS__, ExpectedError, tester.assembler.__VA_ARGS__)

static void ASMJIT_NOINLINE testX86AssemblerBase(AssemblerTester<x86::Assembler>& tester) noexcept {
  using namespace x86;

  TEST_INSTRUCTION("37"                            , aaa(ax));
  TEST_INSTRUCTION("D501"                          , aad(ax, 1));
  TEST_INSTRUCTION("D401"                          , aam(ax, 1));
  TEST_INSTRUCTION("3F"                            , aas(ax));
  TEST_INSTRUCTION("80D101"                        , adc(cl, 1));
  TEST_INSTRUCTION("80D501"                        , adc(ch, 1));
  TEST_INSTRUCTION("8094118000000001"              , adc(byte_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("6683D101"                      , adc(cx, 1));
  TEST_INSTRUCTION("668394118000000001"            , adc(word_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("83D101"                        , adc(ecx, 1));
  TEST_INSTRUCTION("8394118000000001"              , adc(dword_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("10D1"                          , adc(cl, dl));
  TEST_INSTRUCTION("10F1"                          , adc(cl, dh));
  TEST_INSTRUCTION("10D5"                          , adc(ch, dl));
  TEST_INSTRUCTION("10F5"                          , adc(ch, dh));
  TEST_INSTRUCTION("109C1180000000"                , adc(ptr(ecx, edx, 0, 128), bl));
  TEST_INSTRUCTION("10BC1180000000"                , adc(ptr(ecx, edx, 0, 128), bh));
  TEST_INSTRUCTION("109C1180000000"                , adc(byte_ptr(ecx, edx, 0, 128), bl));
  TEST_INSTRUCTION("10BC1180000000"                , adc(byte_ptr(ecx, edx, 0, 128), bh));
  TEST_INSTRUCTION("6611D1"                        , adc(cx, dx));
  TEST_INSTRUCTION("66119C1180000000"              , adc(ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("66119C1180000000"              , adc(word_ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("11D1"                          , adc(ecx, edx));
  TEST_INSTRUCTION("119C1180000000"                , adc(ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("119C1180000000"                , adc(dword_ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("128C1A80000000"                , adc(cl, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("128C1A80000000"                , adc(cl, byte_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("12AC1A80000000"                , adc(ch, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("12AC1A80000000"                , adc(ch, byte_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("66138C1A80000000"              , adc(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("66138C1A80000000"              , adc(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("138C1A80000000"                , adc(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("138C1A80000000"                , adc(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("80C101"                        , add(cl, 1));
  TEST_INSTRUCTION("80C501"                        , add(ch, 1));
  TEST_INSTRUCTION("8084118000000001"              , add(byte_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("6683C101"                      , add(cx, 1));
  TEST_INSTRUCTION("668384118000000001"            , add(word_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("83C101"                        , add(ecx, 1));
  TEST_INSTRUCTION("8384118000000001"              , add(dword_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("00D1"                          , add(cl, dl));
  TEST_INSTRUCTION("00F1"                          , add(cl, dh));
  TEST_INSTRUCTION("00D5"                          , add(ch, dl));
  TEST_INSTRUCTION("00F5"                          , add(ch, dh));
  TEST_INSTRUCTION("009C1180000000"                , add(ptr(ecx, edx, 0, 128), bl));
  TEST_INSTRUCTION("00BC1180000000"                , add(ptr(ecx, edx, 0, 128), bh));
  TEST_INSTRUCTION("009C1180000000"                , add(byte_ptr(ecx, edx, 0, 128), bl));
  TEST_INSTRUCTION("00BC1180000000"                , add(byte_ptr(ecx, edx, 0, 128), bh));
  TEST_INSTRUCTION("6601D1"                        , add(cx, dx));
  TEST_INSTRUCTION("66019C1180000000"              , add(ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("66019C1180000000"              , add(word_ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("01D1"                          , add(ecx, edx));
  TEST_INSTRUCTION("019C1180000000"                , add(ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("019C1180000000"                , add(dword_ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("028C1A80000000"                , add(cl, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("028C1A80000000"                , add(cl, byte_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("02AC1A80000000"                , add(ch, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("02AC1A80000000"                , add(ch, byte_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("66038C1A80000000"              , add(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("66038C1A80000000"              , add(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("038C1A80000000"                , add(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("038C1A80000000"                , add(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("80E101"                        , and_(cl, 1));
  TEST_INSTRUCTION("80E501"                        , and_(ch, 1));
  TEST_INSTRUCTION("6683E101"                      , and_(cx, 1));
  TEST_INSTRUCTION("83E101"                        , and_(ecx, 1));
  TEST_INSTRUCTION("80A4118000000001"              , and_(byte_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("6683A4118000000001"            , and_(word_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("83A4118000000001"              , and_(dword_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("20D1"                          , and_(cl, dl));
  TEST_INSTRUCTION("20F1"                          , and_(cl, dh));
  TEST_INSTRUCTION("20D5"                          , and_(ch, dl));
  TEST_INSTRUCTION("20F5"                          , and_(ch, dh));
  TEST_INSTRUCTION("209C1180000000"                , and_(ptr(ecx, edx, 0, 128), bl));
  TEST_INSTRUCTION("20BC1180000000"                , and_(ptr(ecx, edx, 0, 128), bh));
  TEST_INSTRUCTION("209C1180000000"                , and_(byte_ptr(ecx, edx, 0, 128), bl));
  TEST_INSTRUCTION("20BC1180000000"                , and_(byte_ptr(ecx, edx, 0, 128), bh));
  TEST_INSTRUCTION("6621D1"                        , and_(cx, dx));
  TEST_INSTRUCTION("66219C1180000000"              , and_(ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("66219C1180000000"              , and_(word_ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("21D1"                          , and_(ecx, edx));
  TEST_INSTRUCTION("219C1180000000"                , and_(ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("219C1180000000"                , and_(dword_ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("228C1A80000000"                , and_(cl, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("228C1A80000000"                , and_(cl, byte_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("22AC1A80000000"                , and_(ch, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("22AC1A80000000"                , and_(ch, byte_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("66238C1A80000000"              , and_(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("66238C1A80000000"              , and_(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("238C1A80000000"                , and_(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("238C1A80000000"                , and_(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("63D1"                          , arpl(cx, dx));
  TEST_INSTRUCTION("639C1180000000"                , arpl(ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("639C1180000000"                , arpl(word_ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("66628C1A80000000"              , bound(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("66628C1A80000000"              , bound(cx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("628C1A80000000"                , bound(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("628C1A80000000"                , bound(ecx, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FBCCA"                      , bsf(cx, dx));
  TEST_INSTRUCTION("660FBC8C1A80000000"            , bsf(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FBC8C1A80000000"            , bsf(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FBCCA"                        , bsf(ecx, edx));
  TEST_INSTRUCTION("0FBC8C1A80000000"              , bsf(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FBC8C1A80000000"              , bsf(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FBDCA"                      , bsr(cx, dx));
  TEST_INSTRUCTION("660FBD8C1A80000000"            , bsr(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FBD8C1A80000000"            , bsr(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FBDCA"                        , bsr(ecx, edx));
  TEST_INSTRUCTION("0FBD8C1A80000000"              , bsr(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FBD8C1A80000000"              , bsr(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FC9"                        , bswap(cx));
  TEST_INSTRUCTION("0FC9"                          , bswap(ecx));
  TEST_INSTRUCTION("660FBAE101"                    , bt(cx, 1));
  TEST_INSTRUCTION("0FBAE101"                      , bt(ecx, 1));
  TEST_INSTRUCTION("660FBAA4118000000001"          , bt(word_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("0FBAA4118000000001"            , bt(dword_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("660FA3D1"                      , bt(cx, dx));
  TEST_INSTRUCTION("660FA39C1180000000"            , bt(ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("660FA39C1180000000"            , bt(word_ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("0FA3D1"                        , bt(ecx, edx));
  TEST_INSTRUCTION("0FA39C1180000000"              , bt(ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("0FA39C1180000000"              , bt(dword_ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("660FBAF901"                    , btc(cx, 1));
  TEST_INSTRUCTION("0FBAF901"                      , btc(ecx, 1));
  TEST_INSTRUCTION("660FBABC118000000001"          , btc(word_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("0FBABC118000000001"            , btc(dword_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("660FBBD1"                      , btc(cx, dx));
  TEST_INSTRUCTION("660FBB9C1180000000"            , btc(ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("660FBB9C1180000000"            , btc(word_ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("0FBBD1"                        , btc(ecx, edx));
  TEST_INSTRUCTION("0FBB9C1180000000"              , btc(ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("0FBB9C1180000000"              , btc(dword_ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("660FBAF101"                    , btr(cx, 1));
  TEST_INSTRUCTION("0FBAF101"                      , btr(ecx, 1));
  TEST_INSTRUCTION("660FBAB4118000000001"          , btr(word_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("0FBAB4118000000001"            , btr(dword_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("660FB3D1"                      , btr(cx, dx));
  TEST_INSTRUCTION("660FB39C1180000000"            , btr(ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("660FB39C1180000000"            , btr(word_ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("0FB3D1"                        , btr(ecx, edx));
  TEST_INSTRUCTION("0FB39C1180000000"              , btr(ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("0FB39C1180000000"              , btr(dword_ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("660FBAE901"                    , bts(cx, 1));
  TEST_INSTRUCTION("0FBAE901"                      , bts(ecx, 1));
  TEST_INSTRUCTION("660FBAAC118000000001"          , bts(word_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("0FBAAC118000000001"            , bts(dword_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("660FABD1"                      , bts(cx, dx));
  TEST_INSTRUCTION("660FAB9C1180000000"            , bts(ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("660FAB9C1180000000"            , bts(word_ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("0FABD1"                        , bts(ecx, edx));
  TEST_INSTRUCTION("0FAB9C1180000000"              , bts(ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("0FAB9C1180000000"              , bts(dword_ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("6698"                          , cbw(ax));
  TEST_INSTRUCTION("99"                            , cdq(edx, eax));
  TEST_INSTRUCTION("F8"                            , clc());
  TEST_INSTRUCTION("FC"                            , cld());
  TEST_INSTRUCTION("FA"                            , cli());
  TEST_INSTRUCTION("F5"                            , cmc());
  TEST_INSTRUCTION("660F47CA"                      , cmova(cx, dx));
  TEST_INSTRUCTION("660F478C1A80000000"            , cmova(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F478C1A80000000"            , cmova(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F47CA"                        , cmova(ecx, edx));
  TEST_INSTRUCTION("0F478C1A80000000"              , cmova(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F478C1A80000000"              , cmova(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F43CA"                      , cmovae(cx, dx));
  TEST_INSTRUCTION("660F438C1A80000000"            , cmovae(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F438C1A80000000"            , cmovae(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F43CA"                        , cmovae(ecx, edx));
  TEST_INSTRUCTION("0F438C1A80000000"              , cmovae(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F438C1A80000000"              , cmovae(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F42CA"                      , cmovb(cx, dx));
  TEST_INSTRUCTION("660F428C1A80000000"            , cmovb(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F428C1A80000000"            , cmovb(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F42CA"                        , cmovb(ecx, edx));
  TEST_INSTRUCTION("0F428C1A80000000"              , cmovb(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F428C1A80000000"              , cmovb(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F46CA"                      , cmovbe(cx, dx));
  TEST_INSTRUCTION("660F468C1A80000000"            , cmovbe(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F468C1A80000000"            , cmovbe(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F46CA"                        , cmovbe(ecx, edx));
  TEST_INSTRUCTION("0F468C1A80000000"              , cmovbe(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F468C1A80000000"              , cmovbe(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F42CA"                      , cmovc(cx, dx));
  TEST_INSTRUCTION("660F428C1A80000000"            , cmovc(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F428C1A80000000"            , cmovc(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F42CA"                        , cmovc(ecx, edx));
  TEST_INSTRUCTION("0F428C1A80000000"              , cmovc(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F428C1A80000000"              , cmovc(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F44CA"                      , cmove(cx, dx));
  TEST_INSTRUCTION("660F448C1A80000000"            , cmove(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F448C1A80000000"            , cmove(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F44CA"                        , cmove(ecx, edx));
  TEST_INSTRUCTION("0F448C1A80000000"              , cmove(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F448C1A80000000"              , cmove(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F4FCA"                      , cmovg(cx, dx));
  TEST_INSTRUCTION("660F4F8C1A80000000"            , cmovg(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F4F8C1A80000000"            , cmovg(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F4FCA"                        , cmovg(ecx, edx));
  TEST_INSTRUCTION("0F4F8C1A80000000"              , cmovg(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F4F8C1A80000000"              , cmovg(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F4DCA"                      , cmovge(cx, dx));
  TEST_INSTRUCTION("660F4D8C1A80000000"            , cmovge(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F4D8C1A80000000"            , cmovge(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F4DCA"                        , cmovge(ecx, edx));
  TEST_INSTRUCTION("0F4D8C1A80000000"              , cmovge(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F4D8C1A80000000"              , cmovge(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F4CCA"                      , cmovl(cx, dx));
  TEST_INSTRUCTION("660F4C8C1A80000000"            , cmovl(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F4C8C1A80000000"            , cmovl(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F4CCA"                        , cmovl(ecx, edx));
  TEST_INSTRUCTION("0F4C8C1A80000000"              , cmovl(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F4C8C1A80000000"              , cmovl(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F4ECA"                      , cmovle(cx, dx));
  TEST_INSTRUCTION("660F4E8C1A80000000"            , cmovle(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F4E8C1A80000000"            , cmovle(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F4ECA"                        , cmovle(ecx, edx));
  TEST_INSTRUCTION("0F4E8C1A80000000"              , cmovle(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F4E8C1A80000000"              , cmovle(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F46CA"                      , cmovna(cx, dx));
  TEST_INSTRUCTION("660F468C1A80000000"            , cmovna(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F468C1A80000000"            , cmovna(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F46CA"                        , cmovna(ecx, edx));
  TEST_INSTRUCTION("0F468C1A80000000"              , cmovna(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F468C1A80000000"              , cmovna(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F42CA"                      , cmovnae(cx, dx));
  TEST_INSTRUCTION("660F428C1A80000000"            , cmovnae(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F428C1A80000000"            , cmovnae(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F42CA"                        , cmovnae(ecx, edx));
  TEST_INSTRUCTION("0F428C1A80000000"              , cmovnae(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F428C1A80000000"              , cmovnae(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F43CA"                      , cmovnb(cx, dx));
  TEST_INSTRUCTION("660F438C1A80000000"            , cmovnb(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F438C1A80000000"            , cmovnb(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F43CA"                        , cmovnb(ecx, edx));
  TEST_INSTRUCTION("0F438C1A80000000"              , cmovnb(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F438C1A80000000"              , cmovnb(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F47CA"                      , cmovnbe(cx, dx));
  TEST_INSTRUCTION("660F478C1A80000000"            , cmovnbe(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F478C1A80000000"            , cmovnbe(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F47CA"                        , cmovnbe(ecx, edx));
  TEST_INSTRUCTION("0F478C1A80000000"              , cmovnbe(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F478C1A80000000"              , cmovnbe(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F43CA"                      , cmovnc(cx, dx));
  TEST_INSTRUCTION("660F438C1A80000000"            , cmovnc(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F438C1A80000000"            , cmovnc(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F43CA"                        , cmovnc(ecx, edx));
  TEST_INSTRUCTION("0F438C1A80000000"              , cmovnc(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F438C1A80000000"              , cmovnc(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F45CA"                      , cmovne(cx, dx));
  TEST_INSTRUCTION("660F458C1A80000000"            , cmovne(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F458C1A80000000"            , cmovne(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F45CA"                        , cmovne(ecx, edx));
  TEST_INSTRUCTION("0F458C1A80000000"              , cmovne(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F458C1A80000000"              , cmovne(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F4ECA"                      , cmovng(cx, dx));
  TEST_INSTRUCTION("660F4E8C1A80000000"            , cmovng(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F4E8C1A80000000"            , cmovng(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F4ECA"                        , cmovng(ecx, edx));
  TEST_INSTRUCTION("0F4E8C1A80000000"              , cmovng(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F4E8C1A80000000"              , cmovng(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F4CCA"                      , cmovnge(cx, dx));
  TEST_INSTRUCTION("660F4C8C1A80000000"            , cmovnge(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F4C8C1A80000000"            , cmovnge(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F4CCA"                        , cmovnge(ecx, edx));
  TEST_INSTRUCTION("0F4C8C1A80000000"              , cmovnge(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F4C8C1A80000000"              , cmovnge(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F4DCA"                      , cmovnl(cx, dx));
  TEST_INSTRUCTION("660F4D8C1A80000000"            , cmovnl(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F4D8C1A80000000"            , cmovnl(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F4DCA"                        , cmovnl(ecx, edx));
  TEST_INSTRUCTION("0F4D8C1A80000000"              , cmovnl(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F4D8C1A80000000"              , cmovnl(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F4FCA"                      , cmovnle(cx, dx));
  TEST_INSTRUCTION("660F4F8C1A80000000"            , cmovnle(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F4F8C1A80000000"            , cmovnle(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F4FCA"                        , cmovnle(ecx, edx));
  TEST_INSTRUCTION("0F4F8C1A80000000"              , cmovnle(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F4F8C1A80000000"              , cmovnle(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F41CA"                      , cmovno(cx, dx));
  TEST_INSTRUCTION("660F418C1A80000000"            , cmovno(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F418C1A80000000"            , cmovno(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F41CA"                        , cmovno(ecx, edx));
  TEST_INSTRUCTION("0F418C1A80000000"              , cmovno(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F418C1A80000000"              , cmovno(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F4BCA"                      , cmovnp(cx, dx));
  TEST_INSTRUCTION("660F4B8C1A80000000"            , cmovnp(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F4B8C1A80000000"            , cmovnp(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F4BCA"                        , cmovnp(ecx, edx));
  TEST_INSTRUCTION("0F4B8C1A80000000"              , cmovnp(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F4B8C1A80000000"              , cmovnp(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F49CA"                      , cmovns(cx, dx));
  TEST_INSTRUCTION("660F498C1A80000000"            , cmovns(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F498C1A80000000"            , cmovns(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F49CA"                        , cmovns(ecx, edx));
  TEST_INSTRUCTION("0F498C1A80000000"              , cmovns(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F498C1A80000000"              , cmovns(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F45CA"                      , cmovnz(cx, dx));
  TEST_INSTRUCTION("660F458C1A80000000"            , cmovnz(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F458C1A80000000"            , cmovnz(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F45CA"                        , cmovnz(ecx, edx));
  TEST_INSTRUCTION("0F458C1A80000000"              , cmovnz(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F458C1A80000000"              , cmovnz(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F40CA"                      , cmovo(cx, dx));
  TEST_INSTRUCTION("660F408C1A80000000"            , cmovo(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F408C1A80000000"            , cmovo(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F40CA"                        , cmovo(ecx, edx));
  TEST_INSTRUCTION("0F408C1A80000000"              , cmovo(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F408C1A80000000"              , cmovo(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F4ACA"                      , cmovp(cx, dx));
  TEST_INSTRUCTION("660F4A8C1A80000000"            , cmovp(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F4A8C1A80000000"            , cmovp(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F4ACA"                        , cmovp(ecx, edx));
  TEST_INSTRUCTION("0F4A8C1A80000000"              , cmovp(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F4A8C1A80000000"              , cmovp(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F4ACA"                      , cmovpe(cx, dx));
  TEST_INSTRUCTION("660F4A8C1A80000000"            , cmovpe(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F4A8C1A80000000"            , cmovpe(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F4ACA"                        , cmovpe(ecx, edx));
  TEST_INSTRUCTION("0F4A8C1A80000000"              , cmovpe(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F4A8C1A80000000"              , cmovpe(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F4BCA"                      , cmovpo(cx, dx));
  TEST_INSTRUCTION("660F4B8C1A80000000"            , cmovpo(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F4B8C1A80000000"            , cmovpo(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F4BCA"                        , cmovpo(ecx, edx));
  TEST_INSTRUCTION("0F4B8C1A80000000"              , cmovpo(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F4B8C1A80000000"              , cmovpo(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F48CA"                      , cmovs(cx, dx));
  TEST_INSTRUCTION("660F488C1A80000000"            , cmovs(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F488C1A80000000"            , cmovs(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F48CA"                        , cmovs(ecx, edx));
  TEST_INSTRUCTION("0F488C1A80000000"              , cmovs(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F488C1A80000000"              , cmovs(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F44CA"                      , cmovz(cx, dx));
  TEST_INSTRUCTION("660F448C1A80000000"            , cmovz(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F448C1A80000000"            , cmovz(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F44CA"                        , cmovz(ecx, edx));
  TEST_INSTRUCTION("0F448C1A80000000"              , cmovz(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F448C1A80000000"              , cmovz(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("80F901"                        , cmp(cl, 1));
  TEST_INSTRUCTION("80FD01"                        , cmp(ch, 1));
  TEST_INSTRUCTION("80BC118000000001"              , cmp(byte_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("6683F901"                      , cmp(cx, 1));
  TEST_INSTRUCTION("6683BC118000000001"            , cmp(word_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("83F901"                        , cmp(ecx, 1));
  TEST_INSTRUCTION("83BC118000000001"              , cmp(dword_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("38D1"                          , cmp(cl, dl));
  TEST_INSTRUCTION("38F1"                          , cmp(cl, dh));
  TEST_INSTRUCTION("38D5"                          , cmp(ch, dl));
  TEST_INSTRUCTION("38F5"                          , cmp(ch, dh));
  TEST_INSTRUCTION("389C1180000000"                , cmp(ptr(ecx, edx, 0, 128), bl));
  TEST_INSTRUCTION("38BC1180000000"                , cmp(ptr(ecx, edx, 0, 128), bh));
  TEST_INSTRUCTION("389C1180000000"                , cmp(byte_ptr(ecx, edx, 0, 128), bl));
  TEST_INSTRUCTION("38BC1180000000"                , cmp(byte_ptr(ecx, edx, 0, 128), bh));
  TEST_INSTRUCTION("6639D1"                        , cmp(cx, dx));
  TEST_INSTRUCTION("66399C1180000000"              , cmp(ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("66399C1180000000"              , cmp(word_ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("39D1"                          , cmp(ecx, edx));
  TEST_INSTRUCTION("399C1180000000"                , cmp(ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("399C1180000000"                , cmp(dword_ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("3A8C1A80000000"                , cmp(cl, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("3A8C1A80000000"                , cmp(cl, byte_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("3AAC1A80000000"                , cmp(ch, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("3AAC1A80000000"                , cmp(ch, byte_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("663B8C1A80000000"              , cmp(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("663B8C1A80000000"              , cmp(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("3B8C1A80000000"                , cmp(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("3B8C1A80000000"                , cmp(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("A6"                            , cmps(byte_ptr(esi), byte_ptr(edi)));
  TEST_INSTRUCTION("66A7"                          , cmps(word_ptr(esi), word_ptr(edi)));
  TEST_INSTRUCTION("A7"                            , cmps(dword_ptr(esi), dword_ptr(edi)));
  TEST_INSTRUCTION("0FB0D1"                        , cmpxchg(cl, dl, al));
  TEST_INSTRUCTION("0FB0F1"                        , cmpxchg(cl, dh, al));
  TEST_INSTRUCTION("0FB0D5"                        , cmpxchg(ch, dl, al));
  TEST_INSTRUCTION("0FB0F5"                        , cmpxchg(ch, dh, al));
  TEST_INSTRUCTION("0FB09C1180000000"              , cmpxchg(ptr(ecx, edx, 0, 128), bl, al));
  TEST_INSTRUCTION("0FB0BC1180000000"              , cmpxchg(ptr(ecx, edx, 0, 128), bh, al));
  TEST_INSTRUCTION("0FB09C1180000000"              , cmpxchg(byte_ptr(ecx, edx, 0, 128), bl, al));
  TEST_INSTRUCTION("0FB0BC1180000000"              , cmpxchg(byte_ptr(ecx, edx, 0, 128), bh, al));
  TEST_INSTRUCTION("660FB1D1"                      , cmpxchg(cx, dx, ax));
  TEST_INSTRUCTION("660FB19C1180000000"            , cmpxchg(ptr(ecx, edx, 0, 128), bx, ax));
  TEST_INSTRUCTION("660FB19C1180000000"            , cmpxchg(word_ptr(ecx, edx, 0, 128), bx, ax));
  TEST_INSTRUCTION("0FB1D1"                        , cmpxchg(ecx, edx, eax));
  TEST_INSTRUCTION("0FB19C1180000000"              , cmpxchg(ptr(ecx, edx, 0, 128), ebx, eax));
  TEST_INSTRUCTION("0FB19C1180000000"              , cmpxchg(dword_ptr(ecx, edx, 0, 128), ebx, eax));
  TEST_INSTRUCTION("0FC78C1180000000"              , cmpxchg8b(ptr(ecx, edx, 0, 128), edx, eax, ecx, ebx));
  TEST_INSTRUCTION("0FC78C1180000000"              , cmpxchg8b(qword_ptr(ecx, edx, 0, 128), edx, eax, ecx, ebx));
  TEST_INSTRUCTION("0FA2"                          , cpuid(eax, ebx, ecx, edx));
  TEST_INSTRUCTION("6699"                          , cwd(dx, ax));
  TEST_INSTRUCTION("98"                            , cwde(eax));
  TEST_INSTRUCTION("27"                            , daa(ax));
  TEST_INSTRUCTION("2F"                            , das(ax));
  TEST_INSTRUCTION("6649"                          , dec(cx));
  TEST_INSTRUCTION("49"                            , dec(ecx));
  TEST_INSTRUCTION("FEC9"                          , dec(cl));
  TEST_INSTRUCTION("FECD"                          , dec(ch));
  TEST_INSTRUCTION("FE8C1180000000"                , dec(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("66FF8C1180000000"              , dec(word_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("FF8C1180000000"                , dec(dword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("F6F1"                          , div(ax, cl));
  TEST_INSTRUCTION("F6F5"                          , div(ax, ch));
  TEST_INSTRUCTION("F6B41180000000"                , div(ax, ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("F6B41180000000"                , div(ax, byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("66F7F1"                        , div(dx, ax, cx));
  TEST_INSTRUCTION("66F7B41180000000"              , div(dx, ax, ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("66F7B41180000000"              , div(dx, ax, word_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("F7F1"                          , div(edx, eax, ecx));
  TEST_INSTRUCTION("F7B41180000000"                , div(edx, eax, ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("F7B41180000000"                , div(edx, eax, dword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("C8010002"                      , enter(1, 2));
  TEST_INSTRUCTION("F4"                            , hlt());
  TEST_INSTRUCTION("F6F9"                          , idiv(ax, cl));
  TEST_INSTRUCTION("F6FD"                          , idiv(ax, ch));
  TEST_INSTRUCTION("F6BC1180000000"                , idiv(ax, ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("F6BC1180000000"                , idiv(ax, byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("66F7F9"                        , idiv(dx, ax, cx));
  TEST_INSTRUCTION("66F7BC1180000000"              , idiv(dx, ax, ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("66F7BC1180000000"              , idiv(dx, ax, word_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("F7F9"                          , idiv(edx, eax, ecx));
  TEST_INSTRUCTION("F7BC1180000000"                , idiv(edx, eax, ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("F7BC1180000000"                , idiv(edx, eax, dword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("F6E9"                          , imul(ax, cl));
  TEST_INSTRUCTION("F6ED"                          , imul(ax, ch));
  TEST_INSTRUCTION("660FAF841180000000"            , imul(ax, ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("F6AC1180000000"                , imul(ax, byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("66F7E9"                        , imul(dx, ax, cx));
  TEST_INSTRUCTION("66F7AC1180000000"              , imul(dx, ax, ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("66F7AC1180000000"              , imul(dx, ax, word_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("F7E9"                          , imul(edx, eax, ecx));
  TEST_INSTRUCTION("F7AC1180000000"                , imul(edx, eax, ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("F7AC1180000000"                , imul(edx, eax, dword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("660FAFCA"                      , imul(cx, dx));
  TEST_INSTRUCTION("666BC901"                      , imul(cx, 1));
  TEST_INSTRUCTION("660FAF8C1A80000000"            , imul(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FAF8C1A80000000"            , imul(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FAFCA"                        , imul(ecx, edx));
  TEST_INSTRUCTION("6BC901"                        , imul(ecx, 1));
  TEST_INSTRUCTION("0FAF8C1A80000000"              , imul(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FAF8C1A80000000"              , imul(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("666BC901"                      , imul(cx, cx, 1));
  TEST_INSTRUCTION("666BCA01"                      , imul(cx, dx, 1));
  TEST_INSTRUCTION("666B8C1A8000000001"            , imul(cx, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("666B8C1A8000000001"            , imul(cx, word_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("6BC901"                        , imul(ecx, ecx, 1));
  TEST_INSTRUCTION("6BCA01"                        , imul(ecx, edx, 1));
  TEST_INSTRUCTION("6B8C1A8000000001"              , imul(ecx, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("6B8C1A8000000001"              , imul(ecx, dword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("EC"                            , in(al, dx));
  TEST_INSTRUCTION("E401"                          , in(al, 1));
  TEST_INSTRUCTION("66ED"                          , in(ax, dx));
  TEST_INSTRUCTION("66E501"                        , in(ax, 1));
  TEST_INSTRUCTION("ED"                            , in(eax, dx));
  TEST_INSTRUCTION("E501"                          , in(eax, 1));
  TEST_INSTRUCTION("6641"                          , inc(cx));
  TEST_INSTRUCTION("41"                            , inc(ecx));
  TEST_INSTRUCTION("FEC1"                          , inc(cl));
  TEST_INSTRUCTION("FEC5"                          , inc(ch));
  TEST_INSTRUCTION("FE841180000000"                , inc(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("66FF841180000000"              , inc(word_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("FF841180000000"                , inc(dword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("6C"                            , ins(byte_ptr(edi), dx));
  TEST_INSTRUCTION("666D"                          , ins(word_ptr(edi), dx));
  TEST_INSTRUCTION("6D"                            , ins(dword_ptr(edi), dx));
  TEST_INSTRUCTION("CD01"                          , int_(1));
  TEST_INSTRUCTION("CC"                            , int3());
  TEST_INSTRUCTION("CE"                            , into());
  TEST_INSTRUCTION("0F08"                          , invd());
  TEST_INSTRUCTION("0F01BC1180000000"              , invlpg(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("660F38828C1A80000000"          , invpcid(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38828C1A80000000"          , invpcid(ecx, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("66CF"                          , iret());
  TEST_INSTRUCTION("CF"                            , iretd());
  TEST_INSTRUCTION("9F"                            , lahf(ah));
  TEST_INSTRUCTION("660F02CA"                      , lar(cx, dx));
  TEST_INSTRUCTION("660F028C1A80000000"            , lar(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F028C1A80000000"            , lar(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F02CA"                        , lar(ecx, edx));
  TEST_INSTRUCTION("0F028C1A80000000"              , lar(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F028C1A80000000"              , lar(ecx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("9A020000000100"                , lcall(1, 2));
  TEST_INSTRUCTION("FF9C1180000000"                , lcall(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("66FF9C1180000000"              , lcall(dword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("FF9C1180000000"                , lcall(fword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("66C58C1A80000000"              , lds(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("66C58C1A80000000"              , lds(cx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C58C1A80000000"                , lds(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C58C1A80000000"                , lds(ecx, fword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("668D8C1A80000000"              , lea(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8D8C1A80000000"                , lea(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C9"                            , leave());
  TEST_INSTRUCTION("66C48C1A80000000"              , les(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("66C48C1A80000000"              , les(cx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C48C1A80000000"                , les(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C48C1A80000000"                , les(ecx, fword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FB48C1A80000000"            , lfs(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FB48C1A80000000"            , lfs(cx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FB48C1A80000000"              , lfs(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FB48C1A80000000"              , lfs(ecx, fword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F01941180000000"              , lgdt(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("660FB58C1A80000000"            , lgs(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FB58C1A80000000"            , lgs(cx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FB58C1A80000000"              , lgs(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FB58C1A80000000"              , lgs(ecx, fword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F019C1180000000"              , lidt(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("EA020000000100"                , ljmp(1, 2));
  TEST_INSTRUCTION("FFAC1180000000"                , ljmp(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("66FFAC1180000000"              , ljmp(dword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("FFAC1180000000"                , ljmp(fword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F00D1"                        , lldt(cx));
  TEST_INSTRUCTION("0F00941180000000"              , lldt(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F00941180000000"              , lldt(word_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F01F1"                        , lmsw(cx));
  TEST_INSTRUCTION("0F01B41180000000"              , lmsw(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F01B41180000000"              , lmsw(word_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("AC"                            , lods(al, ptr(esi)));
  TEST_INSTRUCTION("AC"                            , lods(al, byte_ptr(esi)));
  TEST_INSTRUCTION("66AD"                          , lods(ax, ptr(esi)));
  TEST_INSTRUCTION("66AD"                          , lods(ax, word_ptr(esi)));
  TEST_INSTRUCTION("AD"                            , lods(eax, ptr(esi)));
  TEST_INSTRUCTION("AD"                            , lods(eax, dword_ptr(esi)));
  TEST_INSTRUCTION("660F03CA"                      , lsl(cx, dx));
  TEST_INSTRUCTION("660F038C1A80000000"            , lsl(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F038C1A80000000"            , lsl(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F03CA"                        , lsl(ecx, edx));
  TEST_INSTRUCTION("0F038C1A80000000"              , lsl(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F038C1A80000000"              , lsl(ecx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FB28C1A80000000"            , lss(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FB28C1A80000000"            , lss(cx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FB28C1A80000000"              , lss(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FB28C1A80000000"              , lss(ecx, fword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F00D9"                        , ltr(cx));
  TEST_INSTRUCTION("0F009C1180000000"              , ltr(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F009C1180000000"              , ltr(word_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("88D1"                          , mov(cl, dl));
  TEST_INSTRUCTION("88F1"                          , mov(cl, dh));
  TEST_INSTRUCTION("88D5"                          , mov(ch, dl));
  TEST_INSTRUCTION("88F5"                          , mov(ch, dh));
  TEST_INSTRUCTION("889C1180000000"                , mov(ptr(ecx, edx, 0, 128), bl));
  TEST_INSTRUCTION("88BC1180000000"                , mov(ptr(ecx, edx, 0, 128), bh));
  TEST_INSTRUCTION("889C1180000000"                , mov(byte_ptr(ecx, edx, 0, 128), bl));
  TEST_INSTRUCTION("88BC1180000000"                , mov(byte_ptr(ecx, edx, 0, 128), bh));
  TEST_INSTRUCTION("6689D1"                        , mov(cx, dx));
  TEST_INSTRUCTION("668EE2"                        , mov(fs, dx));
  TEST_INSTRUCTION("66899C1180000000"              , mov(ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("66899C1180000000"              , mov(word_ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("89D1"                          , mov(ecx, edx));
  TEST_INSTRUCTION("8EE2"                          , mov(fs, edx));
  TEST_INSTRUCTION("899C1180000000"                , mov(ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("899C1180000000"                , mov(dword_ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("B101"                          , mov(cl, 1));
  TEST_INSTRUCTION("B501"                          , mov(ch, 1));
  TEST_INSTRUCTION("C684118000000001"              , mov(byte_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("66B90100"                      , mov(cx, 1));
  TEST_INSTRUCTION("66C78411800000000100"          , mov(word_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("B901000000"                    , mov(ecx, 1));
  TEST_INSTRUCTION("C784118000000001000000"        , mov(dword_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("8A8C1A80000000"                , mov(cl, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8A8C1A80000000"                , mov(cl, byte_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8AAC1A80000000"                , mov(ch, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8AAC1A80000000"                , mov(ch, byte_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("668CE1"                        , mov(cx, fs));
  TEST_INSTRUCTION("668B8C1A80000000"              , mov(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("668B8C1A80000000"              , mov(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8CE1"                          , mov(ecx, fs));
  TEST_INSTRUCTION("8B8C1A80000000"                , mov(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8B8C1A80000000"                , mov(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8CA41180000000"                , mov(ptr(ecx, edx, 0, 128), fs));
  TEST_INSTRUCTION("668CA41180000000"              , mov(word_ptr(ecx, edx, 0, 128), fs));
  TEST_INSTRUCTION("8EA41A80000000"                , mov(fs, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("668EA41A80000000"              , mov(fs, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F20D1"                        , mov(ecx, cr2));
  TEST_INSTRUCTION("0F21D1"                        , mov(ecx, dr2));
  TEST_INSTRUCTION("0F22CA"                        , mov(cr1, edx));
  TEST_INSTRUCTION("0F23CA"                        , mov(dr1, edx));
  TEST_INSTRUCTION("A4"                            , movs(byte_ptr(edi), byte_ptr(esi)));
  TEST_INSTRUCTION("66A5"                          , movs(word_ptr(edi), word_ptr(esi)));
  TEST_INSTRUCTION("A5"                            , movs(dword_ptr(edi), dword_ptr(esi)));
  TEST_INSTRUCTION("660FBECA"                      , movsx(cx, dl));
  TEST_INSTRUCTION("660FBECE"                      , movsx(cx, dh));
  TEST_INSTRUCTION("660FBE8C1A80000000"            , movsx(cx, byte_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FBECA"                        , movsx(ecx, dl));
  TEST_INSTRUCTION("0FBECE"                        , movsx(ecx, dh));
  TEST_INSTRUCTION("0FBFCA"                        , movsx(ecx, dx));
  TEST_INSTRUCTION("0FBE8C1A80000000"              , movsx(ecx, byte_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FBF8C1A80000000"              , movsx(ecx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FB6CA"                      , movzx(cx, dl));
  TEST_INSTRUCTION("660FB6CE"                      , movzx(cx, dh));
  TEST_INSTRUCTION("660FB68C1A80000000"            , movzx(cx, byte_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FB6CA"                        , movzx(ecx, dl));
  TEST_INSTRUCTION("0FB6CE"                        , movzx(ecx, dh));
  TEST_INSTRUCTION("0FB7CA"                        , movzx(ecx, dx));
  TEST_INSTRUCTION("0FB68C1A80000000"              , movzx(ecx, byte_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FB78C1A80000000"              , movzx(ecx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F6E1"                          , mul(ax, cl));
  TEST_INSTRUCTION("F6E5"                          , mul(ax, ch));
  TEST_INSTRUCTION("F6A41180000000"                , mul(ax, ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("F6A41180000000"                , mul(ax, byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("66F7E1"                        , mul(dx, ax, cx));
  TEST_INSTRUCTION("66F7A41180000000"              , mul(dx, ax, ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("66F7A41180000000"              , mul(dx, ax, word_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("F7E1"                          , mul(edx, eax, ecx));
  TEST_INSTRUCTION("F7A41180000000"                , mul(edx, eax, ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("F7A41180000000"                , mul(edx, eax, dword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("F6D9"                          , neg(cl));
  TEST_INSTRUCTION("F6DD"                          , neg(ch));
  TEST_INSTRUCTION("66F7D9"                        , neg(cx));
  TEST_INSTRUCTION("F7D9"                          , neg(ecx));
  TEST_INSTRUCTION("F69C1180000000"                , neg(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("66F79C1180000000"              , neg(word_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("F79C1180000000"                , neg(dword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("90"                            , nop());
  TEST_INSTRUCTION("660F1FC1"                      , nop(cx));
  TEST_INSTRUCTION("0F1FC1"                        , nop(ecx));
  TEST_INSTRUCTION("660F1F841180000000"            , nop(word_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F1F841180000000"              , nop(dword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("660F1FD1"                      , nop(cx, dx));
  TEST_INSTRUCTION("660F1F9C1180000000"            , nop(ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("660F1F9C1180000000"            , nop(word_ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("0F1FD1"                        , nop(ecx, edx));
  TEST_INSTRUCTION("0F1F9C1180000000"              , nop(ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("0F1F9C1180000000"              , nop(dword_ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("F6D1"                          , not_(cl));
  TEST_INSTRUCTION("F6D5"                          , not_(ch));
  TEST_INSTRUCTION("66F7D1"                        , not_(cx));
  TEST_INSTRUCTION("F7D1"                          , not_(ecx));
  TEST_INSTRUCTION("F6941180000000"                , not_(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("66F7941180000000"              , not_(word_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("F7941180000000"                , not_(dword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("80C901"                        , or_(cl, 1));
  TEST_INSTRUCTION("80CD01"                        , or_(ch, 1));
  TEST_INSTRUCTION("808C118000000001"              , or_(byte_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("6683C901"                      , or_(cx, 1));
  TEST_INSTRUCTION("66838C118000000001"            , or_(word_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("83C901"                        , or_(ecx, 1));
  TEST_INSTRUCTION("838C118000000001"              , or_(dword_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("08D1"                          , or_(cl, dl));
  TEST_INSTRUCTION("08F1"                          , or_(cl, dh));
  TEST_INSTRUCTION("08D5"                          , or_(ch, dl));
  TEST_INSTRUCTION("08F5"                          , or_(ch, dh));
  TEST_INSTRUCTION("089C1180000000"                , or_(ptr(ecx, edx, 0, 128), bl));
  TEST_INSTRUCTION("08BC1180000000"                , or_(ptr(ecx, edx, 0, 128), bh));
  TEST_INSTRUCTION("089C1180000000"                , or_(byte_ptr(ecx, edx, 0, 128), bl));
  TEST_INSTRUCTION("08BC1180000000"                , or_(byte_ptr(ecx, edx, 0, 128), bh));
  TEST_INSTRUCTION("6609D1"                        , or_(cx, dx));
  TEST_INSTRUCTION("66099C1180000000"              , or_(ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("66099C1180000000"              , or_(word_ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("09D1"                          , or_(ecx, edx));
  TEST_INSTRUCTION("099C1180000000"                , or_(ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("099C1180000000"                , or_(dword_ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("0A8C1A80000000"                , or_(cl, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0A8C1A80000000"                , or_(cl, byte_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0AAC1A80000000"                , or_(ch, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0AAC1A80000000"                , or_(ch, byte_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660B8C1A80000000"              , or_(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660B8C1A80000000"              , or_(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0B8C1A80000000"                , or_(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0B8C1A80000000"                , or_(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("EE"                            , out(dx, al));
  TEST_INSTRUCTION("66EF"                          , out(dx, ax));
  TEST_INSTRUCTION("EF"                            , out(dx, eax));
  TEST_INSTRUCTION("E601"                          , out(1, al));
  TEST_INSTRUCTION("66E701"                        , out(1, ax));
  TEST_INSTRUCTION("E701"                          , out(1, eax));
  TEST_INSTRUCTION("6E"                            , outs(dx, byte_ptr(esi)));
  TEST_INSTRUCTION("666F"                          , outs(dx, word_ptr(esi)));
  TEST_INSTRUCTION("6F"                            , outs(dx, dword_ptr(esi)));
  TEST_INSTRUCTION("F390"                          , pause());
  TEST_INSTRUCTION("6659"                          , pop(cx));
  TEST_INSTRUCTION("668F841180000000"              , pop(word_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("59"                            , pop(ecx));
  TEST_INSTRUCTION("8F841180000000"                , pop(dword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0FA1"                          , pop(fs));
  TEST_INSTRUCTION("6661"                          , popa());
  TEST_INSTRUCTION("61"                            , popad());
  TEST_INSTRUCTION("669D"                          , popf());
  TEST_INSTRUCTION("9D"                            , popfd());
  TEST_INSTRUCTION("6651"                          , push(cx));
  TEST_INSTRUCTION("6A01"                          , push(1));
  TEST_INSTRUCTION("66FFB41180000000"              , push(word_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("51"                            , push(ecx));
  TEST_INSTRUCTION("FFB41180000000"                , push(dword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0FA0"                          , push(fs));
  TEST_INSTRUCTION("6660"                          , pusha());
  TEST_INSTRUCTION("60"                            , pushad());
  TEST_INSTRUCTION("669C"                          , pushf());
  TEST_INSTRUCTION("9C"                            , pushfd());
  TEST_INSTRUCTION("D2D1"                          , rcl(cl, cl));
  TEST_INSTRUCTION("D0D1"                          , rcl(cl, 1));
  TEST_INSTRUCTION("D2D5"                          , rcl(ch, cl));
  TEST_INSTRUCTION("D0D5"                          , rcl(ch, 1));
  TEST_INSTRUCTION("66D3D1"                        , rcl(cx, cl));
  TEST_INSTRUCTION("66D1D1"                        , rcl(cx, 1));
  TEST_INSTRUCTION("D3D1"                          , rcl(ecx, cl));
  TEST_INSTRUCTION("D1D1"                          , rcl(ecx, 1));
  TEST_INSTRUCTION("D2941180000000"                , rcl(byte_ptr(ecx, edx, 0, 128), cl));
  TEST_INSTRUCTION("D0941180000000"                , rcl(byte_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("66D3941180000000"              , rcl(word_ptr(ecx, edx, 0, 128), cl));
  TEST_INSTRUCTION("66D1941180000000"              , rcl(word_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("D3941180000000"                , rcl(dword_ptr(ecx, edx, 0, 128), cl));
  TEST_INSTRUCTION("D1941180000000"                , rcl(dword_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("D2D9"                          , rcr(cl, cl));
  TEST_INSTRUCTION("D0D9"                          , rcr(cl, 1));
  TEST_INSTRUCTION("D2DD"                          , rcr(ch, cl));
  TEST_INSTRUCTION("D0DD"                          , rcr(ch, 1));
  TEST_INSTRUCTION("66D3D9"                        , rcr(cx, cl));
  TEST_INSTRUCTION("66D1D9"                        , rcr(cx, 1));
  TEST_INSTRUCTION("D3D9"                          , rcr(ecx, cl));
  TEST_INSTRUCTION("D1D9"                          , rcr(ecx, 1));
  TEST_INSTRUCTION("D29C1180000000"                , rcr(byte_ptr(ecx, edx, 0, 128), cl));
  TEST_INSTRUCTION("D09C1180000000"                , rcr(byte_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("66D39C1180000000"              , rcr(word_ptr(ecx, edx, 0, 128), cl));
  TEST_INSTRUCTION("66D19C1180000000"              , rcr(word_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("D39C1180000000"                , rcr(dword_ptr(ecx, edx, 0, 128), cl));
  TEST_INSTRUCTION("D19C1180000000"                , rcr(dword_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("0F32"                          , rdmsr(edx, eax, ecx));
  TEST_INSTRUCTION("C3"                            , ret());
  TEST_INSTRUCTION("C20100"                        , ret(1));
  TEST_INSTRUCTION("CB"                            , retf());
  TEST_INSTRUCTION("CA0100"                        , retf(1));
  TEST_INSTRUCTION("D2C1"                          , rol(cl, cl));
  TEST_INSTRUCTION("D0C1"                          , rol(cl, 1));
  TEST_INSTRUCTION("D2C5"                          , rol(ch, cl));
  TEST_INSTRUCTION("D0C5"                          , rol(ch, 1));
  TEST_INSTRUCTION("66D3C1"                        , rol(cx, cl));
  TEST_INSTRUCTION("66D1C1"                        , rol(cx, 1));
  TEST_INSTRUCTION("D3C1"                          , rol(ecx, cl));
  TEST_INSTRUCTION("D1C1"                          , rol(ecx, 1));
  TEST_INSTRUCTION("D2841180000000"                , rol(byte_ptr(ecx, edx, 0, 128), cl));
  TEST_INSTRUCTION("D0841180000000"                , rol(byte_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("66D3841180000000"              , rol(word_ptr(ecx, edx, 0, 128), cl));
  TEST_INSTRUCTION("66D1841180000000"              , rol(word_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("D3841180000000"                , rol(dword_ptr(ecx, edx, 0, 128), cl));
  TEST_INSTRUCTION("D1841180000000"                , rol(dword_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("D2C9"                          , ror(cl, cl));
  TEST_INSTRUCTION("D0C9"                          , ror(cl, 1));
  TEST_INSTRUCTION("D2CD"                          , ror(ch, cl));
  TEST_INSTRUCTION("D0CD"                          , ror(ch, 1));
  TEST_INSTRUCTION("66D3C9"                        , ror(cx, cl));
  TEST_INSTRUCTION("66D1C9"                        , ror(cx, 1));
  TEST_INSTRUCTION("D3C9"                          , ror(ecx, cl));
  TEST_INSTRUCTION("D1C9"                          , ror(ecx, 1));
  TEST_INSTRUCTION("D28C1180000000"                , ror(byte_ptr(ecx, edx, 0, 128), cl));
  TEST_INSTRUCTION("D08C1180000000"                , ror(byte_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("66D38C1180000000"              , ror(word_ptr(ecx, edx, 0, 128), cl));
  TEST_INSTRUCTION("66D18C1180000000"              , ror(word_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("D38C1180000000"                , ror(dword_ptr(ecx, edx, 0, 128), cl));
  TEST_INSTRUCTION("D18C1180000000"                , ror(dword_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("9E"                            , sahf(ah));
  TEST_INSTRUCTION("D2E1"                          , sal(cl, cl));
  TEST_INSTRUCTION("D0E1"                          , sal(cl, 1));
  TEST_INSTRUCTION("D2E5"                          , sal(ch, cl));
  TEST_INSTRUCTION("D0E5"                          , sal(ch, 1));
  TEST_INSTRUCTION("66D3E1"                        , sal(cx, cl));
  TEST_INSTRUCTION("66D1E1"                        , sal(cx, 1));
  TEST_INSTRUCTION("D3E1"                          , sal(ecx, cl));
  TEST_INSTRUCTION("D1E1"                          , sal(ecx, 1));
  TEST_INSTRUCTION("D2A41180000000"                , sal(byte_ptr(ecx, edx, 0, 128), cl));
  TEST_INSTRUCTION("D0A41180000000"                , sal(byte_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("66D3A41180000000"              , sal(word_ptr(ecx, edx, 0, 128), cl));
  TEST_INSTRUCTION("66D1A41180000000"              , sal(word_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("D3A41180000000"                , sal(dword_ptr(ecx, edx, 0, 128), cl));
  TEST_INSTRUCTION("D1A41180000000"                , sal(dword_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("D2F9"                          , sar(cl, cl));
  TEST_INSTRUCTION("D0F9"                          , sar(cl, 1));
  TEST_INSTRUCTION("D2FD"                          , sar(ch, cl));
  TEST_INSTRUCTION("D0FD"                          , sar(ch, 1));
  TEST_INSTRUCTION("66D3F9"                        , sar(cx, cl));
  TEST_INSTRUCTION("66D1F9"                        , sar(cx, 1));
  TEST_INSTRUCTION("D3F9"                          , sar(ecx, cl));
  TEST_INSTRUCTION("D1F9"                          , sar(ecx, 1));
  TEST_INSTRUCTION("D2BC1180000000"                , sar(byte_ptr(ecx, edx, 0, 128), cl));
  TEST_INSTRUCTION("D0BC1180000000"                , sar(byte_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("66D3BC1180000000"              , sar(word_ptr(ecx, edx, 0, 128), cl));
  TEST_INSTRUCTION("66D1BC1180000000"              , sar(word_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("D3BC1180000000"                , sar(dword_ptr(ecx, edx, 0, 128), cl));
  TEST_INSTRUCTION("D1BC1180000000"                , sar(dword_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("80D901"                        , sbb(cl, 1));
  TEST_INSTRUCTION("80DD01"                        , sbb(ch, 1));
  TEST_INSTRUCTION("809C118000000001"              , sbb(byte_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("6683D901"                      , sbb(cx, 1));
  TEST_INSTRUCTION("66839C118000000001"            , sbb(word_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("83D901"                        , sbb(ecx, 1));
  TEST_INSTRUCTION("839C118000000001"              , sbb(dword_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("18D1"                          , sbb(cl, dl));
  TEST_INSTRUCTION("18F1"                          , sbb(cl, dh));
  TEST_INSTRUCTION("18D5"                          , sbb(ch, dl));
  TEST_INSTRUCTION("18F5"                          , sbb(ch, dh));
  TEST_INSTRUCTION("189C1180000000"                , sbb(ptr(ecx, edx, 0, 128), bl));
  TEST_INSTRUCTION("18BC1180000000"                , sbb(ptr(ecx, edx, 0, 128), bh));
  TEST_INSTRUCTION("189C1180000000"                , sbb(byte_ptr(ecx, edx, 0, 128), bl));
  TEST_INSTRUCTION("18BC1180000000"                , sbb(byte_ptr(ecx, edx, 0, 128), bh));
  TEST_INSTRUCTION("6619D1"                        , sbb(cx, dx));
  TEST_INSTRUCTION("66199C1180000000"              , sbb(ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("66199C1180000000"              , sbb(word_ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("19D1"                          , sbb(ecx, edx));
  TEST_INSTRUCTION("199C1180000000"                , sbb(ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("199C1180000000"                , sbb(dword_ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("1A8C1A80000000"                , sbb(cl, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("1A8C1A80000000"                , sbb(cl, byte_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("1AAC1A80000000"                , sbb(ch, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("1AAC1A80000000"                , sbb(ch, byte_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("661B8C1A80000000"              , sbb(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("661B8C1A80000000"              , sbb(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("1B8C1A80000000"                , sbb(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("1B8C1A80000000"                , sbb(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("AE"                            , scas(al, ptr(edi)));
  TEST_INSTRUCTION("AE"                            , scas(al, byte_ptr(edi)));
  TEST_INSTRUCTION("66AF"                          , scas(ax, ptr(edi)));
  TEST_INSTRUCTION("66AF"                          , scas(ax, word_ptr(edi)));
  TEST_INSTRUCTION("AF"                            , scas(eax, ptr(edi)));
  TEST_INSTRUCTION("AF"                            , scas(eax, dword_ptr(edi)));
  TEST_INSTRUCTION("0F97C1"                        , seta(cl));
  TEST_INSTRUCTION("0F97C5"                        , seta(ch));
  TEST_INSTRUCTION("0F97841180000000"              , seta(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F97841180000000"              , seta(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F93C1"                        , setae(cl));
  TEST_INSTRUCTION("0F93C5"                        , setae(ch));
  TEST_INSTRUCTION("0F93841180000000"              , setae(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F93841180000000"              , setae(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F92C1"                        , setb(cl));
  TEST_INSTRUCTION("0F92C5"                        , setb(ch));
  TEST_INSTRUCTION("0F92841180000000"              , setb(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F92841180000000"              , setb(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F96C1"                        , setbe(cl));
  TEST_INSTRUCTION("0F96C5"                        , setbe(ch));
  TEST_INSTRUCTION("0F96841180000000"              , setbe(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F96841180000000"              , setbe(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F92C1"                        , setc(cl));
  TEST_INSTRUCTION("0F92C5"                        , setc(ch));
  TEST_INSTRUCTION("0F92841180000000"              , setc(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F92841180000000"              , setc(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F94C1"                        , sete(cl));
  TEST_INSTRUCTION("0F94C5"                        , sete(ch));
  TEST_INSTRUCTION("0F94841180000000"              , sete(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F94841180000000"              , sete(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F9FC1"                        , setg(cl));
  TEST_INSTRUCTION("0F9FC5"                        , setg(ch));
  TEST_INSTRUCTION("0F9F841180000000"              , setg(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F9F841180000000"              , setg(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F9DC1"                        , setge(cl));
  TEST_INSTRUCTION("0F9DC5"                        , setge(ch));
  TEST_INSTRUCTION("0F9D841180000000"              , setge(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F9D841180000000"              , setge(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F9CC1"                        , setl(cl));
  TEST_INSTRUCTION("0F9CC5"                        , setl(ch));
  TEST_INSTRUCTION("0F9C841180000000"              , setl(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F9C841180000000"              , setl(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F9EC1"                        , setle(cl));
  TEST_INSTRUCTION("0F9EC5"                        , setle(ch));
  TEST_INSTRUCTION("0F9E841180000000"              , setle(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F9E841180000000"              , setle(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F96C1"                        , setna(cl));
  TEST_INSTRUCTION("0F96C5"                        , setna(ch));
  TEST_INSTRUCTION("0F96841180000000"              , setna(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F96841180000000"              , setna(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F92C1"                        , setnae(cl));
  TEST_INSTRUCTION("0F92C5"                        , setnae(ch));
  TEST_INSTRUCTION("0F92841180000000"              , setnae(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F92841180000000"              , setnae(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F93C1"                        , setnb(cl));
  TEST_INSTRUCTION("0F93C5"                        , setnb(ch));
  TEST_INSTRUCTION("0F93841180000000"              , setnb(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F93841180000000"              , setnb(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F97C1"                        , setnbe(cl));
  TEST_INSTRUCTION("0F97C5"                        , setnbe(ch));
  TEST_INSTRUCTION("0F97841180000000"              , setnbe(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F97841180000000"              , setnbe(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F93C1"                        , setnc(cl));
  TEST_INSTRUCTION("0F93C5"                        , setnc(ch));
  TEST_INSTRUCTION("0F93841180000000"              , setnc(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F93841180000000"              , setnc(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F95C1"                        , setne(cl));
  TEST_INSTRUCTION("0F95C5"                        , setne(ch));
  TEST_INSTRUCTION("0F95841180000000"              , setne(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F95841180000000"              , setne(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F9EC1"                        , setng(cl));
  TEST_INSTRUCTION("0F9EC5"                        , setng(ch));
  TEST_INSTRUCTION("0F9E841180000000"              , setng(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F9E841180000000"              , setng(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F9CC1"                        , setnge(cl));
  TEST_INSTRUCTION("0F9CC5"                        , setnge(ch));
  TEST_INSTRUCTION("0F9C841180000000"              , setnge(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F9C841180000000"              , setnge(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F9DC1"                        , setnl(cl));
  TEST_INSTRUCTION("0F9DC5"                        , setnl(ch));
  TEST_INSTRUCTION("0F9D841180000000"              , setnl(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F9D841180000000"              , setnl(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F9FC1"                        , setnle(cl));
  TEST_INSTRUCTION("0F9FC5"                        , setnle(ch));
  TEST_INSTRUCTION("0F9F841180000000"              , setnle(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F9F841180000000"              , setnle(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F91C1"                        , setno(cl));
  TEST_INSTRUCTION("0F91C5"                        , setno(ch));
  TEST_INSTRUCTION("0F91841180000000"              , setno(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F91841180000000"              , setno(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F9BC1"                        , setnp(cl));
  TEST_INSTRUCTION("0F9BC5"                        , setnp(ch));
  TEST_INSTRUCTION("0F9B841180000000"              , setnp(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F9B841180000000"              , setnp(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F99C1"                        , setns(cl));
  TEST_INSTRUCTION("0F99C5"                        , setns(ch));
  TEST_INSTRUCTION("0F99841180000000"              , setns(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F99841180000000"              , setns(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F95C1"                        , setnz(cl));
  TEST_INSTRUCTION("0F95C5"                        , setnz(ch));
  TEST_INSTRUCTION("0F95841180000000"              , setnz(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F95841180000000"              , setnz(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F90C1"                        , seto(cl));
  TEST_INSTRUCTION("0F90C5"                        , seto(ch));
  TEST_INSTRUCTION("0F90841180000000"              , seto(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F90841180000000"              , seto(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F9AC1"                        , setp(cl));
  TEST_INSTRUCTION("0F9AC5"                        , setp(ch));
  TEST_INSTRUCTION("0F9A841180000000"              , setp(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F9A841180000000"              , setp(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F9AC1"                        , setpe(cl));
  TEST_INSTRUCTION("0F9AC5"                        , setpe(ch));
  TEST_INSTRUCTION("0F9A841180000000"              , setpe(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F9A841180000000"              , setpe(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F9BC1"                        , setpo(cl));
  TEST_INSTRUCTION("0F9BC5"                        , setpo(ch));
  TEST_INSTRUCTION("0F9B841180000000"              , setpo(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F9B841180000000"              , setpo(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F98C1"                        , sets(cl));
  TEST_INSTRUCTION("0F98C5"                        , sets(ch));
  TEST_INSTRUCTION("0F98841180000000"              , sets(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F98841180000000"              , sets(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F94C1"                        , setz(cl));
  TEST_INSTRUCTION("0F94C5"                        , setz(ch));
  TEST_INSTRUCTION("0F94841180000000"              , setz(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F94841180000000"              , setz(byte_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F01841180000000"              , sgdt(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("D2E1"                          , shl(cl, cl));
  TEST_INSTRUCTION("D0E1"                          , shl(cl, 1));
  TEST_INSTRUCTION("D2E5"                          , shl(ch, cl));
  TEST_INSTRUCTION("D0E5"                          , shl(ch, 1));
  TEST_INSTRUCTION("66D3E1"                        , shl(cx, cl));
  TEST_INSTRUCTION("66D1E1"                        , shl(cx, 1));
  TEST_INSTRUCTION("D3E1"                          , shl(ecx, cl));
  TEST_INSTRUCTION("D1E1"                          , shl(ecx, 1));
  TEST_INSTRUCTION("D2A41180000000"                , shl(byte_ptr(ecx, edx, 0, 128), cl));
  TEST_INSTRUCTION("D0A41180000000"                , shl(byte_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("66D3A41180000000"              , shl(word_ptr(ecx, edx, 0, 128), cl));
  TEST_INSTRUCTION("66D1A41180000000"              , shl(word_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("D3A41180000000"                , shl(dword_ptr(ecx, edx, 0, 128), cl));
  TEST_INSTRUCTION("D1A41180000000"                , shl(dword_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("660FA5D1"                      , shld(cx, dx, cl));
  TEST_INSTRUCTION("660FA4D101"                    , shld(cx, dx, 1));
  TEST_INSTRUCTION("660FA59C1180000000"            , shld(ptr(ecx, edx, 0, 128), bx, cl));
  TEST_INSTRUCTION("660FA49C118000000001"          , shld(ptr(ecx, edx, 0, 128), bx, 1));
  TEST_INSTRUCTION("660FA59C1180000000"            , shld(word_ptr(ecx, edx, 0, 128), bx, cl));
  TEST_INSTRUCTION("660FA49C118000000001"          , shld(word_ptr(ecx, edx, 0, 128), bx, 1));
  TEST_INSTRUCTION("0FA5D1"                        , shld(ecx, edx, cl));
  TEST_INSTRUCTION("0FA4D101"                      , shld(ecx, edx, 1));
  TEST_INSTRUCTION("0FA59C1180000000"              , shld(ptr(ecx, edx, 0, 128), ebx, cl));
  TEST_INSTRUCTION("0FA49C118000000001"            , shld(ptr(ecx, edx, 0, 128), ebx, 1));
  TEST_INSTRUCTION("0FA59C1180000000"              , shld(dword_ptr(ecx, edx, 0, 128), ebx, cl));
  TEST_INSTRUCTION("0FA49C118000000001"            , shld(dword_ptr(ecx, edx, 0, 128), ebx, 1));
  TEST_INSTRUCTION("D2E9"                          , shr(cl, cl));
  TEST_INSTRUCTION("D0E9"                          , shr(cl, 1));
  TEST_INSTRUCTION("D2ED"                          , shr(ch, cl));
  TEST_INSTRUCTION("D0ED"                          , shr(ch, 1));
  TEST_INSTRUCTION("66D3E9"                        , shr(cx, cl));
  TEST_INSTRUCTION("66D1E9"                        , shr(cx, 1));
  TEST_INSTRUCTION("D3E9"                          , shr(ecx, cl));
  TEST_INSTRUCTION("D1E9"                          , shr(ecx, 1));
  TEST_INSTRUCTION("D2AC1180000000"                , shr(byte_ptr(ecx, edx, 0, 128), cl));
  TEST_INSTRUCTION("D0AC1180000000"                , shr(byte_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("66D3AC1180000000"              , shr(word_ptr(ecx, edx, 0, 128), cl));
  TEST_INSTRUCTION("66D1AC1180000000"              , shr(word_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("D3AC1180000000"                , shr(dword_ptr(ecx, edx, 0, 128), cl));
  TEST_INSTRUCTION("D1AC1180000000"                , shr(dword_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("660FADD1"                      , shrd(cx, dx, cl));
  TEST_INSTRUCTION("660FACD101"                    , shrd(cx, dx, 1));
  TEST_INSTRUCTION("660FAD9C1180000000"            , shrd(ptr(ecx, edx, 0, 128), bx, cl));
  TEST_INSTRUCTION("660FAC9C118000000001"          , shrd(ptr(ecx, edx, 0, 128), bx, 1));
  TEST_INSTRUCTION("660FAD9C1180000000"            , shrd(word_ptr(ecx, edx, 0, 128), bx, cl));
  TEST_INSTRUCTION("660FAC9C118000000001"          , shrd(word_ptr(ecx, edx, 0, 128), bx, 1));
  TEST_INSTRUCTION("0FADD1"                        , shrd(ecx, edx, cl));
  TEST_INSTRUCTION("0FACD101"                      , shrd(ecx, edx, 1));
  TEST_INSTRUCTION("0FAD9C1180000000"              , shrd(ptr(ecx, edx, 0, 128), ebx, cl));
  TEST_INSTRUCTION("0FAC9C118000000001"            , shrd(ptr(ecx, edx, 0, 128), ebx, 1));
  TEST_INSTRUCTION("0FAD9C1180000000"              , shrd(dword_ptr(ecx, edx, 0, 128), ebx, cl));
  TEST_INSTRUCTION("0FAC9C118000000001"            , shrd(dword_ptr(ecx, edx, 0, 128), ebx, 1));
  TEST_INSTRUCTION("0F018C1180000000"              , sidt(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("660F00C1"                      , sldt(cx));
  TEST_INSTRUCTION("0F00C1"                        , sldt(ecx));
  TEST_INSTRUCTION("0F00841180000000"              , sldt(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F00841180000000"              , sldt(word_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("660F01E1"                      , smsw(cx));
  TEST_INSTRUCTION("0F01E1"                        , smsw(ecx));
  TEST_INSTRUCTION("0F01A41180000000"              , smsw(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F01A41180000000"              , smsw(word_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("F9"                            , stc());
  TEST_INSTRUCTION("FD"                            , std());
  TEST_INSTRUCTION("FB"                            , sti());
  TEST_INSTRUCTION("AA"                            , stos(ptr(edi), al));
  TEST_INSTRUCTION("AA"                            , stos(byte_ptr(edi), al));
  TEST_INSTRUCTION("66AB"                          , stos(ptr(edi), ax));
  TEST_INSTRUCTION("66AB"                          , stos(word_ptr(edi), ax));
  TEST_INSTRUCTION("AB"                            , stos(ptr(edi), eax));
  TEST_INSTRUCTION("AB"                            , stos(dword_ptr(edi), eax));
  TEST_INSTRUCTION("660F00C9"                      , str(cx));
  TEST_INSTRUCTION("0F00C9"                        , str(ecx));
  TEST_INSTRUCTION("0F008C1180000000"              , str(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F008C1180000000"              , str(word_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("80E901"                        , sub(cl, 1));
  TEST_INSTRUCTION("80ED01"                        , sub(ch, 1));
  TEST_INSTRUCTION("80AC118000000001"              , sub(byte_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("6683E901"                      , sub(cx, 1));
  TEST_INSTRUCTION("6683AC118000000001"            , sub(word_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("83E901"                        , sub(ecx, 1));
  TEST_INSTRUCTION("83AC118000000001"              , sub(dword_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("28D1"                          , sub(cl, dl));
  TEST_INSTRUCTION("28F1"                          , sub(cl, dh));
  TEST_INSTRUCTION("28D5"                          , sub(ch, dl));
  TEST_INSTRUCTION("28F5"                          , sub(ch, dh));
  TEST_INSTRUCTION("289C1180000000"                , sub(ptr(ecx, edx, 0, 128), bl));
  TEST_INSTRUCTION("28BC1180000000"                , sub(ptr(ecx, edx, 0, 128), bh));
  TEST_INSTRUCTION("289C1180000000"                , sub(byte_ptr(ecx, edx, 0, 128), bl));
  TEST_INSTRUCTION("28BC1180000000"                , sub(byte_ptr(ecx, edx, 0, 128), bh));
  TEST_INSTRUCTION("6629D1"                        , sub(cx, dx));
  TEST_INSTRUCTION("66299C1180000000"              , sub(ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("66299C1180000000"              , sub(word_ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("29D1"                          , sub(ecx, edx));
  TEST_INSTRUCTION("299C1180000000"                , sub(ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("299C1180000000"                , sub(dword_ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("2A8C1A80000000"                , sub(cl, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("2A8C1A80000000"                , sub(cl, byte_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("2AAC1A80000000"                , sub(ch, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("2AAC1A80000000"                , sub(ch, byte_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("662B8C1A80000000"              , sub(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("662B8C1A80000000"              , sub(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("2B8C1A80000000"                , sub(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("2B8C1A80000000"                , sub(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F6C101"                        , test(cl, 1));
  TEST_INSTRUCTION("F6C501"                        , test(ch, 1));
  TEST_INSTRUCTION("F684118000000001"              , test(byte_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("66F7C10100"                    , test(cx, 1));
  TEST_INSTRUCTION("66F78411800000000100"          , test(word_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("F7C101000000"                  , test(ecx, 1));
  TEST_INSTRUCTION("F784118000000001000000"        , test(dword_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("84D1"                          , test(cl, dl));
  TEST_INSTRUCTION("84F1"                          , test(cl, dh));
  TEST_INSTRUCTION("84D5"                          , test(ch, dl));
  TEST_INSTRUCTION("84F5"                          , test(ch, dh));
  TEST_INSTRUCTION("849C1180000000"                , test(ptr(ecx, edx, 0, 128), bl));
  TEST_INSTRUCTION("84BC1180000000"                , test(ptr(ecx, edx, 0, 128), bh));
  TEST_INSTRUCTION("849C1180000000"                , test(byte_ptr(ecx, edx, 0, 128), bl));
  TEST_INSTRUCTION("84BC1180000000"                , test(byte_ptr(ecx, edx, 0, 128), bh));
  TEST_INSTRUCTION("6685D1"                        , test(cx, dx));
  TEST_INSTRUCTION("66859C1180000000"              , test(ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("66859C1180000000"              , test(word_ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("85D1"                          , test(ecx, edx));
  TEST_INSTRUCTION("859C1180000000"                , test(ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("859C1180000000"                , test(dword_ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("0FFFCA"                        , ud0(ecx, edx));
  TEST_INSTRUCTION("0FFF8C1A80000000"              , ud0(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FFF8C1A80000000"              , ud0(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FB9CA"                        , ud1(ecx, edx));
  TEST_INSTRUCTION("0FB98C1A80000000"              , ud1(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FB98C1A80000000"              , ud1(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0B"                          , ud2());
  TEST_INSTRUCTION("0F00E1"                        , verr(cx));
  TEST_INSTRUCTION("0F00A41180000000"              , verr(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F00A41180000000"              , verr(word_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F00E9"                        , verw(cx));
  TEST_INSTRUCTION("0F00AC1180000000"              , verw(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F00AC1180000000"              , verw(word_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F09"                          , wbinvd());
  TEST_INSTRUCTION("0F30"                          , wrmsr(edx, eax, ecx));
  TEST_INSTRUCTION("C6F801"                        , xabort(1));
  TEST_INSTRUCTION("0FC0D1"                        , xadd(cl, dl));
  TEST_INSTRUCTION("0FC0F1"                        , xadd(cl, dh));
  TEST_INSTRUCTION("0FC0D5"                        , xadd(ch, dl));
  TEST_INSTRUCTION("0FC0F5"                        , xadd(ch, dh));
  TEST_INSTRUCTION("0FC09C1180000000"              , xadd(ptr(ecx, edx, 0, 128), bl));
  TEST_INSTRUCTION("0FC0BC1180000000"              , xadd(ptr(ecx, edx, 0, 128), bh));
  TEST_INSTRUCTION("0FC09C1180000000"              , xadd(byte_ptr(ecx, edx, 0, 128), bl));
  TEST_INSTRUCTION("0FC0BC1180000000"              , xadd(byte_ptr(ecx, edx, 0, 128), bh));
  TEST_INSTRUCTION("660FC1D1"                      , xadd(cx, dx));
  TEST_INSTRUCTION("660FC19C1180000000"            , xadd(ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("660FC19C1180000000"            , xadd(word_ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("0FC1D1"                        , xadd(ecx, edx));
  TEST_INSTRUCTION("0FC19C1180000000"              , xadd(ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("0FC19C1180000000"              , xadd(dword_ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("86D1"                          , xchg(cl, dl));
  TEST_INSTRUCTION("86F1"                          , xchg(cl, dh));
  TEST_INSTRUCTION("86D5"                          , xchg(ch, dl));
  TEST_INSTRUCTION("86F5"                          , xchg(ch, dh));
  TEST_INSTRUCTION("869C1180000000"                , xchg(ptr(ecx, edx, 0, 128), bl));
  TEST_INSTRUCTION("86BC1180000000"                , xchg(ptr(ecx, edx, 0, 128), bh));
  TEST_INSTRUCTION("869C1180000000"                , xchg(byte_ptr(ecx, edx, 0, 128), bl));
  TEST_INSTRUCTION("86BC1180000000"                , xchg(byte_ptr(ecx, edx, 0, 128), bh));
  TEST_INSTRUCTION("6687D1"                        , xchg(cx, dx));
  TEST_INSTRUCTION("66879C1180000000"              , xchg(ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("66879C1180000000"              , xchg(word_ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("87D1"                          , xchg(ecx, edx));
  TEST_INSTRUCTION("879C1180000000"                , xchg(ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("879C1180000000"                , xchg(dword_ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("868C1A80000000"                , xchg(cl, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("868C1A80000000"                , xchg(cl, byte_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("86AC1A80000000"                , xchg(ch, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("86AC1A80000000"                , xchg(ch, byte_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("66878C1A80000000"              , xchg(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("66878C1A80000000"              , xchg(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("878C1A80000000"                , xchg(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("878C1A80000000"                , xchg(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("D7"                            , xlatb());
  TEST_INSTRUCTION("80F101"                        , xor_(cl, 1));
  TEST_INSTRUCTION("80F501"                        , xor_(ch, 1));
  TEST_INSTRUCTION("80B4118000000001"              , xor_(byte_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("6683F101"                      , xor_(cx, 1));
  TEST_INSTRUCTION("6683B4118000000001"            , xor_(word_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("83F101"                        , xor_(ecx, 1));
  TEST_INSTRUCTION("83B4118000000001"              , xor_(dword_ptr(ecx, edx, 0, 128), 1));
  TEST_INSTRUCTION("30D1"                          , xor_(cl, dl));
  TEST_INSTRUCTION("30F1"                          , xor_(cl, dh));
  TEST_INSTRUCTION("30D5"                          , xor_(ch, dl));
  TEST_INSTRUCTION("30F5"                          , xor_(ch, dh));
  TEST_INSTRUCTION("309C1180000000"                , xor_(ptr(ecx, edx, 0, 128), bl));
  TEST_INSTRUCTION("30BC1180000000"                , xor_(ptr(ecx, edx, 0, 128), bh));
  TEST_INSTRUCTION("309C1180000000"                , xor_(byte_ptr(ecx, edx, 0, 128), bl));
  TEST_INSTRUCTION("30BC1180000000"                , xor_(byte_ptr(ecx, edx, 0, 128), bh));
  TEST_INSTRUCTION("6631D1"                        , xor_(cx, dx));
  TEST_INSTRUCTION("66319C1180000000"              , xor_(ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("66319C1180000000"              , xor_(word_ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("31D1"                          , xor_(ecx, edx));
  TEST_INSTRUCTION("319C1180000000"                , xor_(ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("319C1180000000"                , xor_(dword_ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("328C1A80000000"                , xor_(cl, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("328C1A80000000"                , xor_(cl, byte_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("32AC1A80000000"                , xor_(ch, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("32AC1A80000000"                , xor_(ch, byte_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("66338C1A80000000"              , xor_(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("66338C1A80000000"              , xor_(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("338C1A80000000"                , xor_(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("338C1A80000000"                , xor_(ecx, dword_ptr(edx, ebx, 0, 128)));
}

static void ASMJIT_NOINLINE testX86AssemblerBaseExt(AssemblerTester<x86::Assembler>& tester) noexcept {
  using namespace x86;

  TEST_INSTRUCTION("0F38FC9C1180000000"            , aadd(ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("0F38FC9C1180000000"            , aadd(dword_ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("660F38FC9C1180000000"          , aand(ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("660F38FC9C1180000000"          , aand(dword_ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("660F38F6CA"                    , adcx(ecx, edx));
  TEST_INSTRUCTION("660F38F68C1A80000000"          , adcx(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38F68C1A80000000"          , adcx(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E268F2CB"                    , andn(ecx, edx, ebx));
  TEST_INSTRUCTION("C4E268F28C2B80000000"          , andn(ecx, edx, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E268F28C2B80000000"          , andn(ecx, edx, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("F20F38FC9C1180000000"          , aor(ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("F20F38FC9C1180000000"          , aor(dword_ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("F30F38FC9C1180000000"          , axor(ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("F30F38FC9C1180000000"          , axor(dword_ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("C4E260F7CA"                    , bextr(ecx, edx, ebx));
  TEST_INSTRUCTION("C4E258F78C1A80000000"          , bextr(ecx, ptr(edx, ebx, 0, 128), esp));
  TEST_INSTRUCTION("C4E258F78C1A80000000"          , bextr(ecx, dword_ptr(edx, ebx, 0, 128), esp));
  TEST_INSTRUCTION("8FE97001CA"                    , blcfill(ecx, edx));
  TEST_INSTRUCTION("8FE970018C1A80000000"          , blcfill(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE970018C1A80000000"          , blcfill(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE97002F2"                    , blci(ecx, edx));
  TEST_INSTRUCTION("8FE97002B41A80000000"          , blci(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE97002B41A80000000"          , blci(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE97001EA"                    , blcic(ecx, edx));
  TEST_INSTRUCTION("8FE97001AC1A80000000"          , blcic(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE97001AC1A80000000"          , blcic(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE97002CA"                    , blcmsk(ecx, edx));
  TEST_INSTRUCTION("8FE970028C1A80000000"          , blcmsk(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE970028C1A80000000"          , blcmsk(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE97001DA"                    , blcs(ecx, edx));
  TEST_INSTRUCTION("8FE970019C1A80000000"          , blcs(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE970019C1A80000000"          , blcs(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE97001D2"                    , blsfill(ecx, edx));
  TEST_INSTRUCTION("8FE97001941A80000000"          , blsfill(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE97001941A80000000"          , blsfill(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E270F3DA"                    , blsi(ecx, edx));
  TEST_INSTRUCTION("C4E270F39C1A80000000"          , blsi(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E270F39C1A80000000"          , blsi(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE97001F2"                    , blsic(ecx, edx));
  TEST_INSTRUCTION("8FE97001B41A80000000"          , blsic(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE97001B41A80000000"          , blsic(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E270F3D2"                    , blsmsk(ecx, edx));
  TEST_INSTRUCTION("C4E270F3941A80000000"          , blsmsk(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E270F3941A80000000"          , blsmsk(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E270F3CA"                    , blsr(ecx, edx));
  TEST_INSTRUCTION("C4E270F38C1A80000000"          , blsr(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E270F38C1A80000000"          , blsr(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F1ACA"                      , bndcl(bnd1, edx));
  TEST_INSTRUCTION("F30F1A8C1A80000000"            , bndcl(bnd1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F1A8C1A80000000"            , bndcl(bnd1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F1BCA"                      , bndcn(bnd1, edx));
  TEST_INSTRUCTION("F20F1B8C1A80000000"            , bndcn(bnd1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F1B8C1A80000000"            , bndcn(bnd1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F1ACA"                      , bndcu(bnd1, edx));
  TEST_INSTRUCTION("F20F1A8C1A80000000"            , bndcu(bnd1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F1A8C1A80000000"            , bndcu(bnd1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F1A8C1A80000000"              , bndldx(bnd1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F1B8C1A80000000"            , bndmk(bnd1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F1ACA"                      , bndmov(bnd1, bnd2));
  TEST_INSTRUCTION("660F1A8C1A80000000"            , bndmov(bnd1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F1B9C1180000000"            , bndmov(ptr(ecx, edx, 0, 128), bnd3));
  TEST_INSTRUCTION("0F1B9C1180000000"              , bndstx(ptr(ecx, edx, 0, 128), bnd3));
  TEST_INSTRUCTION("C4E260F5CA"                    , bzhi(ecx, edx, ebx));
  TEST_INSTRUCTION("C4E258F58C1A80000000"          , bzhi(ecx, ptr(edx, ebx, 0, 128), esp));
  TEST_INSTRUCTION("C4E258F58C1A80000000"          , bzhi(ecx, dword_ptr(edx, ebx, 0, 128), esp));
  TEST_INSTRUCTION("0F01CA"                        , clac());
  TEST_INSTRUCTION("0F1C841180000000"              , cldemote(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0FAEBC1180000000"              , clflush(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("660FAEBC1180000000"            , clflushopt(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F01DD"                        , clgi());
  TEST_INSTRUCTION("F30FAEB41180000000"            , clrssbsy(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("F30FAEB41180000000"            , clrssbsy(qword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F06"                          , clts());
  TEST_INSTRUCTION("660FAEB41180000000"            , clwb(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F01FC"                        , clzero(ptr(eax)));
  TEST_INSTRUCTION("0F01FC"                        , clzero(zmmword_ptr(eax)));
  TEST_INSTRUCTION("C4E259E69C1180000000"          , cmpbexadd(ptr(ecx, edx, 0, 128), ebx, esp));
  TEST_INSTRUCTION("C4E259E69C1180000000"          , cmpbexadd(dword_ptr(ecx, edx, 0, 128), ebx, esp));
  TEST_INSTRUCTION("C4E259E29C1180000000"          , cmpbxadd(ptr(ecx, edx, 0, 128), ebx, esp));
  TEST_INSTRUCTION("C4E259E29C1180000000"          , cmpbxadd(dword_ptr(ecx, edx, 0, 128), ebx, esp));
  TEST_INSTRUCTION("C4E259EE9C1180000000"          , cmplexadd(ptr(ecx, edx, 0, 128), ebx, esp));
  TEST_INSTRUCTION("C4E259EE9C1180000000"          , cmplexadd(dword_ptr(ecx, edx, 0, 128), ebx, esp));
  TEST_INSTRUCTION("C4E259EC9C1180000000"          , cmplxadd(ptr(ecx, edx, 0, 128), ebx, esp));
  TEST_INSTRUCTION("C4E259EC9C1180000000"          , cmplxadd(dword_ptr(ecx, edx, 0, 128), ebx, esp));
  TEST_INSTRUCTION("C4E259E79C1180000000"          , cmpnbexadd(ptr(ecx, edx, 0, 128), ebx, esp));
  TEST_INSTRUCTION("C4E259E79C1180000000"          , cmpnbexadd(dword_ptr(ecx, edx, 0, 128), ebx, esp));
  TEST_INSTRUCTION("C4E259E39C1180000000"          , cmpnbxadd(ptr(ecx, edx, 0, 128), ebx, esp));
  TEST_INSTRUCTION("C4E259E39C1180000000"          , cmpnbxadd(dword_ptr(ecx, edx, 0, 128), ebx, esp));
  TEST_INSTRUCTION("C4E259EF9C1180000000"          , cmpnlexadd(ptr(ecx, edx, 0, 128), ebx, esp));
  TEST_INSTRUCTION("C4E259EF9C1180000000"          , cmpnlexadd(dword_ptr(ecx, edx, 0, 128), ebx, esp));
  TEST_INSTRUCTION("C4E259ED9C1180000000"          , cmpnlxadd(ptr(ecx, edx, 0, 128), ebx, esp));
  TEST_INSTRUCTION("C4E259ED9C1180000000"          , cmpnlxadd(dword_ptr(ecx, edx, 0, 128), ebx, esp));
  TEST_INSTRUCTION("C4E259E19C1180000000"          , cmpnoxadd(ptr(ecx, edx, 0, 128), ebx, esp));
  TEST_INSTRUCTION("C4E259E19C1180000000"          , cmpnoxadd(dword_ptr(ecx, edx, 0, 128), ebx, esp));
  TEST_INSTRUCTION("C4E259EB9C1180000000"          , cmpnpxadd(ptr(ecx, edx, 0, 128), ebx, esp));
  TEST_INSTRUCTION("C4E259EB9C1180000000"          , cmpnpxadd(dword_ptr(ecx, edx, 0, 128), ebx, esp));
  TEST_INSTRUCTION("C4E259E99C1180000000"          , cmpnsxadd(ptr(ecx, edx, 0, 128), ebx, esp));
  TEST_INSTRUCTION("C4E259E99C1180000000"          , cmpnsxadd(dword_ptr(ecx, edx, 0, 128), ebx, esp));
  TEST_INSTRUCTION("C4E259E59C1180000000"          , cmpnzxadd(ptr(ecx, edx, 0, 128), ebx, esp));
  TEST_INSTRUCTION("C4E259E59C1180000000"          , cmpnzxadd(dword_ptr(ecx, edx, 0, 128), ebx, esp));
  TEST_INSTRUCTION("C4E259E09C1180000000"          , cmpoxadd(ptr(ecx, edx, 0, 128), ebx, esp));
  TEST_INSTRUCTION("C4E259E09C1180000000"          , cmpoxadd(dword_ptr(ecx, edx, 0, 128), ebx, esp));
  TEST_INSTRUCTION("C4E259EA9C1180000000"          , cmppxadd(ptr(ecx, edx, 0, 128), ebx, esp));
  TEST_INSTRUCTION("C4E259EA9C1180000000"          , cmppxadd(dword_ptr(ecx, edx, 0, 128), ebx, esp));
  TEST_INSTRUCTION("C4E259E89C1180000000"          , cmpsxadd(ptr(ecx, edx, 0, 128), ebx, esp));
  TEST_INSTRUCTION("C4E259E89C1180000000"          , cmpsxadd(dword_ptr(ecx, edx, 0, 128), ebx, esp));
  TEST_INSTRUCTION("F20F38F0CA"                    , crc32(ecx, dl));
  TEST_INSTRUCTION("F20F38F0CE"                    , crc32(ecx, dh));
  TEST_INSTRUCTION("66F20F38F1CA"                  , crc32(ecx, dx));
  TEST_INSTRUCTION("F20F38F1CA"                    , crc32(ecx, edx));
  TEST_INSTRUCTION("F20F38F08C1A80000000"          , crc32(ecx, byte_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("66F20F38F18C1A80000000"        , crc32(ecx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F38F18C1A80000000"          , crc32(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F77"                          , emms());
  TEST_INSTRUCTION("F30F1EFB"                      , endbr32());
  TEST_INSTRUCTION("F30F1EFA"                      , endbr64());
  TEST_INSTRUCTION("F20F38F88C1A80000000"          , enqcmd(zmmword_ptr(ecx), zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F38F88C1A80000000"          , enqcmds(zmmword_ptr(ecx), zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FAE8C1180000000"              , fxrstor(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0FAE841180000000"              , fxsave(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("D9F4"                          , fxtract());
  TEST_INSTRUCTION("0F37"                          , getsec());
  TEST_INSTRUCTION("F30F3AF0C001"                  , hreset(1, eax));
  TEST_INSTRUCTION("0FAE941180000000"              , ldmxcsr(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0FAE941180000000"              , ldmxcsr(dword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("F30FAEE9"                      , incsspd(ecx));
  TEST_INSTRUCTION("660F38808C1A80000000"          , invept(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38808C1A80000000"          , invept(ecx, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F01DF"                        , invlpga(eax, ecx));
  TEST_INSTRUCTION("660F38818C1A80000000"          , invvpid(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38818C1A80000000"          , invvpid(ecx, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FAEE8"                        , lfence());
  TEST_INSTRUCTION("8FE97812C1"                    , llwpcb(ecx));
  TEST_INSTRUCTION("8FEA7012C201000000"            , lwpins(ecx, edx, 1));
  TEST_INSTRUCTION("8FEA7012841A8000000001000000"  , lwpins(ecx, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("8FEA7012841A8000000001000000"  , lwpins(ecx, dword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("8FEA7012CA01000000"            , lwpval(ecx, edx, 1));
  TEST_INSTRUCTION("8FEA70128C1A8000000001000000"  , lwpval(ecx, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("8FEA70128C1A8000000001000000"  , lwpval(ecx, dword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("66F30FBDCA"                    , lzcnt(cx, dx));
  TEST_INSTRUCTION("66F30FBD8C1A80000000"          , lzcnt(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("66F30FBD8C1A80000000"          , lzcnt(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30FBDCA"                      , lzcnt(ecx, edx));
  TEST_INSTRUCTION("F30FBD8C1A80000000"            , lzcnt(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30FBD8C1A80000000"            , lzcnt(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F01FA"                      , mcommit());
  TEST_INSTRUCTION("0FAEF0"                        , mfence());
  TEST_INSTRUCTION("0F01C8"                        , monitor(ptr(eax), ecx, edx));
  TEST_INSTRUCTION("0F01FA"                        , monitorx(ptr(eax), ecx, edx));
  TEST_INSTRUCTION("660F38F08C1A80000000"          , movbe(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38F08C1A80000000"          , movbe(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F38F08C1A80000000"            , movbe(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F38F08C1A80000000"            , movbe(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38F19C1180000000"          , movbe(ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("660F38F19C1180000000"          , movbe(word_ptr(ecx, edx, 0, 128), bx));
  TEST_INSTRUCTION("0F38F19C1180000000"            , movbe(ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("0F38F19C1180000000"            , movbe(dword_ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("660F38F88C1A80000000"          , movdir64b(zmmword_ptr(ecx), zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F38F99C1180000000"            , movdiri(ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("0F38F99C1180000000"            , movdiri(dword_ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("0FC39C1180000000"              , movnti(ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("0FC39C1180000000"              , movnti(dword_ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("C4E26BF6CB"                    , mulx(ecx, edx, ebx, edx));
  TEST_INSTRUCTION("C4E26BF68C2B80000000"          , mulx(ecx, edx, ptr(ebx, ebp, 0, 128), edx));
  TEST_INSTRUCTION("C4E26BF68C2B80000000"          , mulx(ecx, edx, dword_ptr(ebx, ebp, 0, 128), edx));
  TEST_INSTRUCTION("0F01C9"                        , mwait(eax, ecx));
  TEST_INSTRUCTION("0F01FB"                        , mwaitx(eax, ecx, ebx));
  TEST_INSTRUCTION("0F01C5"                        , pconfig());
  TEST_INSTRUCTION("C4E26BF5CB"                    , pdep(ecx, edx, ebx));
  TEST_INSTRUCTION("C4E26BF58C2B80000000"          , pdep(ecx, edx, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26BF58C2B80000000"          , pdep(ecx, edx, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26AF5CB"                    , pext(ecx, edx, ebx));
  TEST_INSTRUCTION("C4E26AF58C2B80000000"          , pext(ecx, edx, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26AF58C2B80000000"          , pext(ecx, edx, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("66F30FB8CA"                    , popcnt(cx, dx));
  TEST_INSTRUCTION("66F30FB88C1A80000000"          , popcnt(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("66F30FB88C1A80000000"          , popcnt(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30FB8CA"                      , popcnt(ecx, edx));
  TEST_INSTRUCTION("F30FB88C1A80000000"            , popcnt(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30FB88C1A80000000"            , popcnt(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0D841180000000"              , prefetch(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F18841180000000"              , prefetchnta(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F188C1180000000"              , prefetcht0(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F18941180000000"              , prefetcht1(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F189C1180000000"              , prefetcht2(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F0D8C1180000000"              , prefetchw(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F0D941180000000"              , prefetchwt1(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("F30FAEE1"                      , ptwrite(ecx));
  TEST_INSTRUCTION("F30FAEA41180000000"            , ptwrite(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("F30FAEA41180000000"            , ptwrite(dword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("F20F01FF"                      , pvalidate());
  TEST_INSTRUCTION("F30FC7F9"                      , rdpid(ecx));
  TEST_INSTRUCTION("0F01EE"                        , rdpkru(edx, eax, ecx));
  TEST_INSTRUCTION("0F33"                          , rdpmc(edx, eax, ecx));
  TEST_INSTRUCTION("0F01FD"                        , rdpru(edx, eax, ecx));
  TEST_INSTRUCTION("660FC7F1"                      , rdrand(cx));
  TEST_INSTRUCTION("0FC7F1"                        , rdrand(ecx));
  TEST_INSTRUCTION("660FC7F9"                      , rdseed(cx));
  TEST_INSTRUCTION("0FC7F9"                        , rdseed(ecx));
  TEST_INSTRUCTION("F30F1EC9"                      , rdsspd(ecx));
  TEST_INSTRUCTION("0F31"                          , rdtsc(edx, eax));
  TEST_INSTRUCTION("0F01F9"                        , rdtscp(edx, eax, ecx));
  TEST_INSTRUCTION("C4E37BF0CA01"                  , rorx(ecx, edx, 1));
  TEST_INSTRUCTION("C4E37BF08C1A8000000001"        , rorx(ecx, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C4E37BF08C1A8000000001"        , rorx(ecx, dword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("0FAA"                          , rsm());
  TEST_INSTRUCTION("F30F01AC1180000000"            , rstorssp(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("F30F01AC1180000000"            , rstorssp(qword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("C4E262F7CA"                    , sarx(ecx, edx, ebx));
  TEST_INSTRUCTION("C4E25AF78C1A80000000"          , sarx(ecx, ptr(edx, ebx, 0, 128), esp));
  TEST_INSTRUCTION("C4E25AF78C1A80000000"          , sarx(ecx, dword_ptr(edx, ebx, 0, 128), esp));
  TEST_INSTRUCTION("F30F01EA"                      , saveprevssp());
  TEST_INSTRUCTION("0F01E8"                        , serialize());
  TEST_INSTRUCTION("F30F01E8"                      , setssbsy());
  TEST_INSTRUCTION("0FAEF8"                        , sfence());
  TEST_INSTRUCTION("C4E261F7CA"                    , shlx(ecx, edx, ebx));
  TEST_INSTRUCTION("C4E259F78C1A80000000"          , shlx(ecx, ptr(edx, ebx, 0, 128), esp));
  TEST_INSTRUCTION("C4E259F78C1A80000000"          , shlx(ecx, dword_ptr(edx, ebx, 0, 128), esp));
  TEST_INSTRUCTION("C4E263F7CA"                    , shrx(ecx, edx, ebx));
  TEST_INSTRUCTION("C4E25BF78C1A80000000"          , shrx(ecx, ptr(edx, ebx, 0, 128), esp));
  TEST_INSTRUCTION("C4E25BF78C1A80000000"          , shrx(ecx, dword_ptr(edx, ebx, 0, 128), esp));
  TEST_INSTRUCTION("8FE97812C9"                    , slwpcb(ecx));
  TEST_INSTRUCTION("0F01DE"                        , skinit(eax));
  TEST_INSTRUCTION("0F01CB"                        , stac());
  TEST_INSTRUCTION("0F01DC"                        , stgi());
  TEST_INSTRUCTION("0FAE9C1180000000"              , stmxcsr(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0FAE9C1180000000"              , stmxcsr(dword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F34"                          , sysenter());
  TEST_INSTRUCTION("0F35"                          , sysexit());
  TEST_INSTRUCTION("8FE97001FA"                    , t1mskc(ecx, edx));
  TEST_INSTRUCTION("8FE97001BC1A80000000"          , t1mskc(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE97001BC1A80000000"          , t1mskc(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FAEF1"                      , tpause(ecx, edx, eax));
  TEST_INSTRUCTION("66F30FBCCA"                    , tzcnt(cx, dx));
  TEST_INSTRUCTION("66F30FBC8C1A80000000"          , tzcnt(cx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("66F30FBC8C1A80000000"          , tzcnt(cx, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30FBCCA"                      , tzcnt(ecx, edx));
  TEST_INSTRUCTION("F30FBC8C1A80000000"            , tzcnt(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30FBC8C1A80000000"            , tzcnt(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE97001E2"                    , tzmsk(ecx, edx));
  TEST_INSTRUCTION("8FE97001A41A80000000"          , tzmsk(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE97001A41A80000000"          , tzmsk(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30FAEF1"                      , umonitor(ptr(ecx)));
  TEST_INSTRUCTION("F20FAEF1"                      , umwait(ecx, edx, eax));
  TEST_INSTRUCTION("0F01C1"                        , vmcall());
  TEST_INSTRUCTION("660FC7B41180000000"            , vmclear(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("660FC7B41180000000"            , vmclear(qword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F01D4"                        , vmfunc());
  TEST_INSTRUCTION("0F01C2"                        , vmlaunch());
  TEST_INSTRUCTION("0F01DA"                        , vmload(eax));
  TEST_INSTRUCTION("0F01D9"                        , vmmcall());
  TEST_INSTRUCTION("0FC7B41180000000"              , vmptrld(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0FC7B41180000000"              , vmptrld(qword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0FC7BC1180000000"              , vmptrst(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0FC7BC1180000000"              , vmptrst(qword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("0F78D1"                        , vmread(ecx, edx));
  TEST_INSTRUCTION("0F789C1180000000"              , vmread(ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("0F789C1180000000"              , vmread(dword_ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("0F01C3"                        , vmresume());
  TEST_INSTRUCTION("0F01D8"                        , vmrun(eax));
  TEST_INSTRUCTION("0F01DB"                        , vmsave(eax));
  TEST_INSTRUCTION("0F79CA"                        , vmwrite(ecx, edx));
  TEST_INSTRUCTION("0F798C1A80000000"              , vmwrite(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F798C1A80000000"              , vmwrite(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30FC7B41180000000"            , vmxon(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("F30FC7B41180000000"            , vmxon(qword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("F30F09"                        , wbnoinvd());
  TEST_INSTRUCTION("0F38F6D1"                      , wrssd(ecx, edx));
  TEST_INSTRUCTION("0F38F69C1180000000"            , wrssd(ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("0F38F69C1180000000"            , wrssd(dword_ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("660F38F5D1"                    , wrussd(ecx, edx));
  TEST_INSTRUCTION("660F38F59C1180000000"          , wrussd(ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("660F38F59C1180000000"          , wrussd(dword_ptr(ecx, edx, 0, 128), ebx));
  TEST_INSTRUCTION("0F01D5"                        , xend());
  TEST_INSTRUCTION("0F01D0"                        , xgetbv(edx, eax, ecx));
  TEST_INSTRUCTION("F20F01E9"                      , xresldtrk());
  TEST_INSTRUCTION("0FAEAC1180000000"              , xrstor(ptr(ecx, edx, 0, 128), edx, eax));
  TEST_INSTRUCTION("0FC79C1180000000"              , xrstors(ptr(ecx, edx, 0, 128), edx, eax));
  TEST_INSTRUCTION("0FAEA41180000000"              , xsave(ptr(ecx, edx, 0, 128), edx, eax));
  TEST_INSTRUCTION("0FC7A41180000000"              , xsavec(ptr(ecx, edx, 0, 128), edx, eax));
  TEST_INSTRUCTION("0FAEB41180000000"              , xsaveopt(ptr(ecx, edx, 0, 128), edx, eax));
  TEST_INSTRUCTION("0FC7AC1180000000"              , xsaves(ptr(ecx, edx, 0, 128), edx, eax));
  TEST_INSTRUCTION("0F01D1"                        , xsetbv(edx, eax, ecx));
  TEST_INSTRUCTION("F20F01E8"                      , xsusldtrk());
  TEST_INSTRUCTION("0F01D6"                        , xtest());
}

static void ASMJIT_NOINLINE testX86AssemblerMMX_SSE(AssemblerTester<x86::Assembler>& tester) noexcept {
  using namespace x86;

  TEST_INSTRUCTION("660F58CA"                      , addpd(xmm1, xmm2));
  TEST_INSTRUCTION("660F588C1A80000000"            , addpd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F588C1A80000000"            , addpd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F58CA"                        , addps(xmm1, xmm2));
  TEST_INSTRUCTION("0F588C1A80000000"              , addps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F588C1A80000000"              , addps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F58CA"                      , addsd(xmm1, xmm2));
  TEST_INSTRUCTION("F20F588C1A80000000"            , addsd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F588C1A80000000"            , addsd(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F58CA"                      , addss(xmm1, xmm2));
  TEST_INSTRUCTION("F30F588C1A80000000"            , addss(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F588C1A80000000"            , addss(xmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FD0CA"                      , addsubpd(xmm1, xmm2));
  TEST_INSTRUCTION("660FD08C1A80000000"            , addsubpd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FD08C1A80000000"            , addsubpd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20FD0CA"                      , addsubps(xmm1, xmm2));
  TEST_INSTRUCTION("F20FD08C1A80000000"            , addsubps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20FD08C1A80000000"            , addsubps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F38F6CA"                    , adox(ecx, edx));
  TEST_INSTRUCTION("F30F38F68C1A80000000"          , adox(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F38F68C1A80000000"          , adox(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38DECA"                    , aesdec(xmm1, xmm2));
  TEST_INSTRUCTION("660F38DE8C1A80000000"          , aesdec(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38DE8C1A80000000"          , aesdec(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38DFCA"                    , aesdeclast(xmm1, xmm2));
  TEST_INSTRUCTION("660F38DF8C1A80000000"          , aesdeclast(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38DF8C1A80000000"          , aesdeclast(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38DCCA"                    , aesenc(xmm1, xmm2));
  TEST_INSTRUCTION("660F38DC8C1A80000000"          , aesenc(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38DC8C1A80000000"          , aesenc(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38DDCA"                    , aesenclast(xmm1, xmm2));
  TEST_INSTRUCTION("660F38DD8C1A80000000"          , aesenclast(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38DD8C1A80000000"          , aesenclast(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38DBCA"                    , aesimc(xmm1, xmm2));
  TEST_INSTRUCTION("660F38DB8C1A80000000"          , aesimc(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38DB8C1A80000000"          , aesimc(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3ADFCA01"                  , aeskeygenassist(xmm1, xmm2, 1));
  TEST_INSTRUCTION("660F3ADF8C1A8000000001"        , aeskeygenassist(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F3ADF8C1A8000000001"        , aeskeygenassist(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F55CA"                      , andnpd(xmm1, xmm2));
  TEST_INSTRUCTION("660F558C1A80000000"            , andnpd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F558C1A80000000"            , andnpd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F55CA"                        , andnps(xmm1, xmm2));
  TEST_INSTRUCTION("0F558C1A80000000"              , andnps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F558C1A80000000"              , andnps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F54CA"                      , andpd(xmm1, xmm2));
  TEST_INSTRUCTION("660F548C1A80000000"            , andpd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F548C1A80000000"            , andpd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F54CA"                        , andps(xmm1, xmm2));
  TEST_INSTRUCTION("0F548C1A80000000"              , andps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F548C1A80000000"              , andps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3A0DCA01"                  , blendpd(xmm1, xmm2, 1));
  TEST_INSTRUCTION("660F3A0D8C1A8000000001"        , blendpd(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F3A0D8C1A8000000001"        , blendpd(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F3A0CCA01"                  , blendps(xmm1, xmm2, 1));
  TEST_INSTRUCTION("660F3A0C8C1A8000000001"        , blendps(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F3A0C8C1A8000000001"        , blendps(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F3815CA"                    , blendvpd(xmm1, xmm2, xmm0));
  TEST_INSTRUCTION("660F38158C1A80000000"          , blendvpd(xmm1, ptr(edx, ebx, 0, 128), xmm0));
  TEST_INSTRUCTION("660F38158C1A80000000"          , blendvpd(xmm1, xmmword_ptr(edx, ebx, 0, 128), xmm0));
  TEST_INSTRUCTION("660F3814CA"                    , blendvps(xmm1, xmm2, xmm0));
  TEST_INSTRUCTION("660F38148C1A80000000"          , blendvps(xmm1, ptr(edx, ebx, 0, 128), xmm0));
  TEST_INSTRUCTION("660F38148C1A80000000"          , blendvps(xmm1, xmmword_ptr(edx, ebx, 0, 128), xmm0));
  TEST_INSTRUCTION("660FC2CA01"                    , cmppd(xmm1, xmm2, 1));
  TEST_INSTRUCTION("660FC28C1A8000000001"          , cmppd(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660FC28C1A8000000001"          , cmppd(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("0FC2CA01"                      , cmpps(xmm1, xmm2, 1));
  TEST_INSTRUCTION("0FC28C1A8000000001"            , cmpps(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("0FC28C1A8000000001"            , cmpps(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("F20FC2CA01"                    , cmpsd(xmm1, xmm2, 1));
  TEST_INSTRUCTION("F20FC28C1A8000000001"          , cmpsd(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("F20FC28C1A8000000001"          , cmpsd(xmm1, qword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("F30FC2CA01"                    , cmpss(xmm1, xmm2, 1));
  TEST_INSTRUCTION("F30FC28C1A8000000001"          , cmpss(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("F30FC28C1A8000000001"          , cmpss(xmm1, dword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F2FCA"                      , comisd(xmm1, xmm2));
  TEST_INSTRUCTION("660F2F8C1A80000000"            , comisd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F2F8C1A80000000"            , comisd(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F2FCA"                        , comiss(xmm1, xmm2));
  TEST_INSTRUCTION("0F2F8C1A80000000"              , comiss(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F2F8C1A80000000"              , comiss(xmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30FE6CA"                      , cvtdq2pd(xmm1, xmm2));
  TEST_INSTRUCTION("F30FE68C1A80000000"            , cvtdq2pd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30FE68C1A80000000"            , cvtdq2pd(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F5BCA"                        , cvtdq2ps(xmm1, xmm2));
  TEST_INSTRUCTION("0F5B8C1A80000000"              , cvtdq2ps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F5B8C1A80000000"              , cvtdq2ps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20FE6CA"                      , cvtpd2dq(xmm1, xmm2));
  TEST_INSTRUCTION("F20FE68C1A80000000"            , cvtpd2dq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20FE68C1A80000000"            , cvtpd2dq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F2DCA"                      , cvtpd2pi(mm1, xmm2));
  TEST_INSTRUCTION("660F2D8C1A80000000"            , cvtpd2pi(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F2D8C1A80000000"            , cvtpd2pi(mm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F5ACA"                      , cvtpd2ps(xmm1, xmm2));
  TEST_INSTRUCTION("660F5A8C1A80000000"            , cvtpd2ps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F5A8C1A80000000"            , cvtpd2ps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F2ACA"                      , cvtpi2pd(xmm1, mm2));
  TEST_INSTRUCTION("660F2A8C1A80000000"            , cvtpi2pd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F2A8C1A80000000"            , cvtpi2pd(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F2ACA"                        , cvtpi2ps(xmm1, mm2));
  TEST_INSTRUCTION("0F2A8C1A80000000"              , cvtpi2ps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F2A8C1A80000000"              , cvtpi2ps(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F5BCA"                      , cvtps2dq(xmm1, xmm2));
  TEST_INSTRUCTION("660F5B8C1A80000000"            , cvtps2dq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F5B8C1A80000000"            , cvtps2dq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F5ACA"                        , cvtps2pd(xmm1, xmm2));
  TEST_INSTRUCTION("0F5A8C1A80000000"              , cvtps2pd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F5A8C1A80000000"              , cvtps2pd(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F2DCA"                        , cvtps2pi(mm1, xmm2));
  TEST_INSTRUCTION("0F2D8C1A80000000"              , cvtps2pi(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F2D8C1A80000000"              , cvtps2pi(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F2DCA"                      , cvtsd2si(ecx, xmm2));
  TEST_INSTRUCTION("F20F2D8C1A80000000"            , cvtsd2si(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F2D8C1A80000000"            , cvtsd2si(ecx, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F5ACA"                      , cvtsd2ss(xmm1, xmm2));
  TEST_INSTRUCTION("F20F5A8C1A80000000"            , cvtsd2ss(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F5A8C1A80000000"            , cvtsd2ss(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F2ACA"                      , cvtsi2sd(xmm1, edx));
  TEST_INSTRUCTION("F20F2A8C1A80000000"            , cvtsi2sd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F2A8C1A80000000"            , cvtsi2sd(xmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F2ACA"                      , cvtsi2ss(xmm1, edx));
  TEST_INSTRUCTION("F30F2A8C1A80000000"            , cvtsi2ss(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F2A8C1A80000000"            , cvtsi2ss(xmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F5ACA"                      , cvtss2sd(xmm1, xmm2));
  TEST_INSTRUCTION("F30F5A8C1A80000000"            , cvtss2sd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F5A8C1A80000000"            , cvtss2sd(xmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F2DCA"                      , cvtss2si(ecx, xmm2));
  TEST_INSTRUCTION("F30F2D8C1A80000000"            , cvtss2si(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F2D8C1A80000000"            , cvtss2si(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FE6CA"                      , cvttpd2dq(xmm1, xmm2));
  TEST_INSTRUCTION("660FE68C1A80000000"            , cvttpd2dq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FE68C1A80000000"            , cvttpd2dq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F2CCA"                      , cvttpd2pi(mm1, xmm2));
  TEST_INSTRUCTION("660F2C8C1A80000000"            , cvttpd2pi(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F2C8C1A80000000"            , cvttpd2pi(mm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F5BCA"                      , cvttps2dq(xmm1, xmm2));
  TEST_INSTRUCTION("F30F5B8C1A80000000"            , cvttps2dq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F5B8C1A80000000"            , cvttps2dq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F2CCA"                        , cvttps2pi(mm1, xmm2));
  TEST_INSTRUCTION("0F2C8C1A80000000"              , cvttps2pi(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F2C8C1A80000000"              , cvttps2pi(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F2CCA"                      , cvttsd2si(ecx, xmm2));
  TEST_INSTRUCTION("F20F2C8C1A80000000"            , cvttsd2si(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F2C8C1A80000000"            , cvttsd2si(ecx, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F2CCA"                      , cvttss2si(ecx, xmm2));
  TEST_INSTRUCTION("F30F2C8C1A80000000"            , cvttss2si(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F2C8C1A80000000"            , cvttss2si(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F5ECA"                      , divpd(xmm1, xmm2));
  TEST_INSTRUCTION("660F5E8C1A80000000"            , divpd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F5E8C1A80000000"            , divpd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F5ECA"                        , divps(xmm1, xmm2));
  TEST_INSTRUCTION("0F5E8C1A80000000"              , divps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F5E8C1A80000000"              , divps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F5ECA"                      , divsd(xmm1, xmm2));
  TEST_INSTRUCTION("F20F5E8C1A80000000"            , divsd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F5E8C1A80000000"            , divsd(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F5ECA"                      , divss(xmm1, xmm2));
  TEST_INSTRUCTION("F30F5E8C1A80000000"            , divss(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F5E8C1A80000000"            , divss(xmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3A41CA01"                  , dppd(xmm1, xmm2, 1));
  TEST_INSTRUCTION("660F3A418C1A8000000001"        , dppd(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F3A418C1A8000000001"        , dppd(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F3A40CA01"                  , dpps(xmm1, xmm2, 1));
  TEST_INSTRUCTION("660F3A408C1A8000000001"        , dpps(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F3A408C1A8000000001"        , dpps(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F3A17D101"                  , extractps(ecx, xmm2, 1));
  TEST_INSTRUCTION("660F3A179C118000000001"        , extractps(ptr(ecx, edx, 0, 128), xmm3, 1));
  TEST_INSTRUCTION("660F3A179C118000000001"        , extractps(dword_ptr(ecx, edx, 0, 128), xmm3, 1));
  TEST_INSTRUCTION("660F78C10102"                  , extrq(xmm1, 1, 2));
  TEST_INSTRUCTION("660F79CA"                      , extrq(xmm1, xmm2));
  TEST_INSTRUCTION("660F3ACFCA01"                  , gf2p8affineinvqb(xmm1, xmm2, 1));
  TEST_INSTRUCTION("660F3ACF8C1A8000000001"        , gf2p8affineinvqb(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F3ACF8C1A8000000001"        , gf2p8affineinvqb(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F3ACECA01"                  , gf2p8affineqb(xmm1, xmm2, 1));
  TEST_INSTRUCTION("660F3ACE8C1A8000000001"        , gf2p8affineqb(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F3ACE8C1A8000000001"        , gf2p8affineqb(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F38CFCA"                    , gf2p8mulb(xmm1, xmm2));
  TEST_INSTRUCTION("660F38CF8C1A80000000"          , gf2p8mulb(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38CF8C1A80000000"          , gf2p8mulb(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F7CCA"                      , haddpd(xmm1, xmm2));
  TEST_INSTRUCTION("660F7C8C1A80000000"            , haddpd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F7C8C1A80000000"            , haddpd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F7CCA"                      , haddps(xmm1, xmm2));
  TEST_INSTRUCTION("F20F7C8C1A80000000"            , haddps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F7C8C1A80000000"            , haddps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F7DCA"                      , hsubpd(xmm1, xmm2));
  TEST_INSTRUCTION("660F7D8C1A80000000"            , hsubpd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F7D8C1A80000000"            , hsubpd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F7DCA"                      , hsubps(xmm1, xmm2));
  TEST_INSTRUCTION("F20F7D8C1A80000000"            , hsubps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F7D8C1A80000000"            , hsubps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3A21CA01"                  , insertps(xmm1, xmm2, 1));
  TEST_INSTRUCTION("660F3A218C1A8000000001"        , insertps(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F3A218C1A8000000001"        , insertps(xmm1, dword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("F20F79CA"                      , insertq(xmm1, xmm2));
  TEST_INSTRUCTION("F20F78CA0102"                  , insertq(xmm1, xmm2, 1, 2));
  TEST_INSTRUCTION("F20FF08C1A80000000"            , lddqu(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20FF08C1A80000000"            , lddqu(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FF7CA"                      , maskmovdqu(xmm1, xmm2, ptr(edi)));
  TEST_INSTRUCTION("660FF7CA"                      , maskmovdqu(xmm1, xmm2, xmmword_ptr(edi)));
  TEST_INSTRUCTION("0FF7CA"                        , maskmovq(mm1, mm2, ptr(edi)));
  TEST_INSTRUCTION("0FF7CA"                        , maskmovq(mm1, mm2, qword_ptr(edi)));
  TEST_INSTRUCTION("660F5FCA"                      , maxpd(xmm1, xmm2));
  TEST_INSTRUCTION("660F5F8C1A80000000"            , maxpd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F5F8C1A80000000"            , maxpd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F5FCA"                        , maxps(xmm1, xmm2));
  TEST_INSTRUCTION("0F5F8C1A80000000"              , maxps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F5F8C1A80000000"              , maxps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F5FCA"                      , maxsd(xmm1, xmm2));
  TEST_INSTRUCTION("F20F5F8C1A80000000"            , maxsd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F5F8C1A80000000"            , maxsd(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F5FCA"                      , maxss(xmm1, xmm2));
  TEST_INSTRUCTION("F30F5F8C1A80000000"            , maxss(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F5F8C1A80000000"            , maxss(xmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F5DCA"                      , minpd(xmm1, xmm2));
  TEST_INSTRUCTION("660F5D8C1A80000000"            , minpd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F5D8C1A80000000"            , minpd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F5DCA"                        , minps(xmm1, xmm2));
  TEST_INSTRUCTION("0F5D8C1A80000000"              , minps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F5D8C1A80000000"              , minps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F5DCA"                      , minsd(xmm1, xmm2));
  TEST_INSTRUCTION("F20F5D8C1A80000000"            , minsd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F5D8C1A80000000"            , minsd(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F5DCA"                      , minss(xmm1, xmm2));
  TEST_INSTRUCTION("F30F5D8C1A80000000"            , minss(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F5D8C1A80000000"            , minss(xmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F28CA"                      , movapd(xmm1, xmm2));
  TEST_INSTRUCTION("660F288C1A80000000"            , movapd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F288C1A80000000"            , movapd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F299C1180000000"            , movapd(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("660F299C1180000000"            , movapd(xmmword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("0F28CA"                        , movaps(xmm1, xmm2));
  TEST_INSTRUCTION("0F288C1A80000000"              , movaps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F288C1A80000000"              , movaps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F299C1180000000"              , movaps(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("0F299C1180000000"              , movaps(xmmword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("660F6ECA"                      , movd(xmm1, edx));
  TEST_INSTRUCTION("660F6E8C1A80000000"            , movd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F6E8C1A80000000"            , movd(xmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F6ECA"                        , movd(mm1, edx));
  TEST_INSTRUCTION("0F6E8C1A80000000"              , movd(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F6E8C1A80000000"              , movd(mm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F7ED1"                      , movd(ecx, xmm2));
  TEST_INSTRUCTION("0F7ED1"                        , movd(ecx, mm2));
  TEST_INSTRUCTION("660F7E9C1180000000"            , movd(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("0F7E9C1180000000"              , movd(ptr(ecx, edx, 0, 128), mm3));
  TEST_INSTRUCTION("660F7E9C1180000000"            , movd(dword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("0F7E9C1180000000"              , movd(dword_ptr(ecx, edx, 0, 128), mm3));
  TEST_INSTRUCTION("F20F12CA"                      , movddup(xmm1, xmm2));
  TEST_INSTRUCTION("F20F128C1A80000000"            , movddup(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F128C1A80000000"            , movddup(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20FD6CA"                      , movdq2q(mm1, xmm2));
  TEST_INSTRUCTION("660F6FCA"                      , movdqa(xmm1, xmm2));
  TEST_INSTRUCTION("660F6F8C1A80000000"            , movdqa(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F6F8C1A80000000"            , movdqa(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F7F9C1180000000"            , movdqa(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("660F7F9C1180000000"            , movdqa(xmmword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("F30F6FCA"                      , movdqu(xmm1, xmm2));
  TEST_INSTRUCTION("F30F6F8C1A80000000"            , movdqu(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F6F8C1A80000000"            , movdqu(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F7F9C1180000000"            , movdqu(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("F30F7F9C1180000000"            , movdqu(xmmword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("0F12CA"                        , movhlps(xmm1, xmm2));
  TEST_INSTRUCTION("660F179C1180000000"            , movhpd(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("660F179C1180000000"            , movhpd(qword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("660F168C1A80000000"            , movhpd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F168C1A80000000"            , movhpd(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F179C1180000000"              , movhps(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("0F179C1180000000"              , movhps(qword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("0F168C1A80000000"              , movhps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F168C1A80000000"              , movhps(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F16CA"                        , movlhps(xmm1, xmm2));
  TEST_INSTRUCTION("660F139C1180000000"            , movlpd(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("660F139C1180000000"            , movlpd(qword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("660F128C1A80000000"            , movlpd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F128C1A80000000"            , movlpd(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F139C1180000000"              , movlps(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("0F139C1180000000"              , movlps(qword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("0F128C1A80000000"              , movlps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F128C1A80000000"              , movlps(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F50CA"                      , movmskpd(ecx, xmm2));
  TEST_INSTRUCTION("0F50CA"                        , movmskps(ecx, xmm2));
  TEST_INSTRUCTION("660FE79C1180000000"            , movntdq(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("660FE79C1180000000"            , movntdq(xmmword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("660F382A8C1A80000000"          , movntdqa(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F382A8C1A80000000"          , movntdqa(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F2B9C1180000000"            , movntpd(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("660F2B9C1180000000"            , movntpd(xmmword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("0F2B9C1180000000"              , movntps(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("0F2B9C1180000000"              , movntps(xmmword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("0FE79C1180000000"              , movntq(ptr(ecx, edx, 0, 128), mm3));
  TEST_INSTRUCTION("0FE79C1180000000"              , movntq(qword_ptr(ecx, edx, 0, 128), mm3));
  TEST_INSTRUCTION("F20F2B9C1180000000"            , movntsd(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("F20F2B9C1180000000"            , movntsd(qword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("F30F2B9C1180000000"            , movntss(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("F30F2B9C1180000000"            , movntss(dword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("0F6FCA"                        , movq(mm1, mm2));
  TEST_INSTRUCTION("0F6F8C1A80000000"              , movq(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F6F8C1A80000000"              , movq(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FD69C1180000000"            , movq(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("0F7F9C1180000000"              , movq(ptr(ecx, edx, 0, 128), mm3));
  TEST_INSTRUCTION("660FD69C1180000000"            , movq(qword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("0F7F9C1180000000"              , movq(qword_ptr(ecx, edx, 0, 128), mm3));
  TEST_INSTRUCTION("F30F7ECA"                      , movq(xmm1, xmm2));
  TEST_INSTRUCTION("F30F7E8C1A80000000"            , movq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F7E8C1A80000000"            , movq(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30FD6CA"                      , movq2dq(xmm1, mm2));
  TEST_INSTRUCTION("F20F10CA"                      , movsd(xmm1, xmm2));
  TEST_INSTRUCTION("F20F108C1A80000000"            , movsd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F108C1A80000000"            , movsd(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F119C1180000000"            , movsd(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("F20F119C1180000000"            , movsd(qword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("F30F16CA"                      , movshdup(xmm1, xmm2));
  TEST_INSTRUCTION("F30F168C1A80000000"            , movshdup(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F168C1A80000000"            , movshdup(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F12CA"                      , movsldup(xmm1, xmm2));
  TEST_INSTRUCTION("F30F128C1A80000000"            , movsldup(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F128C1A80000000"            , movsldup(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F10CA"                      , movss(xmm1, xmm2));
  TEST_INSTRUCTION("F30F108C1A80000000"            , movss(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F108C1A80000000"            , movss(xmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F119C1180000000"            , movss(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("F30F119C1180000000"            , movss(dword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("660F10CA"                      , movupd(xmm1, xmm2));
  TEST_INSTRUCTION("660F108C1A80000000"            , movupd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F108C1A80000000"            , movupd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F119C1180000000"            , movupd(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("660F119C1180000000"            , movupd(xmmword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("0F10CA"                        , movups(xmm1, xmm2));
  TEST_INSTRUCTION("0F108C1A80000000"              , movups(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F108C1A80000000"              , movups(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F119C1180000000"              , movups(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("0F119C1180000000"              , movups(xmmword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("660F3A42CA01"                  , mpsadbw(xmm1, xmm2, 1));
  TEST_INSTRUCTION("660F3A428C1A8000000001"        , mpsadbw(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F3A428C1A8000000001"        , mpsadbw(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F59CA"                      , mulpd(xmm1, xmm2));
  TEST_INSTRUCTION("660F598C1A80000000"            , mulpd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F598C1A80000000"            , mulpd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F59CA"                        , mulps(xmm1, xmm2));
  TEST_INSTRUCTION("0F598C1A80000000"              , mulps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F598C1A80000000"              , mulps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F59CA"                      , mulsd(xmm1, xmm2));
  TEST_INSTRUCTION("F20F598C1A80000000"            , mulsd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F598C1A80000000"            , mulsd(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F59CA"                      , mulss(xmm1, xmm2));
  TEST_INSTRUCTION("F30F598C1A80000000"            , mulss(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F598C1A80000000"            , mulss(xmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F56CA"                      , orpd(xmm1, xmm2));
  TEST_INSTRUCTION("660F568C1A80000000"            , orpd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F568C1A80000000"            , orpd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F56CA"                        , orps(xmm1, xmm2));
  TEST_INSTRUCTION("0F568C1A80000000"              , orps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F568C1A80000000"              , orps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F381CCA"                      , pabsb(mm1, mm2));
  TEST_INSTRUCTION("0F381C8C1A80000000"            , pabsb(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F381C8C1A80000000"            , pabsb(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F381CCA"                    , pabsb(xmm1, xmm2));
  TEST_INSTRUCTION("660F381C8C1A80000000"          , pabsb(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F381C8C1A80000000"          , pabsb(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F381ECA"                      , pabsd(mm1, mm2));
  TEST_INSTRUCTION("0F381E8C1A80000000"            , pabsd(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F381E8C1A80000000"            , pabsd(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F381ECA"                    , pabsd(xmm1, xmm2));
  TEST_INSTRUCTION("660F381E8C1A80000000"          , pabsd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F381E8C1A80000000"          , pabsd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F381DCA"                      , pabsw(mm1, mm2));
  TEST_INSTRUCTION("0F381D8C1A80000000"            , pabsw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F381D8C1A80000000"            , pabsw(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F381DCA"                    , pabsw(xmm1, xmm2));
  TEST_INSTRUCTION("660F381D8C1A80000000"          , pabsw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F381D8C1A80000000"          , pabsw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F6BCA"                        , packssdw(mm1, mm2));
  TEST_INSTRUCTION("0F6B8C1A80000000"              , packssdw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F6B8C1A80000000"              , packssdw(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F6BCA"                      , packssdw(xmm1, xmm2));
  TEST_INSTRUCTION("660F6B8C1A80000000"            , packssdw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F6B8C1A80000000"            , packssdw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F63CA"                        , packsswb(mm1, mm2));
  TEST_INSTRUCTION("0F638C1A80000000"              , packsswb(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F638C1A80000000"              , packsswb(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F63CA"                      , packsswb(xmm1, xmm2));
  TEST_INSTRUCTION("660F638C1A80000000"            , packsswb(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F638C1A80000000"            , packsswb(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F382BCA"                    , packusdw(xmm1, xmm2));
  TEST_INSTRUCTION("660F382B8C1A80000000"          , packusdw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F382B8C1A80000000"          , packusdw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F67CA"                        , packuswb(mm1, mm2));
  TEST_INSTRUCTION("0F678C1A80000000"              , packuswb(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F678C1A80000000"              , packuswb(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F67CA"                      , packuswb(xmm1, xmm2));
  TEST_INSTRUCTION("660F678C1A80000000"            , packuswb(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F678C1A80000000"            , packuswb(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FFCCA"                        , paddb(mm1, mm2));
  TEST_INSTRUCTION("0FFC8C1A80000000"              , paddb(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FFC8C1A80000000"              , paddb(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FFCCA"                      , paddb(xmm1, xmm2));
  TEST_INSTRUCTION("660FFC8C1A80000000"            , paddb(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FFC8C1A80000000"            , paddb(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FFECA"                        , paddd(mm1, mm2));
  TEST_INSTRUCTION("0FFE8C1A80000000"              , paddd(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FFE8C1A80000000"              , paddd(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FFECA"                      , paddd(xmm1, xmm2));
  TEST_INSTRUCTION("660FFE8C1A80000000"            , paddd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FFE8C1A80000000"            , paddd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FD4CA"                        , paddq(mm1, mm2));
  TEST_INSTRUCTION("0FD48C1A80000000"              , paddq(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FD48C1A80000000"              , paddq(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FD4CA"                      , paddq(xmm1, xmm2));
  TEST_INSTRUCTION("660FD48C1A80000000"            , paddq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FD48C1A80000000"            , paddq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FECCA"                        , paddsb(mm1, mm2));
  TEST_INSTRUCTION("0FEC8C1A80000000"              , paddsb(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FEC8C1A80000000"              , paddsb(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FECCA"                      , paddsb(xmm1, xmm2));
  TEST_INSTRUCTION("660FEC8C1A80000000"            , paddsb(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FEC8C1A80000000"            , paddsb(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FEDCA"                        , paddsw(mm1, mm2));
  TEST_INSTRUCTION("0FED8C1A80000000"              , paddsw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FED8C1A80000000"              , paddsw(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FEDCA"                      , paddsw(xmm1, xmm2));
  TEST_INSTRUCTION("660FED8C1A80000000"            , paddsw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FED8C1A80000000"            , paddsw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FDCCA"                        , paddusb(mm1, mm2));
  TEST_INSTRUCTION("0FDC8C1A80000000"              , paddusb(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FDC8C1A80000000"              , paddusb(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FDCCA"                      , paddusb(xmm1, xmm2));
  TEST_INSTRUCTION("660FDC8C1A80000000"            , paddusb(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FDC8C1A80000000"            , paddusb(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FDDCA"                        , paddusw(mm1, mm2));
  TEST_INSTRUCTION("0FDD8C1A80000000"              , paddusw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FDD8C1A80000000"              , paddusw(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FDDCA"                      , paddusw(xmm1, xmm2));
  TEST_INSTRUCTION("660FDD8C1A80000000"            , paddusw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FDD8C1A80000000"            , paddusw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FFDCA"                        , paddw(mm1, mm2));
  TEST_INSTRUCTION("0FFD8C1A80000000"              , paddw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FFD8C1A80000000"              , paddw(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FFDCA"                      , paddw(xmm1, xmm2));
  TEST_INSTRUCTION("660FFD8C1A80000000"            , paddw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FFD8C1A80000000"            , paddw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F3A0FCA01"                    , palignr(mm1, mm2, 1));
  TEST_INSTRUCTION("0F3A0F8C1A8000000001"          , palignr(mm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("0F3A0F8C1A8000000001"          , palignr(mm1, qword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F3A0FCA01"                  , palignr(xmm1, xmm2, 1));
  TEST_INSTRUCTION("660F3A0F8C1A8000000001"        , palignr(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F3A0F8C1A8000000001"        , palignr(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("0FDBCA"                        , pand(mm1, mm2));
  TEST_INSTRUCTION("0FDB8C1A80000000"              , pand(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FDB8C1A80000000"              , pand(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FDBCA"                      , pand(xmm1, xmm2));
  TEST_INSTRUCTION("660FDB8C1A80000000"            , pand(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FDB8C1A80000000"            , pand(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FDFCA"                        , pandn(mm1, mm2));
  TEST_INSTRUCTION("0FDF8C1A80000000"              , pandn(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FDF8C1A80000000"              , pandn(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FDFCA"                      , pandn(xmm1, xmm2));
  TEST_INSTRUCTION("660FDF8C1A80000000"            , pandn(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FDF8C1A80000000"            , pandn(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FE0CA"                        , pavgb(mm1, mm2));
  TEST_INSTRUCTION("0FE08C1A80000000"              , pavgb(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FE08C1A80000000"              , pavgb(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FE0CA"                      , pavgb(xmm1, xmm2));
  TEST_INSTRUCTION("660FE08C1A80000000"            , pavgb(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FE08C1A80000000"            , pavgb(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0FCABF"                      , pavgusb(mm1, mm2));
  TEST_INSTRUCTION("0F0F8C1A80000000BF"            , pavgusb(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0F8C1A80000000BF"            , pavgusb(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FE3CA"                        , pavgw(mm1, mm2));
  TEST_INSTRUCTION("0FE38C1A80000000"              , pavgw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FE38C1A80000000"              , pavgw(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FE3CA"                      , pavgw(xmm1, xmm2));
  TEST_INSTRUCTION("660FE38C1A80000000"            , pavgw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FE38C1A80000000"            , pavgw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3810CA"                    , pblendvb(xmm1, xmm2, xmm0));
  TEST_INSTRUCTION("660F38108C1A80000000"          , pblendvb(xmm1, ptr(edx, ebx, 0, 128), xmm0));
  TEST_INSTRUCTION("660F38108C1A80000000"          , pblendvb(xmm1, xmmword_ptr(edx, ebx, 0, 128), xmm0));
  TEST_INSTRUCTION("660F3A0ECA01"                  , pblendw(xmm1, xmm2, 1));
  TEST_INSTRUCTION("660F3A0E8C1A8000000001"        , pblendw(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F3A0E8C1A8000000001"        , pblendw(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F3A44CA01"                  , pclmulqdq(xmm1, xmm2, 1));
  TEST_INSTRUCTION("660F3A448C1A8000000001"        , pclmulqdq(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F3A448C1A8000000001"        , pclmulqdq(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("0F74CA"                        , pcmpeqb(mm1, mm2));
  TEST_INSTRUCTION("0F748C1A80000000"              , pcmpeqb(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F748C1A80000000"              , pcmpeqb(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F74CA"                      , pcmpeqb(xmm1, xmm2));
  TEST_INSTRUCTION("660F748C1A80000000"            , pcmpeqb(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F748C1A80000000"            , pcmpeqb(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F76CA"                        , pcmpeqd(mm1, mm2));
  TEST_INSTRUCTION("0F768C1A80000000"              , pcmpeqd(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F768C1A80000000"              , pcmpeqd(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F76CA"                      , pcmpeqd(xmm1, xmm2));
  TEST_INSTRUCTION("660F768C1A80000000"            , pcmpeqd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F768C1A80000000"            , pcmpeqd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3829CA"                    , pcmpeqq(xmm1, xmm2));
  TEST_INSTRUCTION("660F38298C1A80000000"          , pcmpeqq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38298C1A80000000"          , pcmpeqq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F75CA"                        , pcmpeqw(mm1, mm2));
  TEST_INSTRUCTION("0F758C1A80000000"              , pcmpeqw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F758C1A80000000"              , pcmpeqw(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F75CA"                      , pcmpeqw(xmm1, xmm2));
  TEST_INSTRUCTION("660F758C1A80000000"            , pcmpeqw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F758C1A80000000"            , pcmpeqw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3A61CA01"                  , pcmpestri(xmm1, xmm2, 1, ecx, eax, edx));
  TEST_INSTRUCTION("660F3A618C1A8000000001"        , pcmpestri(xmm1, ptr(edx, ebx, 0, 128), 1, ecx, eax, edx));
  TEST_INSTRUCTION("660F3A618C1A8000000001"        , pcmpestri(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1, ecx, eax, edx));
  TEST_INSTRUCTION("660F3A60CA01"                  , pcmpestrm(xmm1, xmm2, 1, xmm0, eax, edx));
  TEST_INSTRUCTION("660F3A608C1A8000000001"        , pcmpestrm(xmm1, ptr(edx, ebx, 0, 128), 1, xmm0, eax, edx));
  TEST_INSTRUCTION("660F3A608C1A8000000001"        , pcmpestrm(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1, xmm0, eax, edx));
  TEST_INSTRUCTION("0F64CA"                        , pcmpgtb(mm1, mm2));
  TEST_INSTRUCTION("0F648C1A80000000"              , pcmpgtb(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F648C1A80000000"              , pcmpgtb(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F64CA"                      , pcmpgtb(xmm1, xmm2));
  TEST_INSTRUCTION("660F648C1A80000000"            , pcmpgtb(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F648C1A80000000"            , pcmpgtb(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F66CA"                        , pcmpgtd(mm1, mm2));
  TEST_INSTRUCTION("0F668C1A80000000"              , pcmpgtd(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F668C1A80000000"              , pcmpgtd(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F66CA"                      , pcmpgtd(xmm1, xmm2));
  TEST_INSTRUCTION("660F668C1A80000000"            , pcmpgtd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F668C1A80000000"            , pcmpgtd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3837CA"                    , pcmpgtq(xmm1, xmm2));
  TEST_INSTRUCTION("660F38378C1A80000000"          , pcmpgtq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38378C1A80000000"          , pcmpgtq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F65CA"                        , pcmpgtw(mm1, mm2));
  TEST_INSTRUCTION("0F658C1A80000000"              , pcmpgtw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F658C1A80000000"              , pcmpgtw(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F65CA"                      , pcmpgtw(xmm1, xmm2));
  TEST_INSTRUCTION("660F658C1A80000000"            , pcmpgtw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F658C1A80000000"            , pcmpgtw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3A63CA01"                  , pcmpistri(xmm1, xmm2, 1, ecx));
  TEST_INSTRUCTION("660F3A638C1A8000000001"        , pcmpistri(xmm1, ptr(edx, ebx, 0, 128), 1, ecx));
  TEST_INSTRUCTION("660F3A638C1A8000000001"        , pcmpistri(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1, ecx));
  TEST_INSTRUCTION("660F3A62CA01"                  , pcmpistrm(xmm1, xmm2, 1, xmm0));
  TEST_INSTRUCTION("660F3A628C1A8000000001"        , pcmpistrm(xmm1, ptr(edx, ebx, 0, 128), 1, xmm0));
  TEST_INSTRUCTION("660F3A628C1A8000000001"        , pcmpistrm(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1, xmm0));
  TEST_INSTRUCTION("660F3A14D101"                  , pextrb(ecx, xmm2, 1));
  TEST_INSTRUCTION("660F3A149C118000000001"        , pextrb(ptr(ecx, edx, 0, 128), xmm3, 1));
  TEST_INSTRUCTION("660F3A149C118000000001"        , pextrb(byte_ptr(ecx, edx, 0, 128), xmm3, 1));
  TEST_INSTRUCTION("660F3A16D101"                  , pextrd(ecx, xmm2, 1));
  TEST_INSTRUCTION("660F3A169C118000000001"        , pextrd(ptr(ecx, edx, 0, 128), xmm3, 1));
  TEST_INSTRUCTION("660F3A169C118000000001"        , pextrd(dword_ptr(ecx, edx, 0, 128), xmm3, 1));
  TEST_INSTRUCTION("660FC5CA01"                    , pextrw(ecx, xmm2, 1));
  TEST_INSTRUCTION("0FC5CA01"                      , pextrw(ecx, mm2, 1));
  TEST_INSTRUCTION("660F3A159C118000000001"        , pextrw(ptr(ecx, edx, 0, 128), xmm3, 1));
  TEST_INSTRUCTION("660F3A159C118000000001"        , pextrw(word_ptr(ecx, edx, 0, 128), xmm3, 1));
  TEST_INSTRUCTION("0F0FCA1D"                      , pf2id(mm1, mm2));
  TEST_INSTRUCTION("0F0F8C1A800000001D"            , pf2id(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0F8C1A800000001D"            , pf2id(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0FCA1C"                      , pf2iw(mm1, mm2));
  TEST_INSTRUCTION("0F0F8C1A800000001C"            , pf2iw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0F8C1A800000001C"            , pf2iw(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0FCAAE"                      , pfacc(mm1, mm2));
  TEST_INSTRUCTION("0F0F8C1A80000000AE"            , pfacc(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0F8C1A80000000AE"            , pfacc(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0FCA9E"                      , pfadd(mm1, mm2));
  TEST_INSTRUCTION("0F0F8C1A800000009E"            , pfadd(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0F8C1A800000009E"            , pfadd(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0FCAB0"                      , pfcmpeq(mm1, mm2));
  TEST_INSTRUCTION("0F0F8C1A80000000B0"            , pfcmpeq(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0F8C1A80000000B0"            , pfcmpeq(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0FCA90"                      , pfcmpge(mm1, mm2));
  TEST_INSTRUCTION("0F0F8C1A8000000090"            , pfcmpge(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0F8C1A8000000090"            , pfcmpge(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0FCAA0"                      , pfcmpgt(mm1, mm2));
  TEST_INSTRUCTION("0F0F8C1A80000000A0"            , pfcmpgt(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0F8C1A80000000A0"            , pfcmpgt(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0FCAA4"                      , pfmax(mm1, mm2));
  TEST_INSTRUCTION("0F0F8C1A80000000A4"            , pfmax(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0F8C1A80000000A4"            , pfmax(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0FCA94"                      , pfmin(mm1, mm2));
  TEST_INSTRUCTION("0F0F8C1A8000000094"            , pfmin(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0F8C1A8000000094"            , pfmin(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0FCAB4"                      , pfmul(mm1, mm2));
  TEST_INSTRUCTION("0F0F8C1A80000000B4"            , pfmul(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0F8C1A80000000B4"            , pfmul(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0FCA8A"                      , pfnacc(mm1, mm2));
  TEST_INSTRUCTION("0F0F8C1A800000008A"            , pfnacc(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0F8C1A800000008A"            , pfnacc(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0FCA8E"                      , pfpnacc(mm1, mm2));
  TEST_INSTRUCTION("0F0F8C1A800000008E"            , pfpnacc(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0F8C1A800000008E"            , pfpnacc(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0FCA96"                      , pfrcp(mm1, mm2));
  TEST_INSTRUCTION("0F0F8C1A8000000096"            , pfrcp(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0F8C1A8000000096"            , pfrcp(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0FCAA6"                      , pfrcpit1(mm1, mm2));
  TEST_INSTRUCTION("0F0F8C1A80000000A6"            , pfrcpit1(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0F8C1A80000000A6"            , pfrcpit1(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0FCAB6"                      , pfrcpit2(mm1, mm2));
  TEST_INSTRUCTION("0F0F8C1A80000000B6"            , pfrcpit2(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0F8C1A80000000B6"            , pfrcpit2(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0FCA86"                      , pfrcpv(mm1, mm2));
  TEST_INSTRUCTION("0F0F8C1A8000000086"            , pfrcpv(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0F8C1A8000000086"            , pfrcpv(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0FCAA7"                      , pfrsqit1(mm1, mm2));
  TEST_INSTRUCTION("0F0F8C1A80000000A7"            , pfrsqit1(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0F8C1A80000000A7"            , pfrsqit1(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0FCA97"                      , pfrsqrt(mm1, mm2));
  TEST_INSTRUCTION("0F0F8C1A8000000097"            , pfrsqrt(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0F8C1A8000000097"            , pfrsqrt(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0FCA87"                      , pfrsqrtv(mm1, mm2));
  TEST_INSTRUCTION("0F0F8C1A8000000087"            , pfrsqrtv(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0F8C1A8000000087"            , pfrsqrtv(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0FCA9A"                      , pfsub(mm1, mm2));
  TEST_INSTRUCTION("0F0F8C1A800000009A"            , pfsub(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0F8C1A800000009A"            , pfsub(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0FCAAA"                      , pfsubr(mm1, mm2));
  TEST_INSTRUCTION("0F0F8C1A80000000AA"            , pfsubr(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0F8C1A80000000AA"            , pfsubr(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F3802CA"                      , phaddd(mm1, mm2));
  TEST_INSTRUCTION("0F38028C1A80000000"            , phaddd(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F38028C1A80000000"            , phaddd(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3802CA"                    , phaddd(xmm1, xmm2));
  TEST_INSTRUCTION("660F38028C1A80000000"          , phaddd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38028C1A80000000"          , phaddd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F3803CA"                      , phaddsw(mm1, mm2));
  TEST_INSTRUCTION("0F38038C1A80000000"            , phaddsw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F38038C1A80000000"            , phaddsw(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3803CA"                    , phaddsw(xmm1, xmm2));
  TEST_INSTRUCTION("660F38038C1A80000000"          , phaddsw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38038C1A80000000"          , phaddsw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F3801CA"                      , phaddw(mm1, mm2));
  TEST_INSTRUCTION("0F38018C1A80000000"            , phaddw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F38018C1A80000000"            , phaddw(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3801CA"                    , phaddw(xmm1, xmm2));
  TEST_INSTRUCTION("660F38018C1A80000000"          , phaddw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38018C1A80000000"          , phaddw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3841CA"                    , phminposuw(xmm1, xmm2));
  TEST_INSTRUCTION("660F38418C1A80000000"          , phminposuw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38418C1A80000000"          , phminposuw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F3806CA"                      , phsubd(mm1, mm2));
  TEST_INSTRUCTION("0F38068C1A80000000"            , phsubd(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F38068C1A80000000"            , phsubd(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3806CA"                    , phsubd(xmm1, xmm2));
  TEST_INSTRUCTION("660F38068C1A80000000"          , phsubd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38068C1A80000000"          , phsubd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F3807CA"                      , phsubsw(mm1, mm2));
  TEST_INSTRUCTION("0F38078C1A80000000"            , phsubsw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F38078C1A80000000"            , phsubsw(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3807CA"                    , phsubsw(xmm1, xmm2));
  TEST_INSTRUCTION("660F38078C1A80000000"          , phsubsw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38078C1A80000000"          , phsubsw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F3805CA"                      , phsubw(mm1, mm2));
  TEST_INSTRUCTION("0F38058C1A80000000"            , phsubw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F38058C1A80000000"            , phsubw(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3805CA"                    , phsubw(xmm1, xmm2));
  TEST_INSTRUCTION("660F38058C1A80000000"          , phsubw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38058C1A80000000"          , phsubw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0FCA0D"                      , pi2fd(mm1, mm2));
  TEST_INSTRUCTION("0F0F8C1A800000000D"            , pi2fd(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0F8C1A800000000D"            , pi2fd(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0FCA0C"                      , pi2fw(mm1, mm2));
  TEST_INSTRUCTION("0F0F8C1A800000000C"            , pi2fw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0F8C1A800000000C"            , pi2fw(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3A20CA01"                  , pinsrb(xmm1, edx, 1));
  TEST_INSTRUCTION("660F3A208C1A8000000001"        , pinsrb(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F3A208C1A8000000001"        , pinsrb(xmm1, byte_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F3A22CA01"                  , pinsrd(xmm1, edx, 1));
  TEST_INSTRUCTION("660F3A228C1A8000000001"        , pinsrd(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F3A228C1A8000000001"        , pinsrd(xmm1, dword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660FC4CA01"                    , pinsrw(xmm1, edx, 1));
  TEST_INSTRUCTION("660FC48C1A8000000001"          , pinsrw(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660FC48C1A8000000001"          , pinsrw(xmm1, word_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("0FC4CA01"                      , pinsrw(mm1, edx, 1));
  TEST_INSTRUCTION("0FC48C1A8000000001"            , pinsrw(mm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("0FC48C1A8000000001"            , pinsrw(mm1, word_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("0F3804CA"                      , pmaddubsw(mm1, mm2));
  TEST_INSTRUCTION("0F38048C1A80000000"            , pmaddubsw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F38048C1A80000000"            , pmaddubsw(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3804CA"                    , pmaddubsw(xmm1, xmm2));
  TEST_INSTRUCTION("660F38048C1A80000000"          , pmaddubsw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38048C1A80000000"          , pmaddubsw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FF5CA"                        , pmaddwd(mm1, mm2));
  TEST_INSTRUCTION("0FF58C1A80000000"              , pmaddwd(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FF58C1A80000000"              , pmaddwd(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FF5CA"                      , pmaddwd(xmm1, xmm2));
  TEST_INSTRUCTION("660FF58C1A80000000"            , pmaddwd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FF58C1A80000000"            , pmaddwd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F383CCA"                    , pmaxsb(xmm1, xmm2));
  TEST_INSTRUCTION("660F383C8C1A80000000"          , pmaxsb(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F383C8C1A80000000"          , pmaxsb(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F383DCA"                    , pmaxsd(xmm1, xmm2));
  TEST_INSTRUCTION("660F383D8C1A80000000"          , pmaxsd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F383D8C1A80000000"          , pmaxsd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FEECA"                        , pmaxsw(mm1, mm2));
  TEST_INSTRUCTION("0FEE8C1A80000000"              , pmaxsw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FEE8C1A80000000"              , pmaxsw(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FEECA"                      , pmaxsw(xmm1, xmm2));
  TEST_INSTRUCTION("660FEE8C1A80000000"            , pmaxsw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FEE8C1A80000000"            , pmaxsw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FDECA"                        , pmaxub(mm1, mm2));
  TEST_INSTRUCTION("0FDE8C1A80000000"              , pmaxub(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FDE8C1A80000000"              , pmaxub(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FDECA"                      , pmaxub(xmm1, xmm2));
  TEST_INSTRUCTION("660FDE8C1A80000000"            , pmaxub(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FDE8C1A80000000"            , pmaxub(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F383FCA"                    , pmaxud(xmm1, xmm2));
  TEST_INSTRUCTION("660F383F8C1A80000000"          , pmaxud(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F383F8C1A80000000"          , pmaxud(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F383ECA"                    , pmaxuw(xmm1, xmm2));
  TEST_INSTRUCTION("660F383E8C1A80000000"          , pmaxuw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F383E8C1A80000000"          , pmaxuw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3838CA"                    , pminsb(xmm1, xmm2));
  TEST_INSTRUCTION("660F38388C1A80000000"          , pminsb(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38388C1A80000000"          , pminsb(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3839CA"                    , pminsd(xmm1, xmm2));
  TEST_INSTRUCTION("660F38398C1A80000000"          , pminsd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38398C1A80000000"          , pminsd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FEACA"                        , pminsw(mm1, mm2));
  TEST_INSTRUCTION("0FEA8C1A80000000"              , pminsw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FEA8C1A80000000"              , pminsw(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FEACA"                      , pminsw(xmm1, xmm2));
  TEST_INSTRUCTION("660FEA8C1A80000000"            , pminsw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FEA8C1A80000000"            , pminsw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FDACA"                        , pminub(mm1, mm2));
  TEST_INSTRUCTION("0FDA8C1A80000000"              , pminub(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FDA8C1A80000000"              , pminub(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FDACA"                      , pminub(xmm1, xmm2));
  TEST_INSTRUCTION("660FDA8C1A80000000"            , pminub(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FDA8C1A80000000"            , pminub(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F383BCA"                    , pminud(xmm1, xmm2));
  TEST_INSTRUCTION("660F383B8C1A80000000"          , pminud(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F383B8C1A80000000"          , pminud(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F383ACA"                    , pminuw(xmm1, xmm2));
  TEST_INSTRUCTION("660F383A8C1A80000000"          , pminuw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F383A8C1A80000000"          , pminuw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FD7CA"                      , pmovmskb(ecx, xmm2));
  TEST_INSTRUCTION("0FD7CA"                        , pmovmskb(ecx, mm2));
  TEST_INSTRUCTION("660F3821CA"                    , pmovsxbd(xmm1, xmm2));
  TEST_INSTRUCTION("660F38218C1A80000000"          , pmovsxbd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38218C1A80000000"          , pmovsxbd(xmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3822CA"                    , pmovsxbq(xmm1, xmm2));
  TEST_INSTRUCTION("660F38228C1A80000000"          , pmovsxbq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38228C1A80000000"          , pmovsxbq(xmm1, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3820CA"                    , pmovsxbw(xmm1, xmm2));
  TEST_INSTRUCTION("660F38208C1A80000000"          , pmovsxbw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38208C1A80000000"          , pmovsxbw(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3825CA"                    , pmovsxdq(xmm1, xmm2));
  TEST_INSTRUCTION("660F38258C1A80000000"          , pmovsxdq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38258C1A80000000"          , pmovsxdq(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3823CA"                    , pmovsxwd(xmm1, xmm2));
  TEST_INSTRUCTION("660F38238C1A80000000"          , pmovsxwd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38238C1A80000000"          , pmovsxwd(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3824CA"                    , pmovsxwq(xmm1, xmm2));
  TEST_INSTRUCTION("660F38248C1A80000000"          , pmovsxwq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38248C1A80000000"          , pmovsxwq(xmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3831CA"                    , pmovzxbd(xmm1, xmm2));
  TEST_INSTRUCTION("660F38318C1A80000000"          , pmovzxbd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38318C1A80000000"          , pmovzxbd(xmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3832CA"                    , pmovzxbq(xmm1, xmm2));
  TEST_INSTRUCTION("660F38328C1A80000000"          , pmovzxbq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38328C1A80000000"          , pmovzxbq(xmm1, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3830CA"                    , pmovzxbw(xmm1, xmm2));
  TEST_INSTRUCTION("660F38308C1A80000000"          , pmovzxbw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38308C1A80000000"          , pmovzxbw(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3835CA"                    , pmovzxdq(xmm1, xmm2));
  TEST_INSTRUCTION("660F38358C1A80000000"          , pmovzxdq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38358C1A80000000"          , pmovzxdq(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3833CA"                    , pmovzxwd(xmm1, xmm2));
  TEST_INSTRUCTION("660F38338C1A80000000"          , pmovzxwd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38338C1A80000000"          , pmovzxwd(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3834CA"                    , pmovzxwq(xmm1, xmm2));
  TEST_INSTRUCTION("660F38348C1A80000000"          , pmovzxwq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38348C1A80000000"          , pmovzxwq(xmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3828CA"                    , pmuldq(xmm1, xmm2));
  TEST_INSTRUCTION("660F38288C1A80000000"          , pmuldq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38288C1A80000000"          , pmuldq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F380BCA"                      , pmulhrsw(mm1, mm2));
  TEST_INSTRUCTION("0F380B8C1A80000000"            , pmulhrsw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F380B8C1A80000000"            , pmulhrsw(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F380BCA"                    , pmulhrsw(xmm1, xmm2));
  TEST_INSTRUCTION("660F380B8C1A80000000"          , pmulhrsw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F380B8C1A80000000"          , pmulhrsw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0FCAB7"                      , pmulhrw(mm1, mm2));
  TEST_INSTRUCTION("0F0F8C1A80000000B7"            , pmulhrw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0F8C1A80000000B7"            , pmulhrw(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FE4CA"                        , pmulhuw(mm1, mm2));
  TEST_INSTRUCTION("0FE48C1A80000000"              , pmulhuw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FE48C1A80000000"              , pmulhuw(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FE4CA"                      , pmulhuw(xmm1, xmm2));
  TEST_INSTRUCTION("660FE48C1A80000000"            , pmulhuw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FE48C1A80000000"            , pmulhuw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FE5CA"                        , pmulhw(mm1, mm2));
  TEST_INSTRUCTION("0FE58C1A80000000"              , pmulhw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FE58C1A80000000"              , pmulhw(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FE5CA"                      , pmulhw(xmm1, xmm2));
  TEST_INSTRUCTION("660FE58C1A80000000"            , pmulhw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FE58C1A80000000"            , pmulhw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3840CA"                    , pmulld(xmm1, xmm2));
  TEST_INSTRUCTION("660F38408C1A80000000"          , pmulld(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38408C1A80000000"          , pmulld(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FD5CA"                        , pmullw(mm1, mm2));
  TEST_INSTRUCTION("0FD58C1A80000000"              , pmullw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FD58C1A80000000"              , pmullw(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FD5CA"                      , pmullw(xmm1, xmm2));
  TEST_INSTRUCTION("660FD58C1A80000000"            , pmullw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FD58C1A80000000"            , pmullw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FF4CA"                        , pmuludq(mm1, mm2));
  TEST_INSTRUCTION("0FF48C1A80000000"              , pmuludq(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FF48C1A80000000"              , pmuludq(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FF4CA"                      , pmuludq(xmm1, xmm2));
  TEST_INSTRUCTION("660FF48C1A80000000"            , pmuludq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FF48C1A80000000"            , pmuludq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FEBCA"                        , por(mm1, mm2));
  TEST_INSTRUCTION("0FEB8C1A80000000"              , por(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FEB8C1A80000000"              , por(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FEBCA"                      , por(xmm1, xmm2));
  TEST_INSTRUCTION("660FEB8C1A80000000"            , por(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FEB8C1A80000000"            , por(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FF6CA"                        , psadbw(mm1, mm2));
  TEST_INSTRUCTION("0FF68C1A80000000"              , psadbw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FF68C1A80000000"              , psadbw(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FF6CA"                      , psadbw(xmm1, xmm2));
  TEST_INSTRUCTION("660FF68C1A80000000"            , psadbw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FF68C1A80000000"            , psadbw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F3800CA"                      , pshufb(mm1, mm2));
  TEST_INSTRUCTION("0F38008C1A80000000"            , pshufb(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F38008C1A80000000"            , pshufb(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3800CA"                    , pshufb(xmm1, xmm2));
  TEST_INSTRUCTION("660F38008C1A80000000"          , pshufb(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38008C1A80000000"          , pshufb(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F70CA01"                    , pshufd(xmm1, xmm2, 1));
  TEST_INSTRUCTION("660F708C1A8000000001"          , pshufd(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F708C1A8000000001"          , pshufd(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("F30F70CA01"                    , pshufhw(xmm1, xmm2, 1));
  TEST_INSTRUCTION("F30F708C1A8000000001"          , pshufhw(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("F30F708C1A8000000001"          , pshufhw(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("F20F70CA01"                    , pshuflw(xmm1, xmm2, 1));
  TEST_INSTRUCTION("F20F708C1A8000000001"          , pshuflw(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("F20F708C1A8000000001"          , pshuflw(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("0F70CA01"                      , pshufw(mm1, mm2, 1));
  TEST_INSTRUCTION("0F708C1A8000000001"            , pshufw(mm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("0F708C1A8000000001"            , pshufw(mm1, qword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("0F3808CA"                      , psignb(mm1, mm2));
  TEST_INSTRUCTION("0F38088C1A80000000"            , psignb(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F38088C1A80000000"            , psignb(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3808CA"                    , psignb(xmm1, xmm2));
  TEST_INSTRUCTION("660F38088C1A80000000"          , psignb(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38088C1A80000000"          , psignb(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F380ACA"                      , psignd(mm1, mm2));
  TEST_INSTRUCTION("0F380A8C1A80000000"            , psignd(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F380A8C1A80000000"            , psignd(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F380ACA"                    , psignd(xmm1, xmm2));
  TEST_INSTRUCTION("660F380A8C1A80000000"          , psignd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F380A8C1A80000000"          , psignd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F3809CA"                      , psignw(mm1, mm2));
  TEST_INSTRUCTION("0F38098C1A80000000"            , psignw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F38098C1A80000000"            , psignw(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3809CA"                    , psignw(xmm1, xmm2));
  TEST_INSTRUCTION("660F38098C1A80000000"          , psignw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38098C1A80000000"          , psignw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FF2CA"                        , pslld(mm1, mm2));
  TEST_INSTRUCTION("0F72F101"                      , pslld(mm1, 1));
  TEST_INSTRUCTION("0FF28C1A80000000"              , pslld(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FF28C1A80000000"              , pslld(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FF2CA"                      , pslld(xmm1, xmm2));
  TEST_INSTRUCTION("660F72F101"                    , pslld(xmm1, 1));
  TEST_INSTRUCTION("660FF28C1A80000000"            , pslld(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FF28C1A80000000"            , pslld(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F73F901"                    , pslldq(xmm1, 1));
  TEST_INSTRUCTION("0FF3CA"                        , psllq(mm1, mm2));
  TEST_INSTRUCTION("0F73F101"                      , psllq(mm1, 1));
  TEST_INSTRUCTION("0FF38C1A80000000"              , psllq(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FF38C1A80000000"              , psllq(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FF3CA"                      , psllq(xmm1, xmm2));
  TEST_INSTRUCTION("660F73F101"                    , psllq(xmm1, 1));
  TEST_INSTRUCTION("660FF38C1A80000000"            , psllq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FF38C1A80000000"            , psllq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FF1CA"                        , psllw(mm1, mm2));
  TEST_INSTRUCTION("0F71F101"                      , psllw(mm1, 1));
  TEST_INSTRUCTION("0FF18C1A80000000"              , psllw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FF18C1A80000000"              , psllw(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FF1CA"                      , psllw(xmm1, xmm2));
  TEST_INSTRUCTION("660F71F101"                    , psllw(xmm1, 1));
  TEST_INSTRUCTION("660FF18C1A80000000"            , psllw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FF18C1A80000000"            , psllw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FE2CA"                        , psrad(mm1, mm2));
  TEST_INSTRUCTION("0F72E101"                      , psrad(mm1, 1));
  TEST_INSTRUCTION("0FE28C1A80000000"              , psrad(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FE28C1A80000000"              , psrad(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FE2CA"                      , psrad(xmm1, xmm2));
  TEST_INSTRUCTION("660F72E101"                    , psrad(xmm1, 1));
  TEST_INSTRUCTION("660FE28C1A80000000"            , psrad(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FE28C1A80000000"            , psrad(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FE1CA"                        , psraw(mm1, mm2));
  TEST_INSTRUCTION("0F71E101"                      , psraw(mm1, 1));
  TEST_INSTRUCTION("0FE18C1A80000000"              , psraw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FE18C1A80000000"              , psraw(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FE1CA"                      , psraw(xmm1, xmm2));
  TEST_INSTRUCTION("660F71E101"                    , psraw(xmm1, 1));
  TEST_INSTRUCTION("660FE18C1A80000000"            , psraw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FE18C1A80000000"            , psraw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FD2CA"                        , psrld(mm1, mm2));
  TEST_INSTRUCTION("0F72D101"                      , psrld(mm1, 1));
  TEST_INSTRUCTION("0FD28C1A80000000"              , psrld(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FD28C1A80000000"              , psrld(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FD2CA"                      , psrld(xmm1, xmm2));
  TEST_INSTRUCTION("660F72D101"                    , psrld(xmm1, 1));
  TEST_INSTRUCTION("660FD28C1A80000000"            , psrld(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FD28C1A80000000"            , psrld(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F73D901"                    , psrldq(xmm1, 1));
  TEST_INSTRUCTION("0FD3CA"                        , psrlq(mm1, mm2));
  TEST_INSTRUCTION("0F73D101"                      , psrlq(mm1, 1));
  TEST_INSTRUCTION("0FD38C1A80000000"              , psrlq(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FD38C1A80000000"              , psrlq(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FD3CA"                      , psrlq(xmm1, xmm2));
  TEST_INSTRUCTION("660F73D101"                    , psrlq(xmm1, 1));
  TEST_INSTRUCTION("660FD38C1A80000000"            , psrlq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FD38C1A80000000"            , psrlq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FD1CA"                        , psrlw(mm1, mm2));
  TEST_INSTRUCTION("0F71D101"                      , psrlw(mm1, 1));
  TEST_INSTRUCTION("0FD18C1A80000000"              , psrlw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FD18C1A80000000"              , psrlw(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FD1CA"                      , psrlw(xmm1, xmm2));
  TEST_INSTRUCTION("660F71D101"                    , psrlw(xmm1, 1));
  TEST_INSTRUCTION("660FD18C1A80000000"            , psrlw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FD18C1A80000000"            , psrlw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FF8CA"                        , psubb(mm1, mm2));
  TEST_INSTRUCTION("0FF88C1A80000000"              , psubb(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FF88C1A80000000"              , psubb(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FF8CA"                      , psubb(xmm1, xmm2));
  TEST_INSTRUCTION("660FF88C1A80000000"            , psubb(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FF88C1A80000000"            , psubb(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FFACA"                        , psubd(mm1, mm2));
  TEST_INSTRUCTION("0FFA8C1A80000000"              , psubd(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FFA8C1A80000000"              , psubd(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FFACA"                      , psubd(xmm1, xmm2));
  TEST_INSTRUCTION("660FFA8C1A80000000"            , psubd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FFA8C1A80000000"            , psubd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FFBCA"                        , psubq(mm1, mm2));
  TEST_INSTRUCTION("0FFB8C1A80000000"              , psubq(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FFB8C1A80000000"              , psubq(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FFBCA"                      , psubq(xmm1, xmm2));
  TEST_INSTRUCTION("660FFB8C1A80000000"            , psubq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FFB8C1A80000000"            , psubq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FE8CA"                        , psubsb(mm1, mm2));
  TEST_INSTRUCTION("0FE88C1A80000000"              , psubsb(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FE88C1A80000000"              , psubsb(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FE8CA"                      , psubsb(xmm1, xmm2));
  TEST_INSTRUCTION("660FE88C1A80000000"            , psubsb(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FE88C1A80000000"            , psubsb(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FE9CA"                        , psubsw(mm1, mm2));
  TEST_INSTRUCTION("0FE98C1A80000000"              , psubsw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FE98C1A80000000"              , psubsw(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FE9CA"                      , psubsw(xmm1, xmm2));
  TEST_INSTRUCTION("660FE98C1A80000000"            , psubsw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FE98C1A80000000"            , psubsw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FD8CA"                        , psubusb(mm1, mm2));
  TEST_INSTRUCTION("0FD88C1A80000000"              , psubusb(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FD88C1A80000000"              , psubusb(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FD8CA"                      , psubusb(xmm1, xmm2));
  TEST_INSTRUCTION("660FD88C1A80000000"            , psubusb(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FD88C1A80000000"            , psubusb(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FD9CA"                        , psubusw(mm1, mm2));
  TEST_INSTRUCTION("0FD98C1A80000000"              , psubusw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FD98C1A80000000"              , psubusw(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FD9CA"                      , psubusw(xmm1, xmm2));
  TEST_INSTRUCTION("660FD98C1A80000000"            , psubusw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FD98C1A80000000"            , psubusw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FF9CA"                        , psubw(mm1, mm2));
  TEST_INSTRUCTION("0FF98C1A80000000"              , psubw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FF98C1A80000000"              , psubw(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FF9CA"                      , psubw(xmm1, xmm2));
  TEST_INSTRUCTION("660FF98C1A80000000"            , psubw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FF98C1A80000000"            , psubw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0FCABB"                      , pswapd(mm1, mm2));
  TEST_INSTRUCTION("0F0F8C1A80000000BB"            , pswapd(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F0F8C1A80000000BB"            , pswapd(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3817CA"                    , ptest(xmm1, xmm2));
  TEST_INSTRUCTION("660F38178C1A80000000"          , ptest(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F38178C1A80000000"          , ptest(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F68CA"                        , punpckhbw(mm1, mm2));
  TEST_INSTRUCTION("0F688C1A80000000"              , punpckhbw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F688C1A80000000"              , punpckhbw(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F68CA"                      , punpckhbw(xmm1, xmm2));
  TEST_INSTRUCTION("660F688C1A80000000"            , punpckhbw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F688C1A80000000"            , punpckhbw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F6ACA"                        , punpckhdq(mm1, mm2));
  TEST_INSTRUCTION("0F6A8C1A80000000"              , punpckhdq(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F6A8C1A80000000"              , punpckhdq(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F6ACA"                      , punpckhdq(xmm1, xmm2));
  TEST_INSTRUCTION("660F6A8C1A80000000"            , punpckhdq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F6A8C1A80000000"            , punpckhdq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F6DCA"                      , punpckhqdq(xmm1, xmm2));
  TEST_INSTRUCTION("660F6D8C1A80000000"            , punpckhqdq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F6D8C1A80000000"            , punpckhqdq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F69CA"                        , punpckhwd(mm1, mm2));
  TEST_INSTRUCTION("0F698C1A80000000"              , punpckhwd(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F698C1A80000000"              , punpckhwd(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F69CA"                      , punpckhwd(xmm1, xmm2));
  TEST_INSTRUCTION("660F698C1A80000000"            , punpckhwd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F698C1A80000000"            , punpckhwd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F60CA"                        , punpcklbw(mm1, mm2));
  TEST_INSTRUCTION("0F608C1A80000000"              , punpcklbw(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F608C1A80000000"              , punpcklbw(mm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F60CA"                      , punpcklbw(xmm1, xmm2));
  TEST_INSTRUCTION("660F608C1A80000000"            , punpcklbw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F608C1A80000000"            , punpcklbw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F62CA"                        , punpckldq(mm1, mm2));
  TEST_INSTRUCTION("0F628C1A80000000"              , punpckldq(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F628C1A80000000"              , punpckldq(mm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F62CA"                      , punpckldq(xmm1, xmm2));
  TEST_INSTRUCTION("660F628C1A80000000"            , punpckldq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F628C1A80000000"            , punpckldq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F6CCA"                      , punpcklqdq(xmm1, xmm2));
  TEST_INSTRUCTION("660F6C8C1A80000000"            , punpcklqdq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F6C8C1A80000000"            , punpcklqdq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F61CA"                        , punpcklwd(mm1, mm2));
  TEST_INSTRUCTION("0F618C1A80000000"              , punpcklwd(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F618C1A80000000"              , punpcklwd(mm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F61CA"                      , punpcklwd(xmm1, xmm2));
  TEST_INSTRUCTION("660F618C1A80000000"            , punpcklwd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F618C1A80000000"            , punpcklwd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FEFCA"                        , pxor(mm1, mm2));
  TEST_INSTRUCTION("0FEF8C1A80000000"              , pxor(mm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0FEF8C1A80000000"              , pxor(mm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FEFCA"                      , pxor(xmm1, xmm2));
  TEST_INSTRUCTION("660FEF8C1A80000000"            , pxor(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660FEF8C1A80000000"            , pxor(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F53CA"                        , rcpps(xmm1, xmm2));
  TEST_INSTRUCTION("0F538C1A80000000"              , rcpps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F538C1A80000000"              , rcpps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F53CA"                      , rcpss(xmm1, xmm2));
  TEST_INSTRUCTION("F30F538C1A80000000"            , rcpss(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F538C1A80000000"            , rcpss(xmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F3A09CA01"                  , roundpd(xmm1, xmm2, 1));
  TEST_INSTRUCTION("660F3A098C1A8000000001"        , roundpd(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F3A098C1A8000000001"        , roundpd(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F3A08CA01"                  , roundps(xmm1, xmm2, 1));
  TEST_INSTRUCTION("660F3A088C1A8000000001"        , roundps(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F3A088C1A8000000001"        , roundps(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F3A0BCA01"                  , roundsd(xmm1, xmm2, 1));
  TEST_INSTRUCTION("660F3A0B8C1A8000000001"        , roundsd(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F3A0B8C1A8000000001"        , roundsd(xmm1, qword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F3A0ACA01"                  , roundss(xmm1, xmm2, 1));
  TEST_INSTRUCTION("660F3A0A8C1A8000000001"        , roundss(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F3A0A8C1A8000000001"        , roundss(xmm1, dword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("0F52CA"                        , rsqrtps(xmm1, xmm2));
  TEST_INSTRUCTION("0F528C1A80000000"              , rsqrtps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F528C1A80000000"              , rsqrtps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F52CA"                      , rsqrtss(xmm1, xmm2));
  TEST_INSTRUCTION("F30F528C1A80000000"            , rsqrtss(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F528C1A80000000"            , rsqrtss(xmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F38C9CA"                      , sha1msg1(xmm1, xmm2));
  TEST_INSTRUCTION("0F38C98C1A80000000"            , sha1msg1(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F38C98C1A80000000"            , sha1msg1(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F38CACA"                      , sha1msg2(xmm1, xmm2));
  TEST_INSTRUCTION("0F38CA8C1A80000000"            , sha1msg2(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F38CA8C1A80000000"            , sha1msg2(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F38C8CA"                      , sha1nexte(xmm1, xmm2));
  TEST_INSTRUCTION("0F38C88C1A80000000"            , sha1nexte(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F38C88C1A80000000"            , sha1nexte(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F3ACCCA01"                    , sha1rnds4(xmm1, xmm2, 1));
  TEST_INSTRUCTION("0F3ACC8C1A8000000001"          , sha1rnds4(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("0F3ACC8C1A8000000001"          , sha1rnds4(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("0F38CCCA"                      , sha256msg1(xmm1, xmm2));
  TEST_INSTRUCTION("0F38CC8C1A80000000"            , sha256msg1(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F38CC8C1A80000000"            , sha256msg1(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F38CDCA"                      , sha256msg2(xmm1, xmm2));
  TEST_INSTRUCTION("0F38CD8C1A80000000"            , sha256msg2(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F38CD8C1A80000000"            , sha256msg2(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F38CBCA"                      , sha256rnds2(xmm1, xmm2, xmm0));
  TEST_INSTRUCTION("0F38CB8C1A80000000"            , sha256rnds2(xmm1, ptr(edx, ebx, 0, 128), xmm0));
  TEST_INSTRUCTION("0F38CB8C1A80000000"            , sha256rnds2(xmm1, xmmword_ptr(edx, ebx, 0, 128), xmm0));
  TEST_INSTRUCTION("660FC6CA01"                    , shufpd(xmm1, xmm2, 1));
  TEST_INSTRUCTION("660FC68C1A8000000001"          , shufpd(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660FC68C1A8000000001"          , shufpd(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("0FC6CA01"                      , shufps(xmm1, xmm2, 1));
  TEST_INSTRUCTION("0FC68C1A8000000001"            , shufps(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("0FC68C1A8000000001"            , shufps(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("660F51CA"                      , sqrtpd(xmm1, xmm2));
  TEST_INSTRUCTION("660F518C1A80000000"            , sqrtpd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F518C1A80000000"            , sqrtpd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F51CA"                        , sqrtps(xmm1, xmm2));
  TEST_INSTRUCTION("0F518C1A80000000"              , sqrtps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F518C1A80000000"              , sqrtps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F51CA"                      , sqrtsd(xmm1, xmm2));
  TEST_INSTRUCTION("F20F518C1A80000000"            , sqrtsd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F518C1A80000000"            , sqrtsd(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F51CA"                      , sqrtss(xmm1, xmm2));
  TEST_INSTRUCTION("F30F518C1A80000000"            , sqrtss(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F518C1A80000000"            , sqrtss(xmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F5CCA"                      , subpd(xmm1, xmm2));
  TEST_INSTRUCTION("660F5C8C1A80000000"            , subpd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F5C8C1A80000000"            , subpd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F5CCA"                        , subps(xmm1, xmm2));
  TEST_INSTRUCTION("0F5C8C1A80000000"              , subps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F5C8C1A80000000"              , subps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F5CCA"                      , subsd(xmm1, xmm2));
  TEST_INSTRUCTION("F20F5C8C1A80000000"            , subsd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F20F5C8C1A80000000"            , subsd(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F5CCA"                      , subss(xmm1, xmm2));
  TEST_INSTRUCTION("F30F5C8C1A80000000"            , subss(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("F30F5C8C1A80000000"            , subss(xmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F2ECA"                      , ucomisd(xmm1, xmm2));
  TEST_INSTRUCTION("660F2E8C1A80000000"            , ucomisd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F2E8C1A80000000"            , ucomisd(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F2ECA"                        , ucomiss(xmm1, xmm2));
  TEST_INSTRUCTION("0F2E8C1A80000000"              , ucomiss(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F2E8C1A80000000"              , ucomiss(xmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F15CA"                      , unpckhpd(xmm1, xmm2));
  TEST_INSTRUCTION("660F158C1A80000000"            , unpckhpd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F158C1A80000000"            , unpckhpd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F15CA"                        , unpckhps(xmm1, xmm2));
  TEST_INSTRUCTION("0F158C1A80000000"              , unpckhps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F158C1A80000000"              , unpckhps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F14CA"                      , unpcklpd(xmm1, xmm2));
  TEST_INSTRUCTION("660F148C1A80000000"            , unpcklpd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F148C1A80000000"            , unpcklpd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F14CA"                        , unpcklps(xmm1, xmm2));
  TEST_INSTRUCTION("0F148C1A80000000"              , unpcklps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F148C1A80000000"              , unpcklps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F57CA"                      , xorpd(xmm1, xmm2));
  TEST_INSTRUCTION("660F578C1A80000000"            , xorpd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("660F578C1A80000000"            , xorpd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F57CA"                        , xorps(xmm1, xmm2));
  TEST_INSTRUCTION("0F578C1A80000000"              , xorps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("0F578C1A80000000"              , xorps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
}

static void ASMJIT_NOINLINE testX86AssemblerAVX(AssemblerTester<x86::Assembler>& tester) noexcept {
  using namespace x86;

  TEST_INSTRUCTION("C5E958CB"                      , vaddpd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9588C2B80000000"            , vaddpd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9588C2B80000000"            , vaddpd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED58CB"                      , vaddpd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5ED588C2B80000000"            , vaddpd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED588C2B80000000"            , vaddpd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E858CB"                      , vaddps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E8588C2B80000000"            , vaddps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E8588C2B80000000"            , vaddps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EC58CB"                      , vaddps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EC588C2B80000000"            , vaddps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EC588C2B80000000"            , vaddps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EB58CB"                      , vaddsd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5EB588C2B80000000"            , vaddsd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EB588C2B80000000"            , vaddsd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EA58CB"                      , vaddss(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5EA588C2B80000000"            , vaddss(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EA588C2B80000000"            , vaddss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9D0CB"                      , vaddsubpd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9D08C2B80000000"            , vaddsubpd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9D08C2B80000000"            , vaddsubpd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDD0CB"                      , vaddsubpd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDD08C2B80000000"            , vaddsubpd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDD08C2B80000000"            , vaddsubpd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EBD0CB"                      , vaddsubps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5EBD08C2B80000000"            , vaddsubps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EBD08C2B80000000"            , vaddsubps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EFD0CB"                      , vaddsubps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EFD08C2B80000000"            , vaddsubps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EFD08C2B80000000"            , vaddsubps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269DECB"                    , vaesdec(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269DE8C2B80000000"          , vaesdec(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269DE8C2B80000000"          , vaesdec(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DDECB"                    , vaesdec(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26DDE8C2B80000000"          , vaesdec(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DDE8C2B80000000"          , vaesdec(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269DFCB"                    , vaesdeclast(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269DF8C2B80000000"          , vaesdeclast(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269DF8C2B80000000"          , vaesdeclast(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DDFCB"                    , vaesdeclast(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26DDF8C2B80000000"          , vaesdeclast(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DDF8C2B80000000"          , vaesdeclast(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269DCCB"                    , vaesenc(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269DC8C2B80000000"          , vaesenc(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269DC8C2B80000000"          , vaesenc(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DDCCB"                    , vaesenc(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26DDC8C2B80000000"          , vaesenc(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DDC8C2B80000000"          , vaesenc(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269DDCB"                    , vaesenclast(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269DD8C2B80000000"          , vaesenclast(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269DD8C2B80000000"          , vaesenclast(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DDDCB"                    , vaesenclast(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26DDD8C2B80000000"          , vaesenclast(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DDD8C2B80000000"          , vaesenclast(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E279DBCA"                    , vaesimc(xmm1, xmm2));
  TEST_INSTRUCTION("C4E279DB8C1A80000000"          , vaesimc(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E279DB8C1A80000000"          , vaesimc(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E379DFCA01"                  , vaeskeygenassist(xmm1, xmm2, 1));
  TEST_INSTRUCTION("C4E379DF8C1A8000000001"        , vaeskeygenassist(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C4E379DF8C1A8000000001"        , vaeskeygenassist(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C5E955CB"                      , vandnpd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9558C2B80000000"            , vandnpd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9558C2B80000000"            , vandnpd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED55CB"                      , vandnpd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5ED558C2B80000000"            , vandnpd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED558C2B80000000"            , vandnpd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E855CB"                      , vandnps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E8558C2B80000000"            , vandnps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E8558C2B80000000"            , vandnps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EC55CB"                      , vandnps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EC558C2B80000000"            , vandnps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EC558C2B80000000"            , vandnps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E954CB"                      , vandpd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9548C2B80000000"            , vandpd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9548C2B80000000"            , vandpd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED54CB"                      , vandpd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5ED548C2B80000000"            , vandpd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED548C2B80000000"            , vandpd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E854CB"                      , vandps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E8548C2B80000000"            , vandps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E8548C2B80000000"            , vandps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EC54CB"                      , vandps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EC548C2B80000000"            , vandps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EC548C2B80000000"            , vandps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E3690DCB01"                  , vblendpd(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("C4E3690D8C2B8000000001"        , vblendpd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E3690D8C2B8000000001"        , vblendpd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E36D0DCB01"                  , vblendpd(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("C4E36D0D8C2B8000000001"        , vblendpd(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E36D0D8C2B8000000001"        , vblendpd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E3690CCB01"                  , vblendps(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("C4E3690C8C2B8000000001"        , vblendps(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E3690C8C2B8000000001"        , vblendps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E36D0CCB01"                  , vblendps(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("C4E36D0C8C2B8000000001"        , vblendps(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E36D0C8C2B8000000001"        , vblendps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E3694BCB40"                  , vblendvpd(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("C4E3694B8C2B8000000060"        , vblendvpd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3694B8C2B8000000060"        , vblendvpd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E36D4BCB40"                  , vblendvpd(ymm1, ymm2, ymm3, ymm4));
  TEST_INSTRUCTION("C4E36D4B8C2B8000000060"        , vblendvpd(ymm1, ymm2, ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("C4E36D4B8C2B8000000060"        , vblendvpd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("C4E3694ACB40"                  , vblendvps(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("C4E3694A8C2B8000000060"        , vblendvps(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3694A8C2B8000000060"        , vblendvps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E36D4ACB40"                  , vblendvps(ymm1, ymm2, ymm3, ymm4));
  TEST_INSTRUCTION("C4E36D4A8C2B8000000060"        , vblendvps(ymm1, ymm2, ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("C4E36D4A8C2B8000000060"        , vblendvps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("C4E27D1A8C1A80000000"          , vbroadcastf128(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D1A8C1A80000000"          , vbroadcastf128(ymm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D5A8C1A80000000"          , vbroadcasti128(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D5A8C1A80000000"          , vbroadcasti128(ymm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D19CA"                    , vbroadcastsd(ymm1, xmm2));
  TEST_INSTRUCTION("C4E27D198C1A80000000"          , vbroadcastsd(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D198C1A80000000"          , vbroadcastsd(ymm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27918CA"                    , vbroadcastss(xmm1, xmm2));
  TEST_INSTRUCTION("C4E279188C1A80000000"          , vbroadcastss(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E279188C1A80000000"          , vbroadcastss(xmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D18CA"                    , vbroadcastss(ymm1, xmm2));
  TEST_INSTRUCTION("C4E27D188C1A80000000"          , vbroadcastss(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D188C1A80000000"          , vbroadcastss(ymm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5E9C2CB01"                    , vcmppd(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("C5E9C28C2B8000000001"          , vcmppd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C5E9C28C2B8000000001"          , vcmppd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C5EDC2CB01"                    , vcmppd(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("C5EDC28C2B8000000001"          , vcmppd(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C5EDC28C2B8000000001"          , vcmppd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C5E8C2CB01"                    , vcmpps(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("C5E8C28C2B8000000001"          , vcmpps(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C5E8C28C2B8000000001"          , vcmpps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C5ECC2CB01"                    , vcmpps(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("C5ECC28C2B8000000001"          , vcmpps(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C5ECC28C2B8000000001"          , vcmpps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C5EBC2CB01"                    , vcmpsd(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("C5EBC28C2B8000000001"          , vcmpsd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C5EBC28C2B8000000001"          , vcmpsd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C5EAC2CB01"                    , vcmpss(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("C5EAC28C2B8000000001"          , vcmpss(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C5EAC28C2B8000000001"          , vcmpss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C5F92FCA"                      , vcomisd(xmm1, xmm2));
  TEST_INSTRUCTION("C5F92F8C1A80000000"            , vcomisd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F92F8C1A80000000"            , vcomisd(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F82FCA"                      , vcomiss(xmm1, xmm2));
  TEST_INSTRUCTION("C5F82F8C1A80000000"            , vcomiss(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F82F8C1A80000000"            , vcomiss(xmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FAE6CA"                      , vcvtdq2pd(xmm1, xmm2));
  TEST_INSTRUCTION("C5FAE68C1A80000000"            , vcvtdq2pd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FAE68C1A80000000"            , vcvtdq2pd(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FEE6CA"                      , vcvtdq2pd(ymm1, xmm2));
  TEST_INSTRUCTION("C5FEE68C1A80000000"            , vcvtdq2pd(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FEE68C1A80000000"            , vcvtdq2pd(ymm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F85BCA"                      , vcvtdq2ps(xmm1, xmm2));
  TEST_INSTRUCTION("C5F85B8C1A80000000"            , vcvtdq2ps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F85B8C1A80000000"            , vcvtdq2ps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FC5BCA"                      , vcvtdq2ps(ymm1, ymm2));
  TEST_INSTRUCTION("C5FC5B8C1A80000000"            , vcvtdq2ps(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FC5B8C1A80000000"            , vcvtdq2ps(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FBE6CA"                      , vcvtpd2dq(xmm1, xmm2));
  TEST_INSTRUCTION("C5FFE6CA"                      , vcvtpd2dq(xmm1, ymm2));
  TEST_INSTRUCTION("C5FBE68C1A80000000"            , vcvtpd2dq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FFE68C1A80000000"            , vcvtpd2dq(xmm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F95ACA"                      , vcvtpd2ps(xmm1, xmm2));
  TEST_INSTRUCTION("C5FD5ACA"                      , vcvtpd2ps(xmm1, ymm2));
  TEST_INSTRUCTION("C5F95A8C1A80000000"            , vcvtpd2ps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FD5A8C1A80000000"            , vcvtpd2ps(xmm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27913CA"                    , vcvtph2ps(xmm1, xmm2));
  TEST_INSTRUCTION("C4E279138C1A80000000"          , vcvtph2ps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E279138C1A80000000"          , vcvtph2ps(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D13CA"                    , vcvtph2ps(ymm1, xmm2));
  TEST_INSTRUCTION("C4E27D138C1A80000000"          , vcvtph2ps(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D138C1A80000000"          , vcvtph2ps(ymm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F95BCA"                      , vcvtps2dq(xmm1, xmm2));
  TEST_INSTRUCTION("C5F95B8C1A80000000"            , vcvtps2dq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F95B8C1A80000000"            , vcvtps2dq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FD5BCA"                      , vcvtps2dq(ymm1, ymm2));
  TEST_INSTRUCTION("C5FD5B8C1A80000000"            , vcvtps2dq(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FD5B8C1A80000000"            , vcvtps2dq(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F85ACA"                      , vcvtps2pd(xmm1, xmm2));
  TEST_INSTRUCTION("C5F85A8C1A80000000"            , vcvtps2pd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F85A8C1A80000000"            , vcvtps2pd(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FC5ACA"                      , vcvtps2pd(ymm1, xmm2));
  TEST_INSTRUCTION("C5FC5A8C1A80000000"            , vcvtps2pd(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FC5A8C1A80000000"            , vcvtps2pd(ymm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E3791DD101"                  , vcvtps2ph(xmm1, xmm2, 1));
  TEST_INSTRUCTION("C4E3791D9C118000000001"        , vcvtps2ph(ptr(ecx, edx, 0, 128), xmm3, 1));
  TEST_INSTRUCTION("C4E3791D9C118000000001"        , vcvtps2ph(qword_ptr(ecx, edx, 0, 128), xmm3, 1));
  TEST_INSTRUCTION("C4E37D1DD101"                  , vcvtps2ph(xmm1, ymm2, 1));
  TEST_INSTRUCTION("C4E37D1D9C118000000001"        , vcvtps2ph(ptr(ecx, edx, 0, 128), ymm3, 1));
  TEST_INSTRUCTION("C4E37D1D9C118000000001"        , vcvtps2ph(xmmword_ptr(ecx, edx, 0, 128), ymm3, 1));
  TEST_INSTRUCTION("C5FB2DCA"                      , vcvtsd2si(ecx, xmm2));
  TEST_INSTRUCTION("C5FB2D8C1A80000000"            , vcvtsd2si(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FB2D8C1A80000000"            , vcvtsd2si(ecx, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5EB5ACB"                      , vcvtsd2ss(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5EB5A8C2B80000000"            , vcvtsd2ss(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EB5A8C2B80000000"            , vcvtsd2ss(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EB2ACB"                      , vcvtsi2sd(xmm1, xmm2, ebx));
  TEST_INSTRUCTION("C5EB2A8C2B80000000"            , vcvtsi2sd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EB2A8C2B80000000"            , vcvtsi2sd(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EA2ACB"                      , vcvtsi2ss(xmm1, xmm2, ebx));
  TEST_INSTRUCTION("C5EA2A8C2B80000000"            , vcvtsi2ss(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EA2A8C2B80000000"            , vcvtsi2ss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EA5ACB"                      , vcvtss2sd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5EA5A8C2B80000000"            , vcvtss2sd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EA5A8C2B80000000"            , vcvtss2sd(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5FA2DCA"                      , vcvtss2si(ecx, xmm2));
  TEST_INSTRUCTION("C5FA2D8C1A80000000"            , vcvtss2si(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FA2D8C1A80000000"            , vcvtss2si(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F9E6CA"                      , vcvttpd2dq(xmm1, xmm2));
  TEST_INSTRUCTION("C5FDE6CA"                      , vcvttpd2dq(xmm1, ymm2));
  TEST_INSTRUCTION("C5F9E68C1A80000000"            , vcvttpd2dq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FDE68C1A80000000"            , vcvttpd2dq(xmm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FA5BCA"                      , vcvttps2dq(xmm1, xmm2));
  TEST_INSTRUCTION("C5FA5B8C1A80000000"            , vcvttps2dq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FA5B8C1A80000000"            , vcvttps2dq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FE5BCA"                      , vcvttps2dq(ymm1, ymm2));
  TEST_INSTRUCTION("C5FE5B8C1A80000000"            , vcvttps2dq(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FE5B8C1A80000000"            , vcvttps2dq(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FB2CCA"                      , vcvttsd2si(ecx, xmm2));
  TEST_INSTRUCTION("C5FB2C8C1A80000000"            , vcvttsd2si(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FB2C8C1A80000000"            , vcvttsd2si(ecx, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FA2CCA"                      , vcvttss2si(ecx, xmm2));
  TEST_INSTRUCTION("C5FA2C8C1A80000000"            , vcvttss2si(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FA2C8C1A80000000"            , vcvttss2si(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5E95ECB"                      , vdivpd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E95E8C2B80000000"            , vdivpd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E95E8C2B80000000"            , vdivpd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED5ECB"                      , vdivpd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5ED5E8C2B80000000"            , vdivpd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED5E8C2B80000000"            , vdivpd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E85ECB"                      , vdivps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E85E8C2B80000000"            , vdivps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E85E8C2B80000000"            , vdivps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EC5ECB"                      , vdivps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EC5E8C2B80000000"            , vdivps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EC5E8C2B80000000"            , vdivps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EB5ECB"                      , vdivsd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5EB5E8C2B80000000"            , vdivsd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EB5E8C2B80000000"            , vdivsd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EA5ECB"                      , vdivss(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5EA5E8C2B80000000"            , vdivss(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EA5E8C2B80000000"            , vdivss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E36941CB01"                  , vdppd(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("C4E369418C2B8000000001"        , vdppd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E369418C2B8000000001"        , vdppd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E36940CB01"                  , vdpps(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("C4E369408C2B8000000001"        , vdpps(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E369408C2B8000000001"        , vdpps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E36D40CB01"                  , vdpps(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("C4E36D408C2B8000000001"        , vdpps(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E36D408C2B8000000001"        , vdpps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E37D19D101"                  , vextractf128(xmm1, ymm2, 1));
  TEST_INSTRUCTION("C4E37D199C118000000001"        , vextractf128(ptr(ecx, edx, 0, 128), ymm3, 1));
  TEST_INSTRUCTION("C4E37D199C118000000001"        , vextractf128(xmmword_ptr(ecx, edx, 0, 128), ymm3, 1));
  TEST_INSTRUCTION("C4E37D39D101"                  , vextracti128(xmm1, ymm2, 1));
  TEST_INSTRUCTION("C4E37D399C118000000001"        , vextracti128(ptr(ecx, edx, 0, 128), ymm3, 1));
  TEST_INSTRUCTION("C4E37D399C118000000001"        , vextracti128(xmmword_ptr(ecx, edx, 0, 128), ymm3, 1));
  TEST_INSTRUCTION("C4E37917D101"                  , vextractps(ecx, xmm2, 1));
  TEST_INSTRUCTION("C4E379179C118000000001"        , vextractps(ptr(ecx, edx, 0, 128), xmm3, 1));
  TEST_INSTRUCTION("C4E379179C118000000001"        , vextractps(dword_ptr(ecx, edx, 0, 128), xmm3, 1));
  TEST_INSTRUCTION("C4E2D9928C1A80000000"          , vgatherdpd(xmm1, ptr(edx, xmm3, 0, 128), xmm4));
  TEST_INSTRUCTION("C4E2DD928C1A80000000"          , vgatherdpd(ymm1, ptr(edx, xmm3, 0, 128), ymm4));
  TEST_INSTRUCTION("C4E259928C1A80000000"          , vgatherdps(xmm1, ptr(edx, xmm3, 0, 128), xmm4));
  TEST_INSTRUCTION("C4E25D928C1A80000000"          , vgatherdps(ymm1, ptr(edx, ymm3, 0, 128), ymm4));
  TEST_INSTRUCTION("C4E2D9938C1A80000000"          , vgatherqpd(xmm1, ptr(edx, xmm3, 0, 128), xmm4));
  TEST_INSTRUCTION("C4E2DD938C1A80000000"          , vgatherqpd(ymm1, ptr(edx, ymm3, 0, 128), ymm4));
  TEST_INSTRUCTION("C4E259938C1A80000000"          , vgatherqps(xmm1, ptr(edx, xmm3, 0, 128), xmm4));
  TEST_INSTRUCTION("C4E25D938C1A80000000"          , vgatherqps(xmm1, ptr(edx, ymm3, 0, 128), xmm4));
  TEST_INSTRUCTION("C4E3E9CFCB01"                  , vgf2p8affineinvqb(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("C4E3E9CF8C2B8000000001"        , vgf2p8affineinvqb(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E3E9CF8C2B8000000001"        , vgf2p8affineinvqb(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E3EDCFCB01"                  , vgf2p8affineinvqb(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("C4E3EDCF8C2B8000000001"        , vgf2p8affineinvqb(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E3EDCF8C2B8000000001"        , vgf2p8affineinvqb(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E3E9CECB01"                  , vgf2p8affineqb(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("C4E3E9CE8C2B8000000001"        , vgf2p8affineqb(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E3E9CE8C2B8000000001"        , vgf2p8affineqb(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E3EDCECB01"                  , vgf2p8affineqb(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("C4E3EDCE8C2B8000000001"        , vgf2p8affineqb(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E3EDCE8C2B8000000001"        , vgf2p8affineqb(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E269CFCB"                    , vgf2p8mulb(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269CF8C2B80000000"          , vgf2p8mulb(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269CF8C2B80000000"          , vgf2p8mulb(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DCFCB"                    , vgf2p8mulb(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26DCF8C2B80000000"          , vgf2p8mulb(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DCF8C2B80000000"          , vgf2p8mulb(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E97CCB"                      , vhaddpd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E97C8C2B80000000"            , vhaddpd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E97C8C2B80000000"            , vhaddpd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED7CCB"                      , vhaddpd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5ED7C8C2B80000000"            , vhaddpd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED7C8C2B80000000"            , vhaddpd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EB7CCB"                      , vhaddps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5EB7C8C2B80000000"            , vhaddps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EB7C8C2B80000000"            , vhaddps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EF7CCB"                      , vhaddps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EF7C8C2B80000000"            , vhaddps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EF7C8C2B80000000"            , vhaddps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E97DCB"                      , vhsubpd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E97D8C2B80000000"            , vhsubpd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E97D8C2B80000000"            , vhsubpd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED7DCB"                      , vhsubpd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5ED7D8C2B80000000"            , vhsubpd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED7D8C2B80000000"            , vhsubpd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EB7DCB"                      , vhsubps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5EB7D8C2B80000000"            , vhsubps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EB7D8C2B80000000"            , vhsubps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EF7DCB"                      , vhsubps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EF7D8C2B80000000"            , vhsubps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EF7D8C2B80000000"            , vhsubps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E36D18CB01"                  , vinsertf128(ymm1, ymm2, xmm3, 1));
  TEST_INSTRUCTION("C4E36D188C2B8000000001"        , vinsertf128(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E36D188C2B8000000001"        , vinsertf128(ymm1, ymm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E36D38CB01"                  , vinserti128(ymm1, ymm2, xmm3, 1));
  TEST_INSTRUCTION("C4E36D388C2B8000000001"        , vinserti128(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E36D388C2B8000000001"        , vinserti128(ymm1, ymm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E36921CB01"                  , vinsertps(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("C4E369218C2B8000000001"        , vinsertps(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E369218C2B8000000001"        , vinsertps(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C5FBF08C1A80000000"            , vlddqu(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FBF08C1A80000000"            , vlddqu(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FFF08C1A80000000"            , vlddqu(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FFF08C1A80000000"            , vlddqu(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F8AE941180000000"            , vldmxcsr(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("C5F8AE941180000000"            , vldmxcsr(dword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("C5F9F7CA"                      , vmaskmovdqu(xmm1, xmm2, ptr(edi)));
  TEST_INSTRUCTION("C5F9F7CA"                      , vmaskmovdqu(xmm1, xmm2, xmmword_ptr(edi)));
  TEST_INSTRUCTION("C4E2612FA41180000000"          , vmaskmovpd(ptr(ecx, edx, 0, 128), xmm3, xmm4));
  TEST_INSTRUCTION("C4E2612FA41180000000"          , vmaskmovpd(xmmword_ptr(ecx, edx, 0, 128), xmm3, xmm4));
  TEST_INSTRUCTION("C4E2652FA41180000000"          , vmaskmovpd(ptr(ecx, edx, 0, 128), ymm3, ymm4));
  TEST_INSTRUCTION("C4E2652FA41180000000"          , vmaskmovpd(ymmword_ptr(ecx, edx, 0, 128), ymm3, ymm4));
  TEST_INSTRUCTION("C4E2692D8C2B80000000"          , vmaskmovpd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2692D8C2B80000000"          , vmaskmovpd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D2D8C2B80000000"          , vmaskmovpd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D2D8C2B80000000"          , vmaskmovpd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2612EA41180000000"          , vmaskmovps(ptr(ecx, edx, 0, 128), xmm3, xmm4));
  TEST_INSTRUCTION("C4E2612EA41180000000"          , vmaskmovps(xmmword_ptr(ecx, edx, 0, 128), xmm3, xmm4));
  TEST_INSTRUCTION("C4E2652EA41180000000"          , vmaskmovps(ptr(ecx, edx, 0, 128), ymm3, ymm4));
  TEST_INSTRUCTION("C4E2652EA41180000000"          , vmaskmovps(ymmword_ptr(ecx, edx, 0, 128), ymm3, ymm4));
  TEST_INSTRUCTION("C4E2692C8C2B80000000"          , vmaskmovps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2692C8C2B80000000"          , vmaskmovps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D2C8C2B80000000"          , vmaskmovps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D2C8C2B80000000"          , vmaskmovps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E95FCB"                      , vmaxpd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E95F8C2B80000000"            , vmaxpd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E95F8C2B80000000"            , vmaxpd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED5FCB"                      , vmaxpd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5ED5F8C2B80000000"            , vmaxpd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED5F8C2B80000000"            , vmaxpd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E85FCB"                      , vmaxps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E85F8C2B80000000"            , vmaxps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E85F8C2B80000000"            , vmaxps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EC5FCB"                      , vmaxps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EC5F8C2B80000000"            , vmaxps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EC5F8C2B80000000"            , vmaxps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EB5FCB"                      , vmaxsd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5EB5F8C2B80000000"            , vmaxsd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EB5F8C2B80000000"            , vmaxsd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EA5FCB"                      , vmaxss(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5EA5F8C2B80000000"            , vmaxss(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EA5F8C2B80000000"            , vmaxss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E95DCB"                      , vminpd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E95D8C2B80000000"            , vminpd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E95D8C2B80000000"            , vminpd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED5DCB"                      , vminpd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5ED5D8C2B80000000"            , vminpd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED5D8C2B80000000"            , vminpd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E85DCB"                      , vminps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E85D8C2B80000000"            , vminps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E85D8C2B80000000"            , vminps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EC5DCB"                      , vminps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EC5D8C2B80000000"            , vminps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EC5D8C2B80000000"            , vminps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EB5DCB"                      , vminsd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5EB5D8C2B80000000"            , vminsd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EB5D8C2B80000000"            , vminsd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EA5DCB"                      , vminss(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5EA5D8C2B80000000"            , vminss(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EA5D8C2B80000000"            , vminss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5F928CA"                      , vmovapd(xmm1, xmm2));
  TEST_INSTRUCTION("C5F9288C1A80000000"            , vmovapd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F9288C1A80000000"            , vmovapd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F9299C1180000000"            , vmovapd(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5F9299C1180000000"            , vmovapd(xmmword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5FD28CA"                      , vmovapd(ymm1, ymm2));
  TEST_INSTRUCTION("C5FD288C1A80000000"            , vmovapd(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FD288C1A80000000"            , vmovapd(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FD299C1180000000"            , vmovapd(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("C5FD299C1180000000"            , vmovapd(ymmword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("C5F828CA"                      , vmovaps(xmm1, xmm2));
  TEST_INSTRUCTION("C5F8288C1A80000000"            , vmovaps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F8288C1A80000000"            , vmovaps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F8299C1180000000"            , vmovaps(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5F8299C1180000000"            , vmovaps(xmmword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5FC28CA"                      , vmovaps(ymm1, ymm2));
  TEST_INSTRUCTION("C5FC288C1A80000000"            , vmovaps(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FC288C1A80000000"            , vmovaps(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FC299C1180000000"            , vmovaps(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("C5FC299C1180000000"            , vmovaps(ymmword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("C5F97ED1"                      , vmovd(ecx, xmm2));
  TEST_INSTRUCTION("C5F97E9C1180000000"            , vmovd(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5F97E9C1180000000"            , vmovd(dword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5F96ECA"                      , vmovd(xmm1, edx));
  TEST_INSTRUCTION("C5F96E8C1A80000000"            , vmovd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F96E8C1A80000000"            , vmovd(xmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FB12CA"                      , vmovddup(xmm1, xmm2));
  TEST_INSTRUCTION("C5FB128C1A80000000"            , vmovddup(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FB128C1A80000000"            , vmovddup(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FF12CA"                      , vmovddup(ymm1, ymm2));
  TEST_INSTRUCTION("C5FF128C1A80000000"            , vmovddup(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FF128C1A80000000"            , vmovddup(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F96FCA"                      , vmovdqa(xmm1, xmm2));
  TEST_INSTRUCTION("C5F96F8C1A80000000"            , vmovdqa(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F96F8C1A80000000"            , vmovdqa(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F97F9C1180000000"            , vmovdqa(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5F97F9C1180000000"            , vmovdqa(xmmword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5FD6FCA"                      , vmovdqa(ymm1, ymm2));
  TEST_INSTRUCTION("C5FD6F8C1A80000000"            , vmovdqa(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FD6F8C1A80000000"            , vmovdqa(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FD7F9C1180000000"            , vmovdqa(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("C5FD7F9C1180000000"            , vmovdqa(ymmword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("C5FA6FCA"                      , vmovdqu(xmm1, xmm2));
  TEST_INSTRUCTION("C5FA6F8C1A80000000"            , vmovdqu(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FA6F8C1A80000000"            , vmovdqu(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FA7F9C1180000000"            , vmovdqu(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5FA7F9C1180000000"            , vmovdqu(xmmword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5FE6FCA"                      , vmovdqu(ymm1, ymm2));
  TEST_INSTRUCTION("C5FE6F8C1A80000000"            , vmovdqu(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FE6F8C1A80000000"            , vmovdqu(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FE7F9C1180000000"            , vmovdqu(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("C5FE7F9C1180000000"            , vmovdqu(ymmword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("C5E812CB"                      , vmovhlps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5F9179C1180000000"            , vmovhpd(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5F9179C1180000000"            , vmovhpd(qword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5E9168C2B80000000"            , vmovhpd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9168C2B80000000"            , vmovhpd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5F8179C1180000000"            , vmovhps(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5F8179C1180000000"            , vmovhps(qword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5E8168C2B80000000"            , vmovhps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E8168C2B80000000"            , vmovhps(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E816CB"                      , vmovlhps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5F9139C1180000000"            , vmovlpd(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5F9139C1180000000"            , vmovlpd(qword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5E9128C2B80000000"            , vmovlpd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9128C2B80000000"            , vmovlpd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5F8139C1180000000"            , vmovlps(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5F8139C1180000000"            , vmovlps(qword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5E8128C2B80000000"            , vmovlps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E8128C2B80000000"            , vmovlps(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5F950CA"                      , vmovmskpd(ecx, xmm2));
  TEST_INSTRUCTION("C5FD50CA"                      , vmovmskpd(ecx, ymm2));
  TEST_INSTRUCTION("C5F850CA"                      , vmovmskps(ecx, xmm2));
  TEST_INSTRUCTION("C5FC50CA"                      , vmovmskps(ecx, ymm2));
  TEST_INSTRUCTION("C5F9E79C1180000000"            , vmovntdq(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5F9E79C1180000000"            , vmovntdq(xmmword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5FDE79C1180000000"            , vmovntdq(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("C5FDE79C1180000000"            , vmovntdq(ymmword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("C4E2792A8C1A80000000"          , vmovntdqa(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E2792A8C1A80000000"          , vmovntdqa(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D2A8C1A80000000"          , vmovntdqa(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D2A8C1A80000000"          , vmovntdqa(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F92B9C1180000000"            , vmovntpd(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5F92B9C1180000000"            , vmovntpd(xmmword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5FD2B9C1180000000"            , vmovntpd(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("C5FD2B9C1180000000"            , vmovntpd(ymmword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("C5F82B9C1180000000"            , vmovntps(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5F82B9C1180000000"            , vmovntps(xmmword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5FC2B9C1180000000"            , vmovntps(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("C5FC2B9C1180000000"            , vmovntps(ymmword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("C5FA7ECA"                      , vmovq(xmm1, xmm2));
  TEST_INSTRUCTION("C5FA7E8C1A80000000"            , vmovq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FA7E8C1A80000000"            , vmovq(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F9D69C1180000000"            , vmovq(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5F9D69C1180000000"            , vmovq(qword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5FB119C1180000000"            , vmovsd(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5FB119C1180000000"            , vmovsd(qword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5FB108C1A80000000"            , vmovsd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FB108C1A80000000"            , vmovsd(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5EB10CB"                      , vmovsd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5FA16CA"                      , vmovshdup(xmm1, xmm2));
  TEST_INSTRUCTION("C5FA168C1A80000000"            , vmovshdup(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FA168C1A80000000"            , vmovshdup(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FE16CA"                      , vmovshdup(ymm1, ymm2));
  TEST_INSTRUCTION("C5FE168C1A80000000"            , vmovshdup(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FE168C1A80000000"            , vmovshdup(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FA12CA"                      , vmovsldup(xmm1, xmm2));
  TEST_INSTRUCTION("C5FA128C1A80000000"            , vmovsldup(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FA128C1A80000000"            , vmovsldup(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FE12CA"                      , vmovsldup(ymm1, ymm2));
  TEST_INSTRUCTION("C5FE128C1A80000000"            , vmovsldup(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FE128C1A80000000"            , vmovsldup(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FA119C1180000000"            , vmovss(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5FA119C1180000000"            , vmovss(dword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5FA108C1A80000000"            , vmovss(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FA108C1A80000000"            , vmovss(xmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5EA10CB"                      , vmovss(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5F910CA"                      , vmovupd(xmm1, xmm2));
  TEST_INSTRUCTION("C5F9108C1A80000000"            , vmovupd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F9108C1A80000000"            , vmovupd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F9119C1180000000"            , vmovupd(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5F9119C1180000000"            , vmovupd(xmmword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5FD10CA"                      , vmovupd(ymm1, ymm2));
  TEST_INSTRUCTION("C5FD108C1A80000000"            , vmovupd(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FD108C1A80000000"            , vmovupd(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FD119C1180000000"            , vmovupd(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("C5FD119C1180000000"            , vmovupd(ymmword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("C5F810CA"                      , vmovups(xmm1, xmm2));
  TEST_INSTRUCTION("C5F8108C1A80000000"            , vmovups(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F8108C1A80000000"            , vmovups(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F8119C1180000000"            , vmovups(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5F8119C1180000000"            , vmovups(xmmword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("C5FC10CA"                      , vmovups(ymm1, ymm2));
  TEST_INSTRUCTION("C5FC108C1A80000000"            , vmovups(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FC108C1A80000000"            , vmovups(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FC119C1180000000"            , vmovups(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("C5FC119C1180000000"            , vmovups(ymmword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("C4E36942CB01"                  , vmpsadbw(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("C4E369428C2B8000000001"        , vmpsadbw(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E369428C2B8000000001"        , vmpsadbw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E36D42CB01"                  , vmpsadbw(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("C4E36D428C2B8000000001"        , vmpsadbw(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E36D428C2B8000000001"        , vmpsadbw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C5E959CB"                      , vmulpd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9598C2B80000000"            , vmulpd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9598C2B80000000"            , vmulpd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED59CB"                      , vmulpd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5ED598C2B80000000"            , vmulpd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED598C2B80000000"            , vmulpd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E859CB"                      , vmulps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E8598C2B80000000"            , vmulps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E8598C2B80000000"            , vmulps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EC59CB"                      , vmulps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EC598C2B80000000"            , vmulps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EC598C2B80000000"            , vmulps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EB59CB"                      , vmulsd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5EB598C2B80000000"            , vmulsd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EB598C2B80000000"            , vmulsd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EA59CB"                      , vmulss(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5EA598C2B80000000"            , vmulss(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EA598C2B80000000"            , vmulss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E956CB"                      , vorpd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9568C2B80000000"            , vorpd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9568C2B80000000"            , vorpd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED56CB"                      , vorpd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5ED568C2B80000000"            , vorpd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED568C2B80000000"            , vorpd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E856CB"                      , vorps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E8568C2B80000000"            , vorps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E8568C2B80000000"            , vorps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EC56CB"                      , vorps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EC568C2B80000000"            , vorps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EC568C2B80000000"            , vorps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2791CCA"                    , vpabsb(xmm1, xmm2));
  TEST_INSTRUCTION("C4E2791C8C1A80000000"          , vpabsb(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E2791C8C1A80000000"          , vpabsb(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D1CCA"                    , vpabsb(ymm1, ymm2));
  TEST_INSTRUCTION("C4E27D1C8C1A80000000"          , vpabsb(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D1C8C1A80000000"          , vpabsb(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E2791ECA"                    , vpabsd(xmm1, xmm2));
  TEST_INSTRUCTION("C4E2791E8C1A80000000"          , vpabsd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E2791E8C1A80000000"          , vpabsd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D1ECA"                    , vpabsd(ymm1, ymm2));
  TEST_INSTRUCTION("C4E27D1E8C1A80000000"          , vpabsd(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D1E8C1A80000000"          , vpabsd(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E2791DCA"                    , vpabsw(xmm1, xmm2));
  TEST_INSTRUCTION("C4E2791D8C1A80000000"          , vpabsw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E2791D8C1A80000000"          , vpabsw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D1DCA"                    , vpabsw(ymm1, ymm2));
  TEST_INSTRUCTION("C4E27D1D8C1A80000000"          , vpabsw(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D1D8C1A80000000"          , vpabsw(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5E96BCB"                      , vpackssdw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E96B8C2B80000000"            , vpackssdw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E96B8C2B80000000"            , vpackssdw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED6BCB"                      , vpackssdw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5ED6B8C2B80000000"            , vpackssdw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED6B8C2B80000000"            , vpackssdw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E963CB"                      , vpacksswb(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9638C2B80000000"            , vpacksswb(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9638C2B80000000"            , vpacksswb(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED63CB"                      , vpacksswb(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5ED638C2B80000000"            , vpacksswb(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED638C2B80000000"            , vpacksswb(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2692BCB"                    , vpackusdw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2692B8C2B80000000"          , vpackusdw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2692B8C2B80000000"          , vpackusdw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D2BCB"                    , vpackusdw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D2B8C2B80000000"          , vpackusdw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D2B8C2B80000000"          , vpackusdw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E967CB"                      , vpackuswb(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9678C2B80000000"            , vpackuswb(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9678C2B80000000"            , vpackuswb(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED67CB"                      , vpackuswb(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5ED678C2B80000000"            , vpackuswb(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED678C2B80000000"            , vpackuswb(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9FCCB"                      , vpaddb(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9FC8C2B80000000"            , vpaddb(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9FC8C2B80000000"            , vpaddb(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDFCCB"                      , vpaddb(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDFC8C2B80000000"            , vpaddb(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDFC8C2B80000000"            , vpaddb(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9FECB"                      , vpaddd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9FE8C2B80000000"            , vpaddd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9FE8C2B80000000"            , vpaddd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDFECB"                      , vpaddd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDFE8C2B80000000"            , vpaddd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDFE8C2B80000000"            , vpaddd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9D4CB"                      , vpaddq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9D48C2B80000000"            , vpaddq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9D48C2B80000000"            , vpaddq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDD4CB"                      , vpaddq(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDD48C2B80000000"            , vpaddq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDD48C2B80000000"            , vpaddq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9ECCB"                      , vpaddsb(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9EC8C2B80000000"            , vpaddsb(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9EC8C2B80000000"            , vpaddsb(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDECCB"                      , vpaddsb(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDEC8C2B80000000"            , vpaddsb(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDEC8C2B80000000"            , vpaddsb(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9EDCB"                      , vpaddsw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9ED8C2B80000000"            , vpaddsw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9ED8C2B80000000"            , vpaddsw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDEDCB"                      , vpaddsw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDED8C2B80000000"            , vpaddsw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDED8C2B80000000"            , vpaddsw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9DCCB"                      , vpaddusb(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9DC8C2B80000000"            , vpaddusb(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9DC8C2B80000000"            , vpaddusb(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDDCCB"                      , vpaddusb(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDDC8C2B80000000"            , vpaddusb(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDDC8C2B80000000"            , vpaddusb(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9DDCB"                      , vpaddusw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9DD8C2B80000000"            , vpaddusw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9DD8C2B80000000"            , vpaddusw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDDDCB"                      , vpaddusw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDDD8C2B80000000"            , vpaddusw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDDD8C2B80000000"            , vpaddusw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9FDCB"                      , vpaddw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9FD8C2B80000000"            , vpaddw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9FD8C2B80000000"            , vpaddw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDFDCB"                      , vpaddw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDFD8C2B80000000"            , vpaddw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDFD8C2B80000000"            , vpaddw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E3690FCB01"                  , vpalignr(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("C4E3690F8C2B8000000001"        , vpalignr(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E3690F8C2B8000000001"        , vpalignr(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E36D0FCB01"                  , vpalignr(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("C4E36D0F8C2B8000000001"        , vpalignr(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E36D0F8C2B8000000001"        , vpalignr(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C5E9DBCB"                      , vpand(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9DB8C2B80000000"            , vpand(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9DB8C2B80000000"            , vpand(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDDBCB"                      , vpand(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDDB8C2B80000000"            , vpand(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDDB8C2B80000000"            , vpand(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9DFCB"                      , vpandn(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9DF8C2B80000000"            , vpandn(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9DF8C2B80000000"            , vpandn(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDDFCB"                      , vpandn(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDDF8C2B80000000"            , vpandn(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDDF8C2B80000000"            , vpandn(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9E0CB"                      , vpavgb(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9E08C2B80000000"            , vpavgb(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9E08C2B80000000"            , vpavgb(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDE0CB"                      , vpavgb(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDE08C2B80000000"            , vpavgb(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDE08C2B80000000"            , vpavgb(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9E3CB"                      , vpavgw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9E38C2B80000000"            , vpavgw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9E38C2B80000000"            , vpavgw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDE3CB"                      , vpavgw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDE38C2B80000000"            , vpavgw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDE38C2B80000000"            , vpavgw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E36902CB01"                  , vpblendd(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("C4E369028C2B8000000001"        , vpblendd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E369028C2B8000000001"        , vpblendd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E36D02CB01"                  , vpblendd(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("C4E36D028C2B8000000001"        , vpblendd(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E36D028C2B8000000001"        , vpblendd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E3694CCB40"                  , vpblendvb(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("C4E3694C8C2B8000000060"        , vpblendvb(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3694C8C2B8000000060"        , vpblendvb(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E36D4CCB40"                  , vpblendvb(ymm1, ymm2, ymm3, ymm4));
  TEST_INSTRUCTION("C4E36D4C8C2B8000000060"        , vpblendvb(ymm1, ymm2, ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("C4E36D4C8C2B8000000060"        , vpblendvb(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("C4E3690ECB01"                  , vpblendw(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("C4E3690E8C2B8000000001"        , vpblendw(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E3690E8C2B8000000001"        , vpblendw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E36D0ECB01"                  , vpblendw(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("C4E36D0E8C2B8000000001"        , vpblendw(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E36D0E8C2B8000000001"        , vpblendw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E27978CA"                    , vpbroadcastb(xmm1, xmm2));
  TEST_INSTRUCTION("C4E279788C1A80000000"          , vpbroadcastb(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E279788C1A80000000"          , vpbroadcastb(xmm1, byte_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D78CA"                    , vpbroadcastb(ymm1, xmm2));
  TEST_INSTRUCTION("C4E27D788C1A80000000"          , vpbroadcastb(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D788C1A80000000"          , vpbroadcastb(ymm1, byte_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27958CA"                    , vpbroadcastd(xmm1, xmm2));
  TEST_INSTRUCTION("C4E279588C1A80000000"          , vpbroadcastd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E279588C1A80000000"          , vpbroadcastd(xmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D58CA"                    , vpbroadcastd(ymm1, xmm2));
  TEST_INSTRUCTION("C4E27D588C1A80000000"          , vpbroadcastd(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D588C1A80000000"          , vpbroadcastd(ymm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27959CA"                    , vpbroadcastq(xmm1, xmm2));
  TEST_INSTRUCTION("C4E279598C1A80000000"          , vpbroadcastq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E279598C1A80000000"          , vpbroadcastq(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D59CA"                    , vpbroadcastq(ymm1, xmm2));
  TEST_INSTRUCTION("C4E27D598C1A80000000"          , vpbroadcastq(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D598C1A80000000"          , vpbroadcastq(ymm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27979CA"                    , vpbroadcastw(xmm1, xmm2));
  TEST_INSTRUCTION("C4E279798C1A80000000"          , vpbroadcastw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E279798C1A80000000"          , vpbroadcastw(xmm1, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D79CA"                    , vpbroadcastw(ymm1, xmm2));
  TEST_INSTRUCTION("C4E27D798C1A80000000"          , vpbroadcastw(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D798C1A80000000"          , vpbroadcastw(ymm1, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E36944CB01"                  , vpclmulqdq(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("C4E369448C2B8000000001"        , vpclmulqdq(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E369448C2B8000000001"        , vpclmulqdq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E36D44CB01"                  , vpclmulqdq(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("C4E36D448C2B8000000001"        , vpclmulqdq(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E36D448C2B8000000001"        , vpclmulqdq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C5E974CB"                      , vpcmpeqb(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9748C2B80000000"            , vpcmpeqb(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9748C2B80000000"            , vpcmpeqb(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED74CB"                      , vpcmpeqb(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5ED748C2B80000000"            , vpcmpeqb(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED748C2B80000000"            , vpcmpeqb(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E976CB"                      , vpcmpeqd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9768C2B80000000"            , vpcmpeqd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9768C2B80000000"            , vpcmpeqd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED76CB"                      , vpcmpeqd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5ED768C2B80000000"            , vpcmpeqd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED768C2B80000000"            , vpcmpeqd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26929CB"                    , vpcmpeqq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269298C2B80000000"          , vpcmpeqq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269298C2B80000000"          , vpcmpeqq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D29CB"                    , vpcmpeqq(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D298C2B80000000"          , vpcmpeqq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D298C2B80000000"          , vpcmpeqq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E975CB"                      , vpcmpeqw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9758C2B80000000"            , vpcmpeqw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9758C2B80000000"            , vpcmpeqw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED75CB"                      , vpcmpeqw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5ED758C2B80000000"            , vpcmpeqw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED758C2B80000000"            , vpcmpeqw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E37961CA01"                  , vpcmpestri(xmm1, xmm2, 1, ecx, eax, edx));
  TEST_INSTRUCTION("C4E379618C1A8000000001"        , vpcmpestri(xmm1, ptr(edx, ebx, 0, 128), 1, ecx, eax, edx));
  TEST_INSTRUCTION("C4E379618C1A8000000001"        , vpcmpestri(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1, ecx, eax, edx));
  TEST_INSTRUCTION("C4E37960CA01"                  , vpcmpestrm(xmm1, xmm2, 1, xmm0, eax, edx));
  TEST_INSTRUCTION("C4E379608C1A8000000001"        , vpcmpestrm(xmm1, ptr(edx, ebx, 0, 128), 1, xmm0, eax, edx));
  TEST_INSTRUCTION("C4E379608C1A8000000001"        , vpcmpestrm(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1, xmm0, eax, edx));
  TEST_INSTRUCTION("C5E964CB"                      , vpcmpgtb(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9648C2B80000000"            , vpcmpgtb(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9648C2B80000000"            , vpcmpgtb(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED64CB"                      , vpcmpgtb(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5ED648C2B80000000"            , vpcmpgtb(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED648C2B80000000"            , vpcmpgtb(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E966CB"                      , vpcmpgtd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9668C2B80000000"            , vpcmpgtd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9668C2B80000000"            , vpcmpgtd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED66CB"                      , vpcmpgtd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5ED668C2B80000000"            , vpcmpgtd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED668C2B80000000"            , vpcmpgtd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26937CB"                    , vpcmpgtq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269378C2B80000000"          , vpcmpgtq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269378C2B80000000"          , vpcmpgtq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D37CB"                    , vpcmpgtq(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D378C2B80000000"          , vpcmpgtq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D378C2B80000000"          , vpcmpgtq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E965CB"                      , vpcmpgtw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9658C2B80000000"            , vpcmpgtw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9658C2B80000000"            , vpcmpgtw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED65CB"                      , vpcmpgtw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5ED658C2B80000000"            , vpcmpgtw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED658C2B80000000"            , vpcmpgtw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E37963CA01"                  , vpcmpistri(xmm1, xmm2, 1, ecx));
  TEST_INSTRUCTION("C4E379638C1A8000000001"        , vpcmpistri(xmm1, ptr(edx, ebx, 0, 128), 1, ecx));
  TEST_INSTRUCTION("C4E379638C1A8000000001"        , vpcmpistri(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1, ecx));
  TEST_INSTRUCTION("C4E37962CA01"                  , vpcmpistrm(xmm1, xmm2, 1, xmm0));
  TEST_INSTRUCTION("C4E379628C1A8000000001"        , vpcmpistrm(xmm1, ptr(edx, ebx, 0, 128), 1, xmm0));
  TEST_INSTRUCTION("C4E379628C1A8000000001"        , vpcmpistrm(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1, xmm0));
  TEST_INSTRUCTION("C4E36D06CB01"                  , vperm2f128(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("C4E36D068C2B8000000001"        , vperm2f128(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E36D068C2B8000000001"        , vperm2f128(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E36D46CB01"                  , vperm2i128(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("C4E36D468C2B8000000001"        , vperm2i128(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E36D468C2B8000000001"        , vperm2i128(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E26D36CB"                    , vpermd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D368C2B80000000"          , vpermd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D368C2B80000000"          , vpermd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E36949CB41"                  , vpermil2pd(xmm1, xmm2, xmm3, xmm4, 1));
  TEST_INSTRUCTION("C4E369498C2B8000000061"        , vpermil2pd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6, 1));
  TEST_INSTRUCTION("C4E369498C2B8000000061"        , vpermil2pd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), xmm6, 1));
  TEST_INSTRUCTION("C4E3E9498C358000000031"        , vpermil2pd(xmm1, xmm2, xmm3, ptr(ebp, esi, 0, 128), 1));
  TEST_INSTRUCTION("C4E3E9498C358000000031"        , vpermil2pd(xmm1, xmm2, xmm3, xmmword_ptr(ebp, esi, 0, 128), 1));
  TEST_INSTRUCTION("C4E36D49CB41"                  , vpermil2pd(ymm1, ymm2, ymm3, ymm4, 1));
  TEST_INSTRUCTION("C4E36D498C2B8000000061"        , vpermil2pd(ymm1, ymm2, ptr(ebx, ebp, 0, 128), ymm6, 1));
  TEST_INSTRUCTION("C4E36D498C2B8000000061"        , vpermil2pd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), ymm6, 1));
  TEST_INSTRUCTION("C4E3ED498C358000000031"        , vpermil2pd(ymm1, ymm2, ymm3, ptr(ebp, esi, 0, 128), 1));
  TEST_INSTRUCTION("C4E3ED498C358000000031"        , vpermil2pd(ymm1, ymm2, ymm3, ymmword_ptr(ebp, esi, 0, 128), 1));
  TEST_INSTRUCTION("C4E36948CB41"                  , vpermil2ps(xmm1, xmm2, xmm3, xmm4, 1));
  TEST_INSTRUCTION("C4E369488C2B8000000061"        , vpermil2ps(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6, 1));
  TEST_INSTRUCTION("C4E369488C2B8000000061"        , vpermil2ps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), xmm6, 1));
  TEST_INSTRUCTION("C4E3E9488C358000000031"        , vpermil2ps(xmm1, xmm2, xmm3, ptr(ebp, esi, 0, 128), 1));
  TEST_INSTRUCTION("C4E3E9488C358000000031"        , vpermil2ps(xmm1, xmm2, xmm3, xmmword_ptr(ebp, esi, 0, 128), 1));
  TEST_INSTRUCTION("C4E36D48CB41"                  , vpermil2ps(ymm1, ymm2, ymm3, ymm4, 1));
  TEST_INSTRUCTION("C4E36D488C2B8000000061"        , vpermil2ps(ymm1, ymm2, ptr(ebx, ebp, 0, 128), ymm6, 1));
  TEST_INSTRUCTION("C4E36D488C2B8000000061"        , vpermil2ps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), ymm6, 1));
  TEST_INSTRUCTION("C4E3ED488C358000000031"        , vpermil2ps(ymm1, ymm2, ymm3, ptr(ebp, esi, 0, 128), 1));
  TEST_INSTRUCTION("C4E3ED488C358000000031"        , vpermil2ps(ymm1, ymm2, ymm3, ymmword_ptr(ebp, esi, 0, 128), 1));
  TEST_INSTRUCTION("C4E2690DCB"                    , vpermilpd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E37905CA01"                  , vpermilpd(xmm1, xmm2, 1));
  TEST_INSTRUCTION("C4E2690D8C2B80000000"          , vpermilpd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2690D8C2B80000000"          , vpermilpd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E379058C1A8000000001"        , vpermilpd(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C4E379058C1A8000000001"        , vpermilpd(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C4E26D0DCB"                    , vpermilpd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E37D05CA01"                  , vpermilpd(ymm1, ymm2, 1));
  TEST_INSTRUCTION("C4E26D0D8C2B80000000"          , vpermilpd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D0D8C2B80000000"          , vpermilpd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E37D058C1A8000000001"        , vpermilpd(ymm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C4E37D058C1A8000000001"        , vpermilpd(ymm1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C4E2690CCB"                    , vpermilps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E37904CA01"                  , vpermilps(xmm1, xmm2, 1));
  TEST_INSTRUCTION("C4E2690C8C2B80000000"          , vpermilps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2690C8C2B80000000"          , vpermilps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E379048C1A8000000001"        , vpermilps(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C4E379048C1A8000000001"        , vpermilps(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C4E26D0CCB"                    , vpermilps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E37D04CA01"                  , vpermilps(ymm1, ymm2, 1));
  TEST_INSTRUCTION("C4E26D0C8C2B80000000"          , vpermilps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D0C8C2B80000000"          , vpermilps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E37D048C1A8000000001"        , vpermilps(ymm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C4E37D048C1A8000000001"        , vpermilps(ymm1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C4E3FD01CA01"                  , vpermpd(ymm1, ymm2, 1));
  TEST_INSTRUCTION("C4E3FD018C1A8000000001"        , vpermpd(ymm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C4E3FD018C1A8000000001"        , vpermpd(ymm1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C4E26D16CB"                    , vpermps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D168C2B80000000"          , vpermps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D168C2B80000000"          , vpermps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E3FD00CA01"                  , vpermq(ymm1, ymm2, 1));
  TEST_INSTRUCTION("C4E3FD008C1A8000000001"        , vpermq(ymm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C4E3FD008C1A8000000001"        , vpermq(ymm1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C4E37914D101"                  , vpextrb(ecx, xmm2, 1));
  TEST_INSTRUCTION("C4E379149C118000000001"        , vpextrb(ptr(ecx, edx, 0, 128), xmm3, 1));
  TEST_INSTRUCTION("C4E379149C118000000001"        , vpextrb(byte_ptr(ecx, edx, 0, 128), xmm3, 1));
  TEST_INSTRUCTION("C4E37916D101"                  , vpextrd(ecx, xmm2, 1));
  TEST_INSTRUCTION("C4E379169C118000000001"        , vpextrd(ptr(ecx, edx, 0, 128), xmm3, 1));
  TEST_INSTRUCTION("C4E379169C118000000001"        , vpextrd(dword_ptr(ecx, edx, 0, 128), xmm3, 1));
  TEST_INSTRUCTION("C5F9C5CA01"                    , vpextrw(ecx, xmm2, 1));
  TEST_INSTRUCTION("C4E379159C118000000001"        , vpextrw(ptr(ecx, edx, 0, 128), xmm3, 1));
  TEST_INSTRUCTION("C4E379159C118000000001"        , vpextrw(word_ptr(ecx, edx, 0, 128), xmm3, 1));
  TEST_INSTRUCTION("C4E259908C1A80000000"          , vpgatherdd(xmm1, ptr(edx, xmm3, 0, 128), xmm4));
  TEST_INSTRUCTION("C4E25D908C1A80000000"          , vpgatherdd(ymm1, ptr(edx, ymm3, 0, 128), ymm4));
  TEST_INSTRUCTION("C4E2D9908C1A80000000"          , vpgatherdq(xmm1, ptr(edx, xmm3, 0, 128), xmm4));
  TEST_INSTRUCTION("C4E2DD908C1A80000000"          , vpgatherdq(ymm1, ptr(edx, xmm3, 0, 128), ymm4));
  TEST_INSTRUCTION("C4E259918C1A80000000"          , vpgatherqd(xmm1, ptr(edx, xmm3, 0, 128), xmm4));
  TEST_INSTRUCTION("C4E25D918C1A80000000"          , vpgatherqd(xmm1, ptr(edx, ymm3, 0, 128), xmm4));
  TEST_INSTRUCTION("C4E2D9918C1A80000000"          , vpgatherqq(xmm1, ptr(edx, xmm3, 0, 128), xmm4));
  TEST_INSTRUCTION("C4E2DD918C1A80000000"          , vpgatherqq(ymm1, ptr(edx, ymm3, 0, 128), ymm4));
  TEST_INSTRUCTION("C4E26902CB"                    , vphaddd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269028C2B80000000"          , vphaddd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269028C2B80000000"          , vphaddd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D02CB"                    , vphaddd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D028C2B80000000"          , vphaddd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D028C2B80000000"          , vphaddd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26903CB"                    , vphaddsw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269038C2B80000000"          , vphaddsw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269038C2B80000000"          , vphaddsw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D03CB"                    , vphaddsw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D038C2B80000000"          , vphaddsw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D038C2B80000000"          , vphaddsw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26901CB"                    , vphaddw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269018C2B80000000"          , vphaddw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269018C2B80000000"          , vphaddw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D01CB"                    , vphaddw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D018C2B80000000"          , vphaddw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D018C2B80000000"          , vphaddw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E27941CA"                    , vphminposuw(xmm1, xmm2));
  TEST_INSTRUCTION("C4E279418C1A80000000"          , vphminposuw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E279418C1A80000000"          , vphminposuw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E26906CB"                    , vphsubd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269068C2B80000000"          , vphsubd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269068C2B80000000"          , vphsubd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D06CB"                    , vphsubd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D068C2B80000000"          , vphsubd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D068C2B80000000"          , vphsubd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26907CB"                    , vphsubsw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269078C2B80000000"          , vphsubsw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269078C2B80000000"          , vphsubsw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D07CB"                    , vphsubsw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D078C2B80000000"          , vphsubsw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D078C2B80000000"          , vphsubsw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26905CB"                    , vphsubw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269058C2B80000000"          , vphsubw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269058C2B80000000"          , vphsubw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D05CB"                    , vphsubw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D058C2B80000000"          , vphsubw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D058C2B80000000"          , vphsubw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E36920CB01"                  , vpinsrb(xmm1, xmm2, ebx, 1));
  TEST_INSTRUCTION("C4E369208C2B8000000001"        , vpinsrb(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E369208C2B8000000001"        , vpinsrb(xmm1, xmm2, byte_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E36922CB01"                  , vpinsrd(xmm1, xmm2, ebx, 1));
  TEST_INSTRUCTION("C4E369228C2B8000000001"        , vpinsrd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E369228C2B8000000001"        , vpinsrd(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C5E9C4CB01"                    , vpinsrw(xmm1, xmm2, ebx, 1));
  TEST_INSTRUCTION("C5E9C48C2B8000000001"          , vpinsrw(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C5E9C48C2B8000000001"          , vpinsrw(xmm1, xmm2, word_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E26904CB"                    , vpmaddubsw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269048C2B80000000"          , vpmaddubsw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269048C2B80000000"          , vpmaddubsw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D04CB"                    , vpmaddubsw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D048C2B80000000"          , vpmaddubsw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D048C2B80000000"          , vpmaddubsw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9F5CB"                      , vpmaddwd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9F58C2B80000000"            , vpmaddwd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9F58C2B80000000"            , vpmaddwd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDF5CB"                      , vpmaddwd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDF58C2B80000000"            , vpmaddwd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDF58C2B80000000"            , vpmaddwd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2618EA41180000000"          , vpmaskmovd(ptr(ecx, edx, 0, 128), xmm3, xmm4));
  TEST_INSTRUCTION("C4E2618EA41180000000"          , vpmaskmovd(xmmword_ptr(ecx, edx, 0, 128), xmm3, xmm4));
  TEST_INSTRUCTION("C4E2658EA41180000000"          , vpmaskmovd(ptr(ecx, edx, 0, 128), ymm3, ymm4));
  TEST_INSTRUCTION("C4E2658EA41180000000"          , vpmaskmovd(ymmword_ptr(ecx, edx, 0, 128), ymm3, ymm4));
  TEST_INSTRUCTION("C4E2698C8C2B80000000"          , vpmaskmovd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2698C8C2B80000000"          , vpmaskmovd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D8C8C2B80000000"          , vpmaskmovd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D8C8C2B80000000"          , vpmaskmovd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E18EA41180000000"          , vpmaskmovq(ptr(ecx, edx, 0, 128), xmm3, xmm4));
  TEST_INSTRUCTION("C4E2E18EA41180000000"          , vpmaskmovq(xmmword_ptr(ecx, edx, 0, 128), xmm3, xmm4));
  TEST_INSTRUCTION("C4E2E58EA41180000000"          , vpmaskmovq(ptr(ecx, edx, 0, 128), ymm3, ymm4));
  TEST_INSTRUCTION("C4E2E58EA41180000000"          , vpmaskmovq(ymmword_ptr(ecx, edx, 0, 128), ymm3, ymm4));
  TEST_INSTRUCTION("C4E2E98C8C2B80000000"          , vpmaskmovq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E98C8C2B80000000"          , vpmaskmovq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2ED8C8C2B80000000"          , vpmaskmovq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2ED8C8C2B80000000"          , vpmaskmovq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2693CCB"                    , vpmaxsb(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2693C8C2B80000000"          , vpmaxsb(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2693C8C2B80000000"          , vpmaxsb(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D3CCB"                    , vpmaxsb(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D3C8C2B80000000"          , vpmaxsb(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D3C8C2B80000000"          , vpmaxsb(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2693DCB"                    , vpmaxsd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2693D8C2B80000000"          , vpmaxsd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2693D8C2B80000000"          , vpmaxsd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D3DCB"                    , vpmaxsd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D3D8C2B80000000"          , vpmaxsd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D3D8C2B80000000"          , vpmaxsd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9EECB"                      , vpmaxsw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9EE8C2B80000000"            , vpmaxsw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9EE8C2B80000000"            , vpmaxsw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDEECB"                      , vpmaxsw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDEE8C2B80000000"            , vpmaxsw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDEE8C2B80000000"            , vpmaxsw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9DECB"                      , vpmaxub(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9DE8C2B80000000"            , vpmaxub(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9DE8C2B80000000"            , vpmaxub(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDDECB"                      , vpmaxub(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDDE8C2B80000000"            , vpmaxub(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDDE8C2B80000000"            , vpmaxub(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2693FCB"                    , vpmaxud(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2693F8C2B80000000"          , vpmaxud(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2693F8C2B80000000"          , vpmaxud(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D3FCB"                    , vpmaxud(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D3F8C2B80000000"          , vpmaxud(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D3F8C2B80000000"          , vpmaxud(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2693ECB"                    , vpmaxuw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2693E8C2B80000000"          , vpmaxuw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2693E8C2B80000000"          , vpmaxuw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D3ECB"                    , vpmaxuw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D3E8C2B80000000"          , vpmaxuw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D3E8C2B80000000"          , vpmaxuw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26938CB"                    , vpminsb(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269388C2B80000000"          , vpminsb(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269388C2B80000000"          , vpminsb(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D38CB"                    , vpminsb(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D388C2B80000000"          , vpminsb(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D388C2B80000000"          , vpminsb(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26939CB"                    , vpminsd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269398C2B80000000"          , vpminsd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269398C2B80000000"          , vpminsd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D39CB"                    , vpminsd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D398C2B80000000"          , vpminsd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D398C2B80000000"          , vpminsd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9EACB"                      , vpminsw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9EA8C2B80000000"            , vpminsw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9EA8C2B80000000"            , vpminsw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDEACB"                      , vpminsw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDEA8C2B80000000"            , vpminsw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDEA8C2B80000000"            , vpminsw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9DACB"                      , vpminub(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9DA8C2B80000000"            , vpminub(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9DA8C2B80000000"            , vpminub(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDDACB"                      , vpminub(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDDA8C2B80000000"            , vpminub(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDDA8C2B80000000"            , vpminub(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2693BCB"                    , vpminud(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2693B8C2B80000000"          , vpminud(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2693B8C2B80000000"          , vpminud(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D3BCB"                    , vpminud(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D3B8C2B80000000"          , vpminud(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D3B8C2B80000000"          , vpminud(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2693ACB"                    , vpminuw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2693A8C2B80000000"          , vpminuw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2693A8C2B80000000"          , vpminuw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D3ACB"                    , vpminuw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D3A8C2B80000000"          , vpminuw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D3A8C2B80000000"          , vpminuw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5F9D7CA"                      , vpmovmskb(ecx, xmm2));
  TEST_INSTRUCTION("C5FDD7CA"                      , vpmovmskb(ecx, ymm2));
  TEST_INSTRUCTION("C4E27921CA"                    , vpmovsxbd(xmm1, xmm2));
  TEST_INSTRUCTION("C4E279218C1A80000000"          , vpmovsxbd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E279218C1A80000000"          , vpmovsxbd(xmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D21CA"                    , vpmovsxbd(ymm1, xmm2));
  TEST_INSTRUCTION("C4E27D218C1A80000000"          , vpmovsxbd(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D218C1A80000000"          , vpmovsxbd(ymm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27922CA"                    , vpmovsxbq(xmm1, xmm2));
  TEST_INSTRUCTION("C4E279228C1A80000000"          , vpmovsxbq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E279228C1A80000000"          , vpmovsxbq(xmm1, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D22CA"                    , vpmovsxbq(ymm1, xmm2));
  TEST_INSTRUCTION("C4E27D228C1A80000000"          , vpmovsxbq(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D228C1A80000000"          , vpmovsxbq(ymm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27920CA"                    , vpmovsxbw(xmm1, xmm2));
  TEST_INSTRUCTION("C4E279208C1A80000000"          , vpmovsxbw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E279208C1A80000000"          , vpmovsxbw(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D20CA"                    , vpmovsxbw(ymm1, xmm2));
  TEST_INSTRUCTION("C4E27D208C1A80000000"          , vpmovsxbw(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D208C1A80000000"          , vpmovsxbw(ymm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27925CA"                    , vpmovsxdq(xmm1, xmm2));
  TEST_INSTRUCTION("C4E279258C1A80000000"          , vpmovsxdq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E279258C1A80000000"          , vpmovsxdq(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D25CA"                    , vpmovsxdq(ymm1, xmm2));
  TEST_INSTRUCTION("C4E27D258C1A80000000"          , vpmovsxdq(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D258C1A80000000"          , vpmovsxdq(ymm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27923CA"                    , vpmovsxwd(xmm1, xmm2));
  TEST_INSTRUCTION("C4E279238C1A80000000"          , vpmovsxwd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E279238C1A80000000"          , vpmovsxwd(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D23CA"                    , vpmovsxwd(ymm1, xmm2));
  TEST_INSTRUCTION("C4E27D238C1A80000000"          , vpmovsxwd(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D238C1A80000000"          , vpmovsxwd(ymm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27924CA"                    , vpmovsxwq(xmm1, xmm2));
  TEST_INSTRUCTION("C4E279248C1A80000000"          , vpmovsxwq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E279248C1A80000000"          , vpmovsxwq(xmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D24CA"                    , vpmovsxwq(ymm1, xmm2));
  TEST_INSTRUCTION("C4E27D248C1A80000000"          , vpmovsxwq(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D248C1A80000000"          , vpmovsxwq(ymm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27931CA"                    , vpmovzxbd(xmm1, xmm2));
  TEST_INSTRUCTION("C4E279318C1A80000000"          , vpmovzxbd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E279318C1A80000000"          , vpmovzxbd(xmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D31CA"                    , vpmovzxbd(ymm1, xmm2));
  TEST_INSTRUCTION("C4E27D318C1A80000000"          , vpmovzxbd(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D318C1A80000000"          , vpmovzxbd(ymm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27932CA"                    , vpmovzxbq(xmm1, xmm2));
  TEST_INSTRUCTION("C4E279328C1A80000000"          , vpmovzxbq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E279328C1A80000000"          , vpmovzxbq(xmm1, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D32CA"                    , vpmovzxbq(ymm1, xmm2));
  TEST_INSTRUCTION("C4E27D328C1A80000000"          , vpmovzxbq(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D328C1A80000000"          , vpmovzxbq(ymm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27930CA"                    , vpmovzxbw(xmm1, xmm2));
  TEST_INSTRUCTION("C4E279308C1A80000000"          , vpmovzxbw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E279308C1A80000000"          , vpmovzxbw(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D30CA"                    , vpmovzxbw(ymm1, xmm2));
  TEST_INSTRUCTION("C4E27D308C1A80000000"          , vpmovzxbw(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D308C1A80000000"          , vpmovzxbw(ymm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27935CA"                    , vpmovzxdq(xmm1, xmm2));
  TEST_INSTRUCTION("C4E279358C1A80000000"          , vpmovzxdq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E279358C1A80000000"          , vpmovzxdq(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D35CA"                    , vpmovzxdq(ymm1, xmm2));
  TEST_INSTRUCTION("C4E27D358C1A80000000"          , vpmovzxdq(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D358C1A80000000"          , vpmovzxdq(ymm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27933CA"                    , vpmovzxwd(xmm1, xmm2));
  TEST_INSTRUCTION("C4E279338C1A80000000"          , vpmovzxwd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E279338C1A80000000"          , vpmovzxwd(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D33CA"                    , vpmovzxwd(ymm1, xmm2));
  TEST_INSTRUCTION("C4E27D338C1A80000000"          , vpmovzxwd(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D338C1A80000000"          , vpmovzxwd(ymm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27934CA"                    , vpmovzxwq(xmm1, xmm2));
  TEST_INSTRUCTION("C4E279348C1A80000000"          , vpmovzxwq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E279348C1A80000000"          , vpmovzxwq(xmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D34CA"                    , vpmovzxwq(ymm1, xmm2));
  TEST_INSTRUCTION("C4E27D348C1A80000000"          , vpmovzxwq(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D348C1A80000000"          , vpmovzxwq(ymm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E26928CB"                    , vpmuldq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269288C2B80000000"          , vpmuldq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269288C2B80000000"          , vpmuldq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D28CB"                    , vpmuldq(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D288C2B80000000"          , vpmuldq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D288C2B80000000"          , vpmuldq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2690BCB"                    , vpmulhrsw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2690B8C2B80000000"          , vpmulhrsw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2690B8C2B80000000"          , vpmulhrsw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D0BCB"                    , vpmulhrsw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D0B8C2B80000000"          , vpmulhrsw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D0B8C2B80000000"          , vpmulhrsw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9E4CB"                      , vpmulhuw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9E48C2B80000000"            , vpmulhuw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9E48C2B80000000"            , vpmulhuw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDE4CB"                      , vpmulhuw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDE48C2B80000000"            , vpmulhuw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDE48C2B80000000"            , vpmulhuw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9E5CB"                      , vpmulhw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9E58C2B80000000"            , vpmulhw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9E58C2B80000000"            , vpmulhw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDE5CB"                      , vpmulhw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDE58C2B80000000"            , vpmulhw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDE58C2B80000000"            , vpmulhw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26940CB"                    , vpmulld(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269408C2B80000000"          , vpmulld(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269408C2B80000000"          , vpmulld(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D40CB"                    , vpmulld(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D408C2B80000000"          , vpmulld(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D408C2B80000000"          , vpmulld(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9D5CB"                      , vpmullw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9D58C2B80000000"            , vpmullw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9D58C2B80000000"            , vpmullw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDD5CB"                      , vpmullw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDD58C2B80000000"            , vpmullw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDD58C2B80000000"            , vpmullw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9F4CB"                      , vpmuludq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9F48C2B80000000"            , vpmuludq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9F48C2B80000000"            , vpmuludq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDF4CB"                      , vpmuludq(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDF48C2B80000000"            , vpmuludq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDF48C2B80000000"            , vpmuludq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9EBCB"                      , vpor(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9EB8C2B80000000"            , vpor(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9EB8C2B80000000"            , vpor(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDEBCB"                      , vpor(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDEB8C2B80000000"            , vpor(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDEB8C2B80000000"            , vpor(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9F6CB"                      , vpsadbw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9F68C2B80000000"            , vpsadbw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9F68C2B80000000"            , vpsadbw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDF6CB"                      , vpsadbw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDF68C2B80000000"            , vpsadbw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDF68C2B80000000"            , vpsadbw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26900CB"                    , vpshufb(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269008C2B80000000"          , vpshufb(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269008C2B80000000"          , vpshufb(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D00CB"                    , vpshufb(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D008C2B80000000"          , vpshufb(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D008C2B80000000"          , vpshufb(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5F970CA01"                    , vpshufd(xmm1, xmm2, 1));
  TEST_INSTRUCTION("C5F9708C1A8000000001"          , vpshufd(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C5F9708C1A8000000001"          , vpshufd(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C5FD70CA01"                    , vpshufd(ymm1, ymm2, 1));
  TEST_INSTRUCTION("C5FD708C1A8000000001"          , vpshufd(ymm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C5FD708C1A8000000001"          , vpshufd(ymm1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C5FA70CA01"                    , vpshufhw(xmm1, xmm2, 1));
  TEST_INSTRUCTION("C5FA708C1A8000000001"          , vpshufhw(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C5FA708C1A8000000001"          , vpshufhw(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C5FE70CA01"                    , vpshufhw(ymm1, ymm2, 1));
  TEST_INSTRUCTION("C5FE708C1A8000000001"          , vpshufhw(ymm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C5FE708C1A8000000001"          , vpshufhw(ymm1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C5FB70CA01"                    , vpshuflw(xmm1, xmm2, 1));
  TEST_INSTRUCTION("C5FB708C1A8000000001"          , vpshuflw(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C5FB708C1A8000000001"          , vpshuflw(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C5FF70CA01"                    , vpshuflw(ymm1, ymm2, 1));
  TEST_INSTRUCTION("C5FF708C1A8000000001"          , vpshuflw(ymm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C5FF708C1A8000000001"          , vpshuflw(ymm1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C4E26908CB"                    , vpsignb(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269088C2B80000000"          , vpsignb(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269088C2B80000000"          , vpsignb(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D08CB"                    , vpsignb(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D088C2B80000000"          , vpsignb(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D088C2B80000000"          , vpsignb(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2690ACB"                    , vpsignd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2690A8C2B80000000"          , vpsignd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2690A8C2B80000000"          , vpsignd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D0ACB"                    , vpsignd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D0A8C2B80000000"          , vpsignd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D0A8C2B80000000"          , vpsignd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26909CB"                    , vpsignw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269098C2B80000000"          , vpsignw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269098C2B80000000"          , vpsignw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D09CB"                    , vpsignw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D098C2B80000000"          , vpsignw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D098C2B80000000"          , vpsignw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9F2CB"                      , vpslld(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5F172F201"                    , vpslld(xmm1, xmm2, 1));
  TEST_INSTRUCTION("C5E9F28C2B80000000"            , vpslld(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9F28C2B80000000"            , vpslld(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDF2CB"                      , vpslld(ymm1, ymm2, xmm3));
  TEST_INSTRUCTION("C5F572F201"                    , vpslld(ymm1, ymm2, 1));
  TEST_INSTRUCTION("C5EDF28C2B80000000"            , vpslld(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDF28C2B80000000"            , vpslld(ymm1, ymm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5F173FA01"                    , vpslldq(xmm1, xmm2, 1));
  TEST_INSTRUCTION("C5F573FA01"                    , vpslldq(ymm1, ymm2, 1));
  TEST_INSTRUCTION("C5E9F3CB"                      , vpsllq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5F173F201"                    , vpsllq(xmm1, xmm2, 1));
  TEST_INSTRUCTION("C5E9F38C2B80000000"            , vpsllq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9F38C2B80000000"            , vpsllq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDF3CB"                      , vpsllq(ymm1, ymm2, xmm3));
  TEST_INSTRUCTION("C5F573F201"                    , vpsllq(ymm1, ymm2, 1));
  TEST_INSTRUCTION("C5EDF38C2B80000000"            , vpsllq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDF38C2B80000000"            , vpsllq(ymm1, ymm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26947CB"                    , vpsllvd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269478C2B80000000"          , vpsllvd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269478C2B80000000"          , vpsllvd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D47CB"                    , vpsllvd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D478C2B80000000"          , vpsllvd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D478C2B80000000"          , vpsllvd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E947CB"                    , vpsllvq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E9478C2B80000000"          , vpsllvq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9478C2B80000000"          , vpsllvq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2ED47CB"                    , vpsllvq(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E2ED478C2B80000000"          , vpsllvq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2ED478C2B80000000"          , vpsllvq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9F1CB"                      , vpsllw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5F171F201"                    , vpsllw(xmm1, xmm2, 1));
  TEST_INSTRUCTION("C5E9F18C2B80000000"            , vpsllw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9F18C2B80000000"            , vpsllw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDF1CB"                      , vpsllw(ymm1, ymm2, xmm3));
  TEST_INSTRUCTION("C5F571F201"                    , vpsllw(ymm1, ymm2, 1));
  TEST_INSTRUCTION("C5EDF18C2B80000000"            , vpsllw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDF18C2B80000000"            , vpsllw(ymm1, ymm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9E2CB"                      , vpsrad(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5F172E201"                    , vpsrad(xmm1, xmm2, 1));
  TEST_INSTRUCTION("C5E9E28C2B80000000"            , vpsrad(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9E28C2B80000000"            , vpsrad(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDE2CB"                      , vpsrad(ymm1, ymm2, xmm3));
  TEST_INSTRUCTION("C5F572E201"                    , vpsrad(ymm1, ymm2, 1));
  TEST_INSTRUCTION("C5EDE28C2B80000000"            , vpsrad(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDE28C2B80000000"            , vpsrad(ymm1, ymm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26946CB"                    , vpsravd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269468C2B80000000"          , vpsravd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269468C2B80000000"          , vpsravd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D46CB"                    , vpsravd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D468C2B80000000"          , vpsravd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D468C2B80000000"          , vpsravd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9E1CB"                      , vpsraw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5F171E201"                    , vpsraw(xmm1, xmm2, 1));
  TEST_INSTRUCTION("C5E9E18C2B80000000"            , vpsraw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9E18C2B80000000"            , vpsraw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDE1CB"                      , vpsraw(ymm1, ymm2, xmm3));
  TEST_INSTRUCTION("C5F571E201"                    , vpsraw(ymm1, ymm2, 1));
  TEST_INSTRUCTION("C5EDE18C2B80000000"            , vpsraw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDE18C2B80000000"            , vpsraw(ymm1, ymm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9D2CB"                      , vpsrld(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5F172D201"                    , vpsrld(xmm1, xmm2, 1));
  TEST_INSTRUCTION("C5E9D28C2B80000000"            , vpsrld(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9D28C2B80000000"            , vpsrld(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDD2CB"                      , vpsrld(ymm1, ymm2, xmm3));
  TEST_INSTRUCTION("C5F572D201"                    , vpsrld(ymm1, ymm2, 1));
  TEST_INSTRUCTION("C5EDD28C2B80000000"            , vpsrld(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDD28C2B80000000"            , vpsrld(ymm1, ymm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5F173DA01"                    , vpsrldq(xmm1, xmm2, 1));
  TEST_INSTRUCTION("C5F573DA01"                    , vpsrldq(ymm1, ymm2, 1));
  TEST_INSTRUCTION("C5E9D3CB"                      , vpsrlq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5F173D201"                    , vpsrlq(xmm1, xmm2, 1));
  TEST_INSTRUCTION("C5E9D38C2B80000000"            , vpsrlq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9D38C2B80000000"            , vpsrlq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDD3CB"                      , vpsrlq(ymm1, ymm2, xmm3));
  TEST_INSTRUCTION("C5F573D201"                    , vpsrlq(ymm1, ymm2, 1));
  TEST_INSTRUCTION("C5EDD38C2B80000000"            , vpsrlq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDD38C2B80000000"            , vpsrlq(ymm1, ymm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26945CB"                    , vpsrlvd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269458C2B80000000"          , vpsrlvd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269458C2B80000000"          , vpsrlvd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D45CB"                    , vpsrlvd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D458C2B80000000"          , vpsrlvd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D458C2B80000000"          , vpsrlvd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E945CB"                    , vpsrlvq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E9458C2B80000000"          , vpsrlvq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9458C2B80000000"          , vpsrlvq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2ED45CB"                    , vpsrlvq(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E2ED458C2B80000000"          , vpsrlvq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2ED458C2B80000000"          , vpsrlvq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9D1CB"                      , vpsrlw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5F171D201"                    , vpsrlw(xmm1, xmm2, 1));
  TEST_INSTRUCTION("C5E9D18C2B80000000"            , vpsrlw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9D18C2B80000000"            , vpsrlw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDD1CB"                      , vpsrlw(ymm1, ymm2, xmm3));
  TEST_INSTRUCTION("C5F571D201"                    , vpsrlw(ymm1, ymm2, 1));
  TEST_INSTRUCTION("C5EDD18C2B80000000"            , vpsrlw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDD18C2B80000000"            , vpsrlw(ymm1, ymm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9F8CB"                      , vpsubb(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9F88C2B80000000"            , vpsubb(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9F88C2B80000000"            , vpsubb(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDF8CB"                      , vpsubb(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDF88C2B80000000"            , vpsubb(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDF88C2B80000000"            , vpsubb(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9FACB"                      , vpsubd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9FA8C2B80000000"            , vpsubd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9FA8C2B80000000"            , vpsubd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDFACB"                      , vpsubd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDFA8C2B80000000"            , vpsubd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDFA8C2B80000000"            , vpsubd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9FBCB"                      , vpsubq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9FB8C2B80000000"            , vpsubq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9FB8C2B80000000"            , vpsubq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDFBCB"                      , vpsubq(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDFB8C2B80000000"            , vpsubq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDFB8C2B80000000"            , vpsubq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9E8CB"                      , vpsubsb(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9E88C2B80000000"            , vpsubsb(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9E88C2B80000000"            , vpsubsb(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDE8CB"                      , vpsubsb(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDE88C2B80000000"            , vpsubsb(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDE88C2B80000000"            , vpsubsb(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9E9CB"                      , vpsubsw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9E98C2B80000000"            , vpsubsw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9E98C2B80000000"            , vpsubsw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDE9CB"                      , vpsubsw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDE98C2B80000000"            , vpsubsw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDE98C2B80000000"            , vpsubsw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9D8CB"                      , vpsubusb(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9D88C2B80000000"            , vpsubusb(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9D88C2B80000000"            , vpsubusb(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDD8CB"                      , vpsubusb(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDD88C2B80000000"            , vpsubusb(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDD88C2B80000000"            , vpsubusb(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9D9CB"                      , vpsubusw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9D98C2B80000000"            , vpsubusw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9D98C2B80000000"            , vpsubusw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDD9CB"                      , vpsubusw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDD98C2B80000000"            , vpsubusw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDD98C2B80000000"            , vpsubusw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9F9CB"                      , vpsubw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9F98C2B80000000"            , vpsubw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9F98C2B80000000"            , vpsubw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDF9CB"                      , vpsubw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDF98C2B80000000"            , vpsubw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDF98C2B80000000"            , vpsubw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E27917CA"                    , vptest(xmm1, xmm2));
  TEST_INSTRUCTION("C4E279178C1A80000000"          , vptest(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E279178C1A80000000"          , vptest(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D17CA"                    , vptest(ymm1, ymm2));
  TEST_INSTRUCTION("C4E27D178C1A80000000"          , vptest(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D178C1A80000000"          , vptest(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5E968CB"                      , vpunpckhbw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9688C2B80000000"            , vpunpckhbw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9688C2B80000000"            , vpunpckhbw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED68CB"                      , vpunpckhbw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5ED688C2B80000000"            , vpunpckhbw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED688C2B80000000"            , vpunpckhbw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E96ACB"                      , vpunpckhdq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E96A8C2B80000000"            , vpunpckhdq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E96A8C2B80000000"            , vpunpckhdq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED6ACB"                      , vpunpckhdq(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5ED6A8C2B80000000"            , vpunpckhdq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED6A8C2B80000000"            , vpunpckhdq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E96DCB"                      , vpunpckhqdq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E96D8C2B80000000"            , vpunpckhqdq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E96D8C2B80000000"            , vpunpckhqdq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED6DCB"                      , vpunpckhqdq(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5ED6D8C2B80000000"            , vpunpckhqdq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED6D8C2B80000000"            , vpunpckhqdq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E969CB"                      , vpunpckhwd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9698C2B80000000"            , vpunpckhwd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9698C2B80000000"            , vpunpckhwd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED69CB"                      , vpunpckhwd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5ED698C2B80000000"            , vpunpckhwd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED698C2B80000000"            , vpunpckhwd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E960CB"                      , vpunpcklbw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9608C2B80000000"            , vpunpcklbw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9608C2B80000000"            , vpunpcklbw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED60CB"                      , vpunpcklbw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5ED608C2B80000000"            , vpunpcklbw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED608C2B80000000"            , vpunpcklbw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E962CB"                      , vpunpckldq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9628C2B80000000"            , vpunpckldq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9628C2B80000000"            , vpunpckldq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED62CB"                      , vpunpckldq(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5ED628C2B80000000"            , vpunpckldq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED628C2B80000000"            , vpunpckldq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E96CCB"                      , vpunpcklqdq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E96C8C2B80000000"            , vpunpcklqdq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E96C8C2B80000000"            , vpunpcklqdq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED6CCB"                      , vpunpcklqdq(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5ED6C8C2B80000000"            , vpunpcklqdq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED6C8C2B80000000"            , vpunpcklqdq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E961CB"                      , vpunpcklwd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9618C2B80000000"            , vpunpcklwd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9618C2B80000000"            , vpunpcklwd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED61CB"                      , vpunpcklwd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5ED618C2B80000000"            , vpunpcklwd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED618C2B80000000"            , vpunpcklwd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9EFCB"                      , vpxor(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9EF8C2B80000000"            , vpxor(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9EF8C2B80000000"            , vpxor(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDEFCB"                      , vpxor(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EDEF8C2B80000000"            , vpxor(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EDEF8C2B80000000"            , vpxor(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5F853CA"                      , vrcpps(xmm1, xmm2));
  TEST_INSTRUCTION("C5F8538C1A80000000"            , vrcpps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F8538C1A80000000"            , vrcpps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FC53CA"                      , vrcpps(ymm1, ymm2));
  TEST_INSTRUCTION("C5FC538C1A80000000"            , vrcpps(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FC538C1A80000000"            , vrcpps(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5EA53CB"                      , vrcpss(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5EA538C2B80000000"            , vrcpss(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EA538C2B80000000"            , vrcpss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E37909CA01"                  , vroundpd(xmm1, xmm2, 1));
  TEST_INSTRUCTION("C4E379098C1A8000000001"        , vroundpd(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C4E379098C1A8000000001"        , vroundpd(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C4E37D09CA01"                  , vroundpd(ymm1, ymm2, 1));
  TEST_INSTRUCTION("C4E37D098C1A8000000001"        , vroundpd(ymm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C4E37D098C1A8000000001"        , vroundpd(ymm1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C4E37908CA01"                  , vroundps(xmm1, xmm2, 1));
  TEST_INSTRUCTION("C4E379088C1A8000000001"        , vroundps(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C4E379088C1A8000000001"        , vroundps(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C4E37D08CA01"                  , vroundps(ymm1, ymm2, 1));
  TEST_INSTRUCTION("C4E37D088C1A8000000001"        , vroundps(ymm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C4E37D088C1A8000000001"        , vroundps(ymm1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("C4E3690BCB01"                  , vroundsd(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("C4E3690B8C2B8000000001"        , vroundsd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E3690B8C2B8000000001"        , vroundsd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E3690ACB01"                  , vroundss(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("C4E3690A8C2B8000000001"        , vroundss(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E3690A8C2B8000000001"        , vroundss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C5F852CA"                      , vrsqrtps(xmm1, xmm2));
  TEST_INSTRUCTION("C5F8528C1A80000000"            , vrsqrtps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F8528C1A80000000"            , vrsqrtps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FC52CA"                      , vrsqrtps(ymm1, ymm2));
  TEST_INSTRUCTION("C5FC528C1A80000000"            , vrsqrtps(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FC528C1A80000000"            , vrsqrtps(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5EA52CB"                      , vrsqrtss(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5EA528C2B80000000"            , vrsqrtss(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EA528C2B80000000"            , vrsqrtss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9C6CB01"                    , vshufpd(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("C5E9C68C2B8000000001"          , vshufpd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C5E9C68C2B8000000001"          , vshufpd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C5EDC6CB01"                    , vshufpd(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("C5EDC68C2B8000000001"          , vshufpd(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C5EDC68C2B8000000001"          , vshufpd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C5E8C6CB01"                    , vshufps(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("C5E8C68C2B8000000001"          , vshufps(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C5E8C68C2B8000000001"          , vshufps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C5ECC6CB01"                    , vshufps(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("C5ECC68C2B8000000001"          , vshufps(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C5ECC68C2B8000000001"          , vshufps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C5F951CA"                      , vsqrtpd(xmm1, xmm2));
  TEST_INSTRUCTION("C5F9518C1A80000000"            , vsqrtpd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F9518C1A80000000"            , vsqrtpd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FD51CA"                      , vsqrtpd(ymm1, ymm2));
  TEST_INSTRUCTION("C5FD518C1A80000000"            , vsqrtpd(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FD518C1A80000000"            , vsqrtpd(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F851CA"                      , vsqrtps(xmm1, xmm2));
  TEST_INSTRUCTION("C5F8518C1A80000000"            , vsqrtps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F8518C1A80000000"            , vsqrtps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FC51CA"                      , vsqrtps(ymm1, ymm2));
  TEST_INSTRUCTION("C5FC518C1A80000000"            , vsqrtps(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FC518C1A80000000"            , vsqrtps(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5EB51CB"                      , vsqrtsd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5EB518C2B80000000"            , vsqrtsd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EB518C2B80000000"            , vsqrtsd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EA51CB"                      , vsqrtss(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5EA518C2B80000000"            , vsqrtss(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EA518C2B80000000"            , vsqrtss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5F8AE9C1180000000"            , vstmxcsr(ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("C5F8AE9C1180000000"            , vstmxcsr(dword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("C5E95CCB"                      , vsubpd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E95C8C2B80000000"            , vsubpd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E95C8C2B80000000"            , vsubpd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED5CCB"                      , vsubpd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5ED5C8C2B80000000"            , vsubpd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED5C8C2B80000000"            , vsubpd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E85CCB"                      , vsubps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E85C8C2B80000000"            , vsubps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E85C8C2B80000000"            , vsubps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EC5CCB"                      , vsubps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EC5C8C2B80000000"            , vsubps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EC5C8C2B80000000"            , vsubps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EB5CCB"                      , vsubsd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5EB5C8C2B80000000"            , vsubsd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EB5C8C2B80000000"            , vsubsd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EA5CCB"                      , vsubss(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5EA5C8C2B80000000"            , vsubss(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EA5C8C2B80000000"            , vsubss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2790FCA"                    , vtestpd(xmm1, xmm2));
  TEST_INSTRUCTION("C4E2790F8C1A80000000"          , vtestpd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E2790F8C1A80000000"          , vtestpd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D0FCA"                    , vtestpd(ymm1, ymm2));
  TEST_INSTRUCTION("C4E27D0F8C1A80000000"          , vtestpd(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D0F8C1A80000000"          , vtestpd(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E2790ECA"                    , vtestps(xmm1, xmm2));
  TEST_INSTRUCTION("C4E2790E8C1A80000000"          , vtestps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E2790E8C1A80000000"          , vtestps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D0ECA"                    , vtestps(ymm1, ymm2));
  TEST_INSTRUCTION("C4E27D0E8C1A80000000"          , vtestps(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27D0E8C1A80000000"          , vtestps(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F92ECA"                      , vucomisd(xmm1, xmm2));
  TEST_INSTRUCTION("C5F92E8C1A80000000"            , vucomisd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F92E8C1A80000000"            , vucomisd(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F82ECA"                      , vucomiss(xmm1, xmm2));
  TEST_INSTRUCTION("C5F82E8C1A80000000"            , vucomiss(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F82E8C1A80000000"            , vucomiss(xmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5E915CB"                      , vunpckhpd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9158C2B80000000"            , vunpckhpd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9158C2B80000000"            , vunpckhpd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED15CB"                      , vunpckhpd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5ED158C2B80000000"            , vunpckhpd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED158C2B80000000"            , vunpckhpd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E815CB"                      , vunpckhps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E8158C2B80000000"            , vunpckhps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E8158C2B80000000"            , vunpckhps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EC15CB"                      , vunpckhps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EC158C2B80000000"            , vunpckhps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EC158C2B80000000"            , vunpckhps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E914CB"                      , vunpcklpd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9148C2B80000000"            , vunpcklpd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9148C2B80000000"            , vunpcklpd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED14CB"                      , vunpcklpd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5ED148C2B80000000"            , vunpcklpd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED148C2B80000000"            , vunpcklpd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E814CB"                      , vunpcklps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E8148C2B80000000"            , vunpcklps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E8148C2B80000000"            , vunpcklps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EC14CB"                      , vunpcklps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EC148C2B80000000"            , vunpcklps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EC148C2B80000000"            , vunpcklps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E957CB"                      , vxorpd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E9578C2B80000000"            , vxorpd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E9578C2B80000000"            , vxorpd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED57CB"                      , vxorpd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5ED578C2B80000000"            , vxorpd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5ED578C2B80000000"            , vxorpd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E857CB"                      , vxorps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C5E8578C2B80000000"            , vxorps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5E8578C2B80000000"            , vxorps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EC57CB"                      , vxorps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C5EC578C2B80000000"            , vxorps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5EC578C2B80000000"            , vxorps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C5FC77"                        , vzeroall());
  TEST_INSTRUCTION("C5F877"                        , vzeroupper());
}

static void ASMJIT_NOINLINE testX86AssemblerAVX_NE_CONVERT(AssemblerTester<x86::Assembler>& tester) noexcept {
  using namespace x86;

  TEST_INSTRUCTION("C4E27AB18C1A80000000"          , vbcstnebf162ps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27AB18C1A80000000"          , vbcstnebf162ps(xmm1, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27EB18C1A80000000"          , vbcstnebf162ps(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27EB18C1A80000000"          , vbcstnebf162ps(ymm1, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E279B18C1A80000000"          , vbcstnesh2ps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E279B18C1A80000000"          , vbcstnesh2ps(xmm1, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27DB18C1A80000000"          , vbcstnesh2ps(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27DB18C1A80000000"          , vbcstnesh2ps(ymm1, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27AB08C1A80000000"          , vcvtneebf162ps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27AB08C1A80000000"          , vcvtneebf162ps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27EB08C1A80000000"          , vcvtneebf162ps(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27EB08C1A80000000"          , vcvtneebf162ps(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E279B08C1A80000000"          , vcvtneeph2ps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E279B08C1A80000000"          , vcvtneeph2ps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27DB08C1A80000000"          , vcvtneeph2ps(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27DB08C1A80000000"          , vcvtneeph2ps(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27BB08C1A80000000"          , vcvtneobf162ps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27BB08C1A80000000"          , vcvtneobf162ps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27FB08C1A80000000"          , vcvtneobf162ps(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27FB08C1A80000000"          , vcvtneobf162ps(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E278B08C1A80000000"          , vcvtneoph2ps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E278B08C1A80000000"          , vcvtneoph2ps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27CB08C1A80000000"          , vcvtneoph2ps(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27CB08C1A80000000"          , vcvtneoph2ps(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27A72CA"                    , vex().vcvtneps2bf16(xmm1, xmm2));
  TEST_INSTRUCTION("C4E27E72CA"                    , vex().vcvtneps2bf16(xmm1, ymm2));
  TEST_INSTRUCTION("C4E27A728C1A80000000"          , vex().vcvtneps2bf16(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E27E728C1A80000000"          , vex().vcvtneps2bf16(xmm1, ymmword_ptr(edx, ebx, 0, 128)));
}

static void ASMJIT_NOINLINE testX86AssemblerAVX_VNNI(AssemblerTester<x86::Assembler>& tester) noexcept {
  using namespace x86;

  TEST_INSTRUCTION("C4E25150F4"                    , vex().vpdpbusd(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("C4E25550F4"                    , vex().vpdpbusd(ymm6, ymm5, ymm4));
  TEST_INSTRUCTION("C4E25151F4"                    , vex().vpdpbusds(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("C4E25551F4"                    , vex().vpdpbusds(ymm6, ymm5, ymm4));
  TEST_INSTRUCTION("C4E25152F4"                    , vex().vpdpwssd(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("C4E25552F4"                    , vex().vpdpwssd(ymm6, ymm5, ymm4));
  TEST_INSTRUCTION("C4E25153F4"                    , vex().vpdpwssds(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("C4E25553F4"                    , vex().vpdpwssds(ymm6, ymm5, ymm4));
}

static void ASMJIT_NOINLINE testX86AssemblerAVX_VNNI_INT8(AssemblerTester<x86::Assembler>& tester) noexcept {
  using namespace x86;

  TEST_INSTRUCTION("C4E26B50CB"                    , vpdpbssd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E26B508C2B80000000"          , vpdpbssd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26B508C2B80000000"          , vpdpbssd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26F50CB"                    , vpdpbssd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26F508C2B80000000"          , vpdpbssd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26F508C2B80000000"          , vpdpbssd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26B51CB"                    , vpdpbssds(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E26B518C2B80000000"          , vpdpbssds(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26B518C2B80000000"          , vpdpbssds(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26F51CB"                    , vpdpbssds(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26F518C2B80000000"          , vpdpbssds(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26F518C2B80000000"          , vpdpbssds(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26A50CB"                    , vpdpbsud(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E26A508C2B80000000"          , vpdpbsud(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26A508C2B80000000"          , vpdpbsud(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26E50CB"                    , vpdpbsud(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26E508C2B80000000"          , vpdpbsud(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26E508C2B80000000"          , vpdpbsud(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26A51CB"                    , vpdpbsuds(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E26A518C2B80000000"          , vpdpbsuds(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26A518C2B80000000"          , vpdpbsuds(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26E51CB"                    , vpdpbsuds(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26E518C2B80000000"          , vpdpbsuds(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26E518C2B80000000"          , vpdpbsuds(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26850CB"                    , vpdpbuud(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E268508C2B80000000"          , vpdpbuud(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E268508C2B80000000"          , vpdpbuud(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26C50CB"                    , vpdpbuud(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26C508C2B80000000"          , vpdpbuud(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26C508C2B80000000"          , vpdpbuud(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26851CB"                    , vpdpbuuds(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E268518C2B80000000"          , vpdpbuuds(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E268518C2B80000000"          , vpdpbuuds(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26C51CB"                    , vpdpbuuds(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26C518C2B80000000"          , vpdpbuuds(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26C518C2B80000000"          , vpdpbuuds(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
}

static void ASMJIT_NOINLINE testX86AssemblerAVX_VNNI_INT16(AssemblerTester<x86::Assembler>& tester) noexcept {
  using namespace x86;

  TEST_INSTRUCTION("C4E26AD2CB"                    , vpdpwsud(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E26AD28C2B80000000"          , vpdpwsud(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26AD28C2B80000000"          , vpdpwsud(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26ED2CB"                    , vpdpwsud(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26ED28C2B80000000"          , vpdpwsud(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26ED28C2B80000000"          , vpdpwsud(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26AD3CB"                    , vpdpwsuds(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E26AD38C2B80000000"          , vpdpwsuds(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26AD38C2B80000000"          , vpdpwsuds(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26ED3CB"                    , vpdpwsuds(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26ED38C2B80000000"          , vpdpwsuds(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26ED38C2B80000000"          , vpdpwsuds(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269D2CB"                    , vpdpwusd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269D28C2B80000000"          , vpdpwusd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269D28C2B80000000"          , vpdpwusd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DD2CB"                    , vpdpwusd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26DD28C2B80000000"          , vpdpwusd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DD28C2B80000000"          , vpdpwusd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269D3CB"                    , vpdpwusds(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269D38C2B80000000"          , vpdpwusds(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269D38C2B80000000"          , vpdpwusds(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DD3CB"                    , vpdpwusds(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26DD38C2B80000000"          , vpdpwusds(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DD38C2B80000000"          , vpdpwusds(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E268D2CB"                    , vpdpwuud(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E268D28C2B80000000"          , vpdpwuud(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E268D28C2B80000000"          , vpdpwuud(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26CD2CB"                    , vpdpwuud(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26CD28C2B80000000"          , vpdpwuud(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26CD28C2B80000000"          , vpdpwuud(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E268D3CB"                    , vpdpwuuds(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E268D38C2B80000000"          , vpdpwuuds(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E268D38C2B80000000"          , vpdpwuuds(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26CD3CB"                    , vpdpwuuds(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26CD38C2B80000000"          , vpdpwuuds(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26CD38C2B80000000"          , vpdpwuuds(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
}

static void ASMJIT_NOINLINE testX86AssemblerAVX_SHA512(AssemblerTester<x86::Assembler>& tester) noexcept {
  using namespace x86;

  TEST_INSTRUCTION("C4E27FCCCA"                    , vsha512msg1(ymm1, xmm2));
  TEST_INSTRUCTION("C4E27FCDCA"                    , vsha512msg2(ymm1, ymm2));
  TEST_INSTRUCTION("C4E26FCBCB"                    , vsha512rnds2(ymm1, ymm2, xmm3));
}

static void ASMJIT_NOINLINE testX86AssemblerAVX_SM3(AssemblerTester<x86::Assembler>& tester) noexcept {
  using namespace x86;

  TEST_INSTRUCTION("C4E268DACB"                    , vsm3msg1(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E268DA8C2B80000000"          , vsm3msg1(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E268DA8C2B80000000"          , vsm3msg1(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269DACB"                    , vsm3msg2(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269DA8C2B80000000"          , vsm3msg2(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269DA8C2B80000000"          , vsm3msg2(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E369DECB01"                  , vsm3rnds2(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("C4E369DE8C2B8000000001"        , vsm3rnds2(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("C4E369DE8C2B8000000001"        , vsm3rnds2(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
}

static void ASMJIT_NOINLINE testX86AssemblerAVX_SM4(AssemblerTester<x86::Assembler>& tester) noexcept {
  using namespace x86;

  TEST_INSTRUCTION("C4E26ADACB"                    , vsm4key4(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E26ADA8C2B80000000"          , vsm4key4(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26ADA8C2B80000000"          , vsm4key4(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26EDACB"                    , vsm4key4(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26EDA8C2B80000000"          , vsm4key4(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26EDA8C2B80000000"          , vsm4key4(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26BDACB"                    , vsm4rnds4(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E26BDA8C2B80000000"          , vsm4rnds4(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26BDA8C2B80000000"          , vsm4rnds4(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26FDACB"                    , vsm4rnds4(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26FDA8C2B80000000"          , vsm4rnds4(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26FDA8C2B80000000"          , vsm4rnds4(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
}

static void ASMJIT_NOINLINE testX86AssemblerFMA(AssemblerTester<x86::Assembler>& tester) noexcept {
  using namespace x86;

  TEST_INSTRUCTION("C4E2E998CB"                    , vfmadd132pd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E9988C2B80000000"          , vfmadd132pd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9988C2B80000000"          , vfmadd132pd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2ED98CB"                    , vfmadd132pd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E2ED988C2B80000000"          , vfmadd132pd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2ED988C2B80000000"          , vfmadd132pd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26998CB"                    , vfmadd132ps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269988C2B80000000"          , vfmadd132ps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269988C2B80000000"          , vfmadd132ps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D98CB"                    , vfmadd132ps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D988C2B80000000"          , vfmadd132ps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D988C2B80000000"          , vfmadd132ps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E999CB"                    , vfmadd132sd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E9998C2B80000000"          , vfmadd132sd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9998C2B80000000"          , vfmadd132sd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26999CB"                    , vfmadd132ss(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269998C2B80000000"          , vfmadd132ss(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269998C2B80000000"          , vfmadd132ss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9A8CB"                    , vfmadd213pd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E9A88C2B80000000"          , vfmadd213pd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9A88C2B80000000"          , vfmadd213pd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2EDA8CB"                    , vfmadd213pd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E2EDA88C2B80000000"          , vfmadd213pd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2EDA88C2B80000000"          , vfmadd213pd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269A8CB"                    , vfmadd213ps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269A88C2B80000000"          , vfmadd213ps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269A88C2B80000000"          , vfmadd213ps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DA8CB"                    , vfmadd213ps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26DA88C2B80000000"          , vfmadd213ps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DA88C2B80000000"          , vfmadd213ps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9A9CB"                    , vfmadd213sd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E9A98C2B80000000"          , vfmadd213sd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9A98C2B80000000"          , vfmadd213sd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269A9CB"                    , vfmadd213ss(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269A98C2B80000000"          , vfmadd213ss(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269A98C2B80000000"          , vfmadd213ss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9B8CB"                    , vfmadd231pd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E9B88C2B80000000"          , vfmadd231pd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9B88C2B80000000"          , vfmadd231pd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2EDB8CB"                    , vfmadd231pd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E2EDB88C2B80000000"          , vfmadd231pd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2EDB88C2B80000000"          , vfmadd231pd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269B8CB"                    , vfmadd231ps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269B88C2B80000000"          , vfmadd231ps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269B88C2B80000000"          , vfmadd231ps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DB8CB"                    , vfmadd231ps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26DB88C2B80000000"          , vfmadd231ps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DB88C2B80000000"          , vfmadd231ps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9B9CB"                    , vfmadd231sd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E9B98C2B80000000"          , vfmadd231sd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9B98C2B80000000"          , vfmadd231sd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269B9CB"                    , vfmadd231ss(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269B98C2B80000000"          , vfmadd231ss(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269B98C2B80000000"          , vfmadd231ss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E996CB"                    , vfmaddsub132pd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E9968C2B80000000"          , vfmaddsub132pd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9968C2B80000000"          , vfmaddsub132pd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2ED96CB"                    , vfmaddsub132pd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E2ED968C2B80000000"          , vfmaddsub132pd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2ED968C2B80000000"          , vfmaddsub132pd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26996CB"                    , vfmaddsub132ps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269968C2B80000000"          , vfmaddsub132ps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269968C2B80000000"          , vfmaddsub132ps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D96CB"                    , vfmaddsub132ps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D968C2B80000000"          , vfmaddsub132ps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D968C2B80000000"          , vfmaddsub132ps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9A6CB"                    , vfmaddsub213pd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E9A68C2B80000000"          , vfmaddsub213pd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9A68C2B80000000"          , vfmaddsub213pd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2EDA6CB"                    , vfmaddsub213pd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E2EDA68C2B80000000"          , vfmaddsub213pd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2EDA68C2B80000000"          , vfmaddsub213pd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269A6CB"                    , vfmaddsub213ps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269A68C2B80000000"          , vfmaddsub213ps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269A68C2B80000000"          , vfmaddsub213ps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DA6CB"                    , vfmaddsub213ps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26DA68C2B80000000"          , vfmaddsub213ps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DA68C2B80000000"          , vfmaddsub213ps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9B6CB"                    , vfmaddsub231pd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E9B68C2B80000000"          , vfmaddsub231pd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9B68C2B80000000"          , vfmaddsub231pd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2EDB6CB"                    , vfmaddsub231pd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E2EDB68C2B80000000"          , vfmaddsub231pd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2EDB68C2B80000000"          , vfmaddsub231pd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269B6CB"                    , vfmaddsub231ps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269B68C2B80000000"          , vfmaddsub231ps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269B68C2B80000000"          , vfmaddsub231ps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DB6CB"                    , vfmaddsub231ps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26DB68C2B80000000"          , vfmaddsub231ps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DB68C2B80000000"          , vfmaddsub231ps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E99ACB"                    , vfmsub132pd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E99A8C2B80000000"          , vfmsub132pd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E99A8C2B80000000"          , vfmsub132pd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2ED9ACB"                    , vfmsub132pd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E2ED9A8C2B80000000"          , vfmsub132pd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2ED9A8C2B80000000"          , vfmsub132pd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2699ACB"                    , vfmsub132ps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2699A8C2B80000000"          , vfmsub132ps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2699A8C2B80000000"          , vfmsub132ps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D9ACB"                    , vfmsub132ps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D9A8C2B80000000"          , vfmsub132ps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D9A8C2B80000000"          , vfmsub132ps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E99BCB"                    , vfmsub132sd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E99B8C2B80000000"          , vfmsub132sd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E99B8C2B80000000"          , vfmsub132sd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2699BCB"                    , vfmsub132ss(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2699B8C2B80000000"          , vfmsub132ss(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2699B8C2B80000000"          , vfmsub132ss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9AACB"                    , vfmsub213pd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E9AA8C2B80000000"          , vfmsub213pd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9AA8C2B80000000"          , vfmsub213pd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2EDAACB"                    , vfmsub213pd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E2EDAA8C2B80000000"          , vfmsub213pd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2EDAA8C2B80000000"          , vfmsub213pd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269AACB"                    , vfmsub213ps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269AA8C2B80000000"          , vfmsub213ps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269AA8C2B80000000"          , vfmsub213ps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DAACB"                    , vfmsub213ps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26DAA8C2B80000000"          , vfmsub213ps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DAA8C2B80000000"          , vfmsub213ps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9ABCB"                    , vfmsub213sd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E9AB8C2B80000000"          , vfmsub213sd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9AB8C2B80000000"          , vfmsub213sd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269ABCB"                    , vfmsub213ss(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269AB8C2B80000000"          , vfmsub213ss(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269AB8C2B80000000"          , vfmsub213ss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9BACB"                    , vfmsub231pd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E9BA8C2B80000000"          , vfmsub231pd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9BA8C2B80000000"          , vfmsub231pd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2EDBACB"                    , vfmsub231pd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E2EDBA8C2B80000000"          , vfmsub231pd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2EDBA8C2B80000000"          , vfmsub231pd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269BACB"                    , vfmsub231ps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269BA8C2B80000000"          , vfmsub231ps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269BA8C2B80000000"          , vfmsub231ps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DBACB"                    , vfmsub231ps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26DBA8C2B80000000"          , vfmsub231ps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DBA8C2B80000000"          , vfmsub231ps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9BBCB"                    , vfmsub231sd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E9BB8C2B80000000"          , vfmsub231sd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9BB8C2B80000000"          , vfmsub231sd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269BBCB"                    , vfmsub231ss(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269BB8C2B80000000"          , vfmsub231ss(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269BB8C2B80000000"          , vfmsub231ss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E997CB"                    , vfmsubadd132pd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E9978C2B80000000"          , vfmsubadd132pd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9978C2B80000000"          , vfmsubadd132pd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2ED97CB"                    , vfmsubadd132pd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E2ED978C2B80000000"          , vfmsubadd132pd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2ED978C2B80000000"          , vfmsubadd132pd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26997CB"                    , vfmsubadd132ps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269978C2B80000000"          , vfmsubadd132ps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269978C2B80000000"          , vfmsubadd132ps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D97CB"                    , vfmsubadd132ps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D978C2B80000000"          , vfmsubadd132ps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D978C2B80000000"          , vfmsubadd132ps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9A7CB"                    , vfmsubadd213pd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E9A78C2B80000000"          , vfmsubadd213pd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9A78C2B80000000"          , vfmsubadd213pd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2EDA7CB"                    , vfmsubadd213pd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E2EDA78C2B80000000"          , vfmsubadd213pd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2EDA78C2B80000000"          , vfmsubadd213pd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269A7CB"                    , vfmsubadd213ps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269A78C2B80000000"          , vfmsubadd213ps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269A78C2B80000000"          , vfmsubadd213ps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DA7CB"                    , vfmsubadd213ps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26DA78C2B80000000"          , vfmsubadd213ps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DA78C2B80000000"          , vfmsubadd213ps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9B7CB"                    , vfmsubadd231pd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E9B78C2B80000000"          , vfmsubadd231pd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9B78C2B80000000"          , vfmsubadd231pd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2EDB7CB"                    , vfmsubadd231pd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E2EDB78C2B80000000"          , vfmsubadd231pd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2EDB78C2B80000000"          , vfmsubadd231pd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269B7CB"                    , vfmsubadd231ps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269B78C2B80000000"          , vfmsubadd231ps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269B78C2B80000000"          , vfmsubadd231ps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DB7CB"                    , vfmsubadd231ps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26DB78C2B80000000"          , vfmsubadd231ps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DB78C2B80000000"          , vfmsubadd231ps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E99CCB"                    , vfnmadd132pd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E99C8C2B80000000"          , vfnmadd132pd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E99C8C2B80000000"          , vfnmadd132pd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2ED9CCB"                    , vfnmadd132pd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E2ED9C8C2B80000000"          , vfnmadd132pd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2ED9C8C2B80000000"          , vfnmadd132pd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2699CCB"                    , vfnmadd132ps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2699C8C2B80000000"          , vfnmadd132ps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2699C8C2B80000000"          , vfnmadd132ps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D9CCB"                    , vfnmadd132ps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D9C8C2B80000000"          , vfnmadd132ps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D9C8C2B80000000"          , vfnmadd132ps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E99DCB"                    , vfnmadd132sd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E99D8C2B80000000"          , vfnmadd132sd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E99D8C2B80000000"          , vfnmadd132sd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2699DCB"                    , vfnmadd132ss(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2699D8C2B80000000"          , vfnmadd132ss(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2699D8C2B80000000"          , vfnmadd132ss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9ACCB"                    , vfnmadd213pd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E9AC8C2B80000000"          , vfnmadd213pd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9AC8C2B80000000"          , vfnmadd213pd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2EDACCB"                    , vfnmadd213pd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E2EDAC8C2B80000000"          , vfnmadd213pd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2EDAC8C2B80000000"          , vfnmadd213pd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269ACCB"                    , vfnmadd213ps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269AC8C2B80000000"          , vfnmadd213ps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269AC8C2B80000000"          , vfnmadd213ps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DACCB"                    , vfnmadd213ps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26DAC8C2B80000000"          , vfnmadd213ps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DAC8C2B80000000"          , vfnmadd213ps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9ADCB"                    , vfnmadd213sd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E9AD8C2B80000000"          , vfnmadd213sd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9AD8C2B80000000"          , vfnmadd213sd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269ADCB"                    , vfnmadd213ss(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269AD8C2B80000000"          , vfnmadd213ss(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269AD8C2B80000000"          , vfnmadd213ss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9BCCB"                    , vfnmadd231pd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E9BC8C2B80000000"          , vfnmadd231pd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9BC8C2B80000000"          , vfnmadd231pd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2EDBCCB"                    , vfnmadd231pd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E2EDBC8C2B80000000"          , vfnmadd231pd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2EDBC8C2B80000000"          , vfnmadd231pd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269BCCB"                    , vfnmadd231ps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269BC8C2B80000000"          , vfnmadd231ps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269BC8C2B80000000"          , vfnmadd231ps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DBCCB"                    , vfnmadd231ps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26DBC8C2B80000000"          , vfnmadd231ps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DBC8C2B80000000"          , vfnmadd231ps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9BDCB"                    , vfnmadd231sd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E9BD8C2B80000000"          , vfnmadd231sd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9BD8C2B80000000"          , vfnmadd231sd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269BDCB"                    , vfnmadd231ss(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269BD8C2B80000000"          , vfnmadd231ss(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269BD8C2B80000000"          , vfnmadd231ss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E99ECB"                    , vfnmsub132pd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E99E8C2B80000000"          , vfnmsub132pd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E99E8C2B80000000"          , vfnmsub132pd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2ED9ECB"                    , vfnmsub132pd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E2ED9E8C2B80000000"          , vfnmsub132pd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2ED9E8C2B80000000"          , vfnmsub132pd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2699ECB"                    , vfnmsub132ps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2699E8C2B80000000"          , vfnmsub132ps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2699E8C2B80000000"          , vfnmsub132ps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D9ECB"                    , vfnmsub132ps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26D9E8C2B80000000"          , vfnmsub132ps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26D9E8C2B80000000"          , vfnmsub132ps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E99FCB"                    , vfnmsub132sd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E99F8C2B80000000"          , vfnmsub132sd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E99F8C2B80000000"          , vfnmsub132sd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2699FCB"                    , vfnmsub132ss(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2699F8C2B80000000"          , vfnmsub132ss(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2699F8C2B80000000"          , vfnmsub132ss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9AECB"                    , vfnmsub213pd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E9AE8C2B80000000"          , vfnmsub213pd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9AE8C2B80000000"          , vfnmsub213pd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2EDAECB"                    , vfnmsub213pd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E2EDAE8C2B80000000"          , vfnmsub213pd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2EDAE8C2B80000000"          , vfnmsub213pd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269AECB"                    , vfnmsub213ps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269AE8C2B80000000"          , vfnmsub213ps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269AE8C2B80000000"          , vfnmsub213ps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DAECB"                    , vfnmsub213ps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26DAE8C2B80000000"          , vfnmsub213ps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DAE8C2B80000000"          , vfnmsub213ps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9AFCB"                    , vfnmsub213sd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E9AF8C2B80000000"          , vfnmsub213sd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9AF8C2B80000000"          , vfnmsub213sd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269AFCB"                    , vfnmsub213ss(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269AF8C2B80000000"          , vfnmsub213ss(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269AF8C2B80000000"          , vfnmsub213ss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9BECB"                    , vfnmsub231pd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E9BE8C2B80000000"          , vfnmsub231pd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9BE8C2B80000000"          , vfnmsub231pd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2EDBECB"                    , vfnmsub231pd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E2EDBE8C2B80000000"          , vfnmsub231pd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2EDBE8C2B80000000"          , vfnmsub231pd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269BECB"                    , vfnmsub231ps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269BE8C2B80000000"          , vfnmsub231ps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269BE8C2B80000000"          , vfnmsub231ps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DBECB"                    , vfnmsub231ps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("C4E26DBE8C2B80000000"          , vfnmsub231ps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E26DBE8C2B80000000"          , vfnmsub231ps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9BFCB"                    , vfnmsub231sd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E2E9BF8C2B80000000"          , vfnmsub231sd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E2E9BF8C2B80000000"          , vfnmsub231sd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269BFCB"                    , vfnmsub231ss(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("C4E269BF8C2B80000000"          , vfnmsub231ss(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("C4E269BF8C2B80000000"          , vfnmsub231ss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
}

static void ASMJIT_NOINLINE testX86AssemblerFMA4(AssemblerTester<x86::Assembler>& tester) noexcept {
  using namespace x86;

  TEST_INSTRUCTION("C4E3E969CC30"                  , vfmaddpd(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("C4E3E9698C358000000030"        , vfmaddpd(xmm1, xmm2, xmm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3E9698C358000000030"        , vfmaddpd(xmm1, xmm2, xmm3, xmmword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E369698C2B8000000060"        , vfmaddpd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E369698C2B8000000060"        , vfmaddpd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3ED69CC30"                  , vfmaddpd(ymm1, ymm2, ymm3, ymm4));
  TEST_INSTRUCTION("C4E3ED698C358000000030"        , vfmaddpd(ymm1, ymm2, ymm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3ED698C358000000030"        , vfmaddpd(ymm1, ymm2, ymm3, ymmword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E36D698C2B8000000060"        , vfmaddpd(ymm1, ymm2, ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("C4E36D698C2B8000000060"        , vfmaddpd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("C4E3E968CC30"                  , vfmaddps(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("C4E3E9688C358000000030"        , vfmaddps(xmm1, xmm2, xmm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3E9688C358000000030"        , vfmaddps(xmm1, xmm2, xmm3, xmmword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E369688C2B8000000060"        , vfmaddps(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E369688C2B8000000060"        , vfmaddps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3ED68CC30"                  , vfmaddps(ymm1, ymm2, ymm3, ymm4));
  TEST_INSTRUCTION("C4E3ED688C358000000030"        , vfmaddps(ymm1, ymm2, ymm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3ED688C358000000030"        , vfmaddps(ymm1, ymm2, ymm3, ymmword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E36D688C2B8000000060"        , vfmaddps(ymm1, ymm2, ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("C4E36D688C2B8000000060"        , vfmaddps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("C4E3E96BCC30"                  , vfmaddsd(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("C4E3E96B8C358000000030"        , vfmaddsd(xmm1, xmm2, xmm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3E96B8C358000000030"        , vfmaddsd(xmm1, xmm2, xmm3, qword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3696B8C2B8000000060"        , vfmaddsd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3696B8C2B8000000060"        , vfmaddsd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3E96ACC30"                  , vfmaddss(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("C4E3E96A8C358000000030"        , vfmaddss(xmm1, xmm2, xmm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3E96A8C358000000030"        , vfmaddss(xmm1, xmm2, xmm3, dword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3696A8C2B8000000060"        , vfmaddss(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3696A8C2B8000000060"        , vfmaddss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3E95DCC30"                  , vfmaddsubpd(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("C4E3E95D8C358000000030"        , vfmaddsubpd(xmm1, xmm2, xmm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3E95D8C358000000030"        , vfmaddsubpd(xmm1, xmm2, xmm3, xmmword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3695D8C2B8000000060"        , vfmaddsubpd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3695D8C2B8000000060"        , vfmaddsubpd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3ED5DCC30"                  , vfmaddsubpd(ymm1, ymm2, ymm3, ymm4));
  TEST_INSTRUCTION("C4E3ED5D8C358000000030"        , vfmaddsubpd(ymm1, ymm2, ymm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3ED5D8C358000000030"        , vfmaddsubpd(ymm1, ymm2, ymm3, ymmword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E36D5D8C2B8000000060"        , vfmaddsubpd(ymm1, ymm2, ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("C4E36D5D8C2B8000000060"        , vfmaddsubpd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("C4E3E95CCC30"                  , vfmaddsubps(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("C4E3E95C8C358000000030"        , vfmaddsubps(xmm1, xmm2, xmm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3E95C8C358000000030"        , vfmaddsubps(xmm1, xmm2, xmm3, xmmword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3695C8C2B8000000060"        , vfmaddsubps(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3695C8C2B8000000060"        , vfmaddsubps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3ED5CCC30"                  , vfmaddsubps(ymm1, ymm2, ymm3, ymm4));
  TEST_INSTRUCTION("C4E3ED5C8C358000000030"        , vfmaddsubps(ymm1, ymm2, ymm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3ED5C8C358000000030"        , vfmaddsubps(ymm1, ymm2, ymm3, ymmword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E36D5C8C2B8000000060"        , vfmaddsubps(ymm1, ymm2, ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("C4E36D5C8C2B8000000060"        , vfmaddsubps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("C4E3E95FCC30"                  , vfmsubaddpd(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("C4E3E95F8C358000000030"        , vfmsubaddpd(xmm1, xmm2, xmm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3E95F8C358000000030"        , vfmsubaddpd(xmm1, xmm2, xmm3, xmmword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3695F8C2B8000000060"        , vfmsubaddpd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3695F8C2B8000000060"        , vfmsubaddpd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3ED5FCC30"                  , vfmsubaddpd(ymm1, ymm2, ymm3, ymm4));
  TEST_INSTRUCTION("C4E3ED5F8C358000000030"        , vfmsubaddpd(ymm1, ymm2, ymm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3ED5F8C358000000030"        , vfmsubaddpd(ymm1, ymm2, ymm3, ymmword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E36D5F8C2B8000000060"        , vfmsubaddpd(ymm1, ymm2, ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("C4E36D5F8C2B8000000060"        , vfmsubaddpd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("C4E3E95ECC30"                  , vfmsubaddps(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("C4E3E95E8C358000000030"        , vfmsubaddps(xmm1, xmm2, xmm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3E95E8C358000000030"        , vfmsubaddps(xmm1, xmm2, xmm3, xmmword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3695E8C2B8000000060"        , vfmsubaddps(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3695E8C2B8000000060"        , vfmsubaddps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3ED5ECC30"                  , vfmsubaddps(ymm1, ymm2, ymm3, ymm4));
  TEST_INSTRUCTION("C4E3ED5E8C358000000030"        , vfmsubaddps(ymm1, ymm2, ymm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3ED5E8C358000000030"        , vfmsubaddps(ymm1, ymm2, ymm3, ymmword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E36D5E8C2B8000000060"        , vfmsubaddps(ymm1, ymm2, ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("C4E36D5E8C2B8000000060"        , vfmsubaddps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("C4E3E96DCC30"                  , vfmsubpd(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("C4E3E96D8C358000000030"        , vfmsubpd(xmm1, xmm2, xmm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3E96D8C358000000030"        , vfmsubpd(xmm1, xmm2, xmm3, xmmword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3696D8C2B8000000060"        , vfmsubpd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3696D8C2B8000000060"        , vfmsubpd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3ED6DCC30"                  , vfmsubpd(ymm1, ymm2, ymm3, ymm4));
  TEST_INSTRUCTION("C4E3ED6D8C358000000030"        , vfmsubpd(ymm1, ymm2, ymm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3ED6D8C358000000030"        , vfmsubpd(ymm1, ymm2, ymm3, ymmword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E36D6D8C2B8000000060"        , vfmsubpd(ymm1, ymm2, ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("C4E36D6D8C2B8000000060"        , vfmsubpd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("C4E3E96CCC30"                  , vfmsubps(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("C4E3E96C8C358000000030"        , vfmsubps(xmm1, xmm2, xmm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3E96C8C358000000030"        , vfmsubps(xmm1, xmm2, xmm3, xmmword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3696C8C2B8000000060"        , vfmsubps(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3696C8C2B8000000060"        , vfmsubps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3ED6CCC30"                  , vfmsubps(ymm1, ymm2, ymm3, ymm4));
  TEST_INSTRUCTION("C4E3ED6C8C358000000030"        , vfmsubps(ymm1, ymm2, ymm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3ED6C8C358000000030"        , vfmsubps(ymm1, ymm2, ymm3, ymmword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E36D6C8C2B8000000060"        , vfmsubps(ymm1, ymm2, ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("C4E36D6C8C2B8000000060"        , vfmsubps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("C4E3E96FCC30"                  , vfmsubsd(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("C4E3E96F8C358000000030"        , vfmsubsd(xmm1, xmm2, xmm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3E96F8C358000000030"        , vfmsubsd(xmm1, xmm2, xmm3, qword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3696F8C2B8000000060"        , vfmsubsd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3696F8C2B8000000060"        , vfmsubsd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3E96ECC30"                  , vfmsubss(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("C4E3E96E8C358000000030"        , vfmsubss(xmm1, xmm2, xmm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3E96E8C358000000030"        , vfmsubss(xmm1, xmm2, xmm3, dword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3696E8C2B8000000060"        , vfmsubss(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3696E8C2B8000000060"        , vfmsubss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3E979CC30"                  , vfnmaddpd(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("C4E3E9798C358000000030"        , vfnmaddpd(xmm1, xmm2, xmm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3E9798C358000000030"        , vfnmaddpd(xmm1, xmm2, xmm3, xmmword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E369798C2B8000000060"        , vfnmaddpd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E369798C2B8000000060"        , vfnmaddpd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3ED79CC30"                  , vfnmaddpd(ymm1, ymm2, ymm3, ymm4));
  TEST_INSTRUCTION("C4E3ED798C358000000030"        , vfnmaddpd(ymm1, ymm2, ymm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3ED798C358000000030"        , vfnmaddpd(ymm1, ymm2, ymm3, ymmword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E36D798C2B8000000060"        , vfnmaddpd(ymm1, ymm2, ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("C4E36D798C2B8000000060"        , vfnmaddpd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("C4E3E978CC30"                  , vfnmaddps(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("C4E3E9788C358000000030"        , vfnmaddps(xmm1, xmm2, xmm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3E9788C358000000030"        , vfnmaddps(xmm1, xmm2, xmm3, xmmword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E369788C2B8000000060"        , vfnmaddps(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E369788C2B8000000060"        , vfnmaddps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3ED78CC30"                  , vfnmaddps(ymm1, ymm2, ymm3, ymm4));
  TEST_INSTRUCTION("C4E3ED788C358000000030"        , vfnmaddps(ymm1, ymm2, ymm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3ED788C358000000030"        , vfnmaddps(ymm1, ymm2, ymm3, ymmword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E36D788C2B8000000060"        , vfnmaddps(ymm1, ymm2, ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("C4E36D788C2B8000000060"        , vfnmaddps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("C4E3E97BCC30"                  , vfnmaddsd(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("C4E3E97B8C358000000030"        , vfnmaddsd(xmm1, xmm2, xmm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3E97B8C358000000030"        , vfnmaddsd(xmm1, xmm2, xmm3, qword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3697B8C2B8000000060"        , vfnmaddsd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3697B8C2B8000000060"        , vfnmaddsd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3E97ACC30"                  , vfnmaddss(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("C4E3E97A8C358000000030"        , vfnmaddss(xmm1, xmm2, xmm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3E97A8C358000000030"        , vfnmaddss(xmm1, xmm2, xmm3, dword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3697A8C2B8000000060"        , vfnmaddss(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3697A8C2B8000000060"        , vfnmaddss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3E97DCC30"                  , vfnmsubpd(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("C4E3E97D8C358000000030"        , vfnmsubpd(xmm1, xmm2, xmm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3E97D8C358000000030"        , vfnmsubpd(xmm1, xmm2, xmm3, xmmword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3697D8C2B8000000060"        , vfnmsubpd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3697D8C2B8000000060"        , vfnmsubpd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3ED7DCC30"                  , vfnmsubpd(ymm1, ymm2, ymm3, ymm4));
  TEST_INSTRUCTION("C4E3ED7D8C358000000030"        , vfnmsubpd(ymm1, ymm2, ymm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3ED7D8C358000000030"        , vfnmsubpd(ymm1, ymm2, ymm3, ymmword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E36D7D8C2B8000000060"        , vfnmsubpd(ymm1, ymm2, ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("C4E36D7D8C2B8000000060"        , vfnmsubpd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("C4E3E97CCC30"                  , vfnmsubps(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("C4E3E97C8C358000000030"        , vfnmsubps(xmm1, xmm2, xmm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3E97C8C358000000030"        , vfnmsubps(xmm1, xmm2, xmm3, xmmword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3697C8C2B8000000060"        , vfnmsubps(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3697C8C2B8000000060"        , vfnmsubps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3ED7CCC30"                  , vfnmsubps(ymm1, ymm2, ymm3, ymm4));
  TEST_INSTRUCTION("C4E3ED7C8C358000000030"        , vfnmsubps(ymm1, ymm2, ymm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3ED7C8C358000000030"        , vfnmsubps(ymm1, ymm2, ymm3, ymmword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E36D7C8C2B8000000060"        , vfnmsubps(ymm1, ymm2, ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("C4E36D7C8C2B8000000060"        , vfnmsubps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("C4E3E97FCC30"                  , vfnmsubsd(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("C4E3E97F8C358000000030"        , vfnmsubsd(xmm1, xmm2, xmm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3E97F8C358000000030"        , vfnmsubsd(xmm1, xmm2, xmm3, qword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3697F8C2B8000000060"        , vfnmsubsd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3697F8C2B8000000060"        , vfnmsubsd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3E97ECC30"                  , vfnmsubss(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("C4E3E97E8C358000000030"        , vfnmsubss(xmm1, xmm2, xmm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3E97E8C358000000030"        , vfnmsubss(xmm1, xmm2, xmm3, dword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("C4E3697E8C2B8000000060"        , vfnmsubss(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("C4E3697E8C2B8000000060"        , vfnmsubss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128), xmm6));
}

static void ASMJIT_NOINLINE testX86AssemblerXOP(AssemblerTester<x86::Assembler>& tester) noexcept {
  using namespace x86;

  TEST_INSTRUCTION("8FE97881CA"                    , vfrczpd(xmm1, xmm2));
  TEST_INSTRUCTION("8FE978818C1A80000000"          , vfrczpd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978818C1A80000000"          , vfrczpd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE97C81CA"                    , vfrczpd(ymm1, ymm2));
  TEST_INSTRUCTION("8FE97C818C1A80000000"          , vfrczpd(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE97C818C1A80000000"          , vfrczpd(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE97880CA"                    , vfrczps(xmm1, xmm2));
  TEST_INSTRUCTION("8FE978808C1A80000000"          , vfrczps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978808C1A80000000"          , vfrczps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE97C80CA"                    , vfrczps(ymm1, ymm2));
  TEST_INSTRUCTION("8FE97C808C1A80000000"          , vfrczps(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE97C808C1A80000000"          , vfrczps(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE97883CA"                    , vfrczsd(xmm1, xmm2));
  TEST_INSTRUCTION("8FE978838C1A80000000"          , vfrczsd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978838C1A80000000"          , vfrczsd(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE97882CA"                    , vfrczss(xmm1, xmm2));
  TEST_INSTRUCTION("8FE978828C1A80000000"          , vfrczss(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978828C1A80000000"          , vfrczss(xmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE868A2CB40"                  , vpcmov(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("8FE8E8A28C358000000030"        , vpcmov(xmm1, xmm2, xmm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("8FE8E8A28C358000000030"        , vpcmov(xmm1, xmm2, xmm3, xmmword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("8FE868A28C2B8000000060"        , vpcmov(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("8FE868A28C2B8000000060"        , vpcmov(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("8FE86CA2CB40"                  , vpcmov(ymm1, ymm2, ymm3, ymm4));
  TEST_INSTRUCTION("8FE8ECA28C358000000030"        , vpcmov(ymm1, ymm2, ymm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("8FE8ECA28C358000000030"        , vpcmov(ymm1, ymm2, ymm3, ymmword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("8FE86CA28C2B8000000060"        , vpcmov(ymm1, ymm2, ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("8FE86CA28C2B8000000060"        , vpcmov(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), ymm6));
  TEST_INSTRUCTION("8FE868CCCB01"                  , vpcomb(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("8FE868CC8C2B8000000001"        , vpcomb(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("8FE868CC8C2B8000000001"        , vpcomb(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("8FE868CECB01"                  , vpcomd(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("8FE868CE8C2B8000000001"        , vpcomd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("8FE868CE8C2B8000000001"        , vpcomd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("8FE868CFCB01"                  , vpcomq(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("8FE868CF8C2B8000000001"        , vpcomq(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("8FE868CF8C2B8000000001"        , vpcomq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("8FE868ECCB01"                  , vpcomub(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("8FE868EC8C2B8000000001"        , vpcomub(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("8FE868EC8C2B8000000001"        , vpcomub(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("8FE868EECB01"                  , vpcomud(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("8FE868EE8C2B8000000001"        , vpcomud(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("8FE868EE8C2B8000000001"        , vpcomud(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("8FE868EFCB01"                  , vpcomuq(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("8FE868EF8C2B8000000001"        , vpcomuq(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("8FE868EF8C2B8000000001"        , vpcomuq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("8FE868EDCB01"                  , vpcomuw(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("8FE868ED8C2B8000000001"        , vpcomuw(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("8FE868ED8C2B8000000001"        , vpcomuw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("8FE868CDCB01"                  , vpcomw(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("8FE868CD8C2B8000000001"        , vpcomw(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("8FE868CD8C2B8000000001"        , vpcomw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("8FE978C2CA"                    , vphaddbd(xmm1, xmm2));
  TEST_INSTRUCTION("8FE978C28C1A80000000"          , vphaddbd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978C28C1A80000000"          , vphaddbd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978C3CA"                    , vphaddbq(xmm1, xmm2));
  TEST_INSTRUCTION("8FE978C38C1A80000000"          , vphaddbq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978C38C1A80000000"          , vphaddbq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978C1CA"                    , vphaddbw(xmm1, xmm2));
  TEST_INSTRUCTION("8FE978C18C1A80000000"          , vphaddbw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978C18C1A80000000"          , vphaddbw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978CBCA"                    , vphadddq(xmm1, xmm2));
  TEST_INSTRUCTION("8FE978CB8C1A80000000"          , vphadddq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978CB8C1A80000000"          , vphadddq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978D2CA"                    , vphaddubd(xmm1, xmm2));
  TEST_INSTRUCTION("8FE978D28C1A80000000"          , vphaddubd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978D28C1A80000000"          , vphaddubd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978D3CA"                    , vphaddubq(xmm1, xmm2));
  TEST_INSTRUCTION("8FE978D38C1A80000000"          , vphaddubq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978D38C1A80000000"          , vphaddubq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978D1CA"                    , vphaddubw(xmm1, xmm2));
  TEST_INSTRUCTION("8FE978D18C1A80000000"          , vphaddubw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978D18C1A80000000"          , vphaddubw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978DBCA"                    , vphaddudq(xmm1, xmm2));
  TEST_INSTRUCTION("8FE978DB8C1A80000000"          , vphaddudq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978DB8C1A80000000"          , vphaddudq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978D6CA"                    , vphadduwd(xmm1, xmm2));
  TEST_INSTRUCTION("8FE978D68C1A80000000"          , vphadduwd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978D68C1A80000000"          , vphadduwd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978D7CA"                    , vphadduwq(xmm1, xmm2));
  TEST_INSTRUCTION("8FE978D78C1A80000000"          , vphadduwq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978D78C1A80000000"          , vphadduwq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978C6CA"                    , vphaddwd(xmm1, xmm2));
  TEST_INSTRUCTION("8FE978C68C1A80000000"          , vphaddwd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978C68C1A80000000"          , vphaddwd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978C7CA"                    , vphaddwq(xmm1, xmm2));
  TEST_INSTRUCTION("8FE978C78C1A80000000"          , vphaddwq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978C78C1A80000000"          , vphaddwq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978E1CA"                    , vphsubbw(xmm1, xmm2));
  TEST_INSTRUCTION("8FE978E18C1A80000000"          , vphsubbw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978E18C1A80000000"          , vphsubbw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978E3CA"                    , vphsubdq(xmm1, xmm2));
  TEST_INSTRUCTION("8FE978E38C1A80000000"          , vphsubdq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978E38C1A80000000"          , vphsubdq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978E2CA"                    , vphsubwd(xmm1, xmm2));
  TEST_INSTRUCTION("8FE978E28C1A80000000"          , vphsubwd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE978E28C1A80000000"          , vphsubwd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("8FE8689ECB40"                  , vpmacsdd(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("8FE8689E8C2B8000000060"        , vpmacsdd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("8FE8689E8C2B8000000060"        , vpmacsdd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("8FE8689FCB40"                  , vpmacsdqh(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("8FE8689F8C2B8000000060"        , vpmacsdqh(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("8FE8689F8C2B8000000060"        , vpmacsdqh(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("8FE86897CB40"                  , vpmacsdql(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("8FE868978C2B8000000060"        , vpmacsdql(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("8FE868978C2B8000000060"        , vpmacsdql(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("8FE8688ECB40"                  , vpmacssdd(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("8FE8688E8C2B8000000060"        , vpmacssdd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("8FE8688E8C2B8000000060"        , vpmacssdd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("8FE8688FCB40"                  , vpmacssdqh(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("8FE8688F8C2B8000000060"        , vpmacssdqh(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("8FE8688F8C2B8000000060"        , vpmacssdqh(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("8FE86887CB40"                  , vpmacssdql(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("8FE868878C2B8000000060"        , vpmacssdql(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("8FE868878C2B8000000060"        , vpmacssdql(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("8FE86886CB40"                  , vpmacsswd(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("8FE868868C2B8000000060"        , vpmacsswd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("8FE868868C2B8000000060"        , vpmacsswd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("8FE86885CB40"                  , vpmacssww(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("8FE868858C2B8000000060"        , vpmacssww(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("8FE868858C2B8000000060"        , vpmacssww(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("8FE86896CB40"                  , vpmacswd(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("8FE868968C2B8000000060"        , vpmacswd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("8FE868968C2B8000000060"        , vpmacswd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("8FE86895CB40"                  , vpmacsww(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("8FE868958C2B8000000060"        , vpmacsww(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("8FE868958C2B8000000060"        , vpmacsww(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("8FE868A6CB40"                  , vpmadcsswd(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("8FE868A68C2B8000000060"        , vpmadcsswd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("8FE868A68C2B8000000060"        , vpmadcsswd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("8FE868B6CB40"                  , vpmadcswd(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("8FE868B68C2B8000000060"        , vpmadcswd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("8FE868B68C2B8000000060"        , vpmadcswd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("8FE868A3CB40"                  , vpperm(xmm1, xmm2, xmm3, xmm4));
  TEST_INSTRUCTION("8FE8E8A38C358000000030"        , vpperm(xmm1, xmm2, xmm3, ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("8FE8E8A38C358000000030"        , vpperm(xmm1, xmm2, xmm3, xmmword_ptr(ebp, esi, 0, 128)));
  TEST_INSTRUCTION("8FE868A38C2B8000000060"        , vpperm(xmm1, xmm2, ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("8FE868A38C2B8000000060"        , vpperm(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), xmm6));
  TEST_INSTRUCTION("8FE96090CA"                    , vprotb(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("8FE878C0CA01"                  , vprotb(xmm1, xmm2, 1));
  TEST_INSTRUCTION("8FE9E8908C2B80000000"          , vprotb(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("8FE9E8908C2B80000000"          , vprotb(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("8FE958908C1A80000000"          , vprotb(xmm1, ptr(edx, ebx, 0, 128), xmm4));
  TEST_INSTRUCTION("8FE878C08C1A8000000001"        , vprotb(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("8FE958908C1A80000000"          , vprotb(xmm1, xmmword_ptr(edx, ebx, 0, 128), xmm4));
  TEST_INSTRUCTION("8FE878C08C1A8000000001"        , vprotb(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("8FE96092CA"                    , vprotd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("8FE878C2CA01"                  , vprotd(xmm1, xmm2, 1));
  TEST_INSTRUCTION("8FE9E8928C2B80000000"          , vprotd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("8FE9E8928C2B80000000"          , vprotd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("8FE958928C1A80000000"          , vprotd(xmm1, ptr(edx, ebx, 0, 128), xmm4));
  TEST_INSTRUCTION("8FE878C28C1A8000000001"        , vprotd(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("8FE958928C1A80000000"          , vprotd(xmm1, xmmword_ptr(edx, ebx, 0, 128), xmm4));
  TEST_INSTRUCTION("8FE878C28C1A8000000001"        , vprotd(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("8FE96093CA"                    , vprotq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("8FE878C3CA01"                  , vprotq(xmm1, xmm2, 1));
  TEST_INSTRUCTION("8FE9E8938C2B80000000"          , vprotq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("8FE9E8938C2B80000000"          , vprotq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("8FE958938C1A80000000"          , vprotq(xmm1, ptr(edx, ebx, 0, 128), xmm4));
  TEST_INSTRUCTION("8FE878C38C1A8000000001"        , vprotq(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("8FE958938C1A80000000"          , vprotq(xmm1, xmmword_ptr(edx, ebx, 0, 128), xmm4));
  TEST_INSTRUCTION("8FE878C38C1A8000000001"        , vprotq(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("8FE96091CA"                    , vprotw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("8FE878C1CA01"                  , vprotw(xmm1, xmm2, 1));
  TEST_INSTRUCTION("8FE9E8918C2B80000000"          , vprotw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("8FE9E8918C2B80000000"          , vprotw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("8FE958918C1A80000000"          , vprotw(xmm1, ptr(edx, ebx, 0, 128), xmm4));
  TEST_INSTRUCTION("8FE878C18C1A8000000001"        , vprotw(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("8FE958918C1A80000000"          , vprotw(xmm1, xmmword_ptr(edx, ebx, 0, 128), xmm4));
  TEST_INSTRUCTION("8FE878C18C1A8000000001"        , vprotw(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("8FE96098CA"                    , vpshab(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("8FE9E8988C2B80000000"          , vpshab(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("8FE9E8988C2B80000000"          , vpshab(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("8FE958988C1A80000000"          , vpshab(xmm1, ptr(edx, ebx, 0, 128), xmm4));
  TEST_INSTRUCTION("8FE958988C1A80000000"          , vpshab(xmm1, xmmword_ptr(edx, ebx, 0, 128), xmm4));
  TEST_INSTRUCTION("8FE9609ACA"                    , vpshad(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("8FE9E89A8C2B80000000"          , vpshad(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("8FE9E89A8C2B80000000"          , vpshad(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("8FE9589A8C1A80000000"          , vpshad(xmm1, ptr(edx, ebx, 0, 128), xmm4));
  TEST_INSTRUCTION("8FE9589A8C1A80000000"          , vpshad(xmm1, xmmword_ptr(edx, ebx, 0, 128), xmm4));
  TEST_INSTRUCTION("8FE9609BCA"                    , vpshaq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("8FE9E89B8C2B80000000"          , vpshaq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("8FE9E89B8C2B80000000"          , vpshaq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("8FE9589B8C1A80000000"          , vpshaq(xmm1, ptr(edx, ebx, 0, 128), xmm4));
  TEST_INSTRUCTION("8FE9589B8C1A80000000"          , vpshaq(xmm1, xmmword_ptr(edx, ebx, 0, 128), xmm4));
  TEST_INSTRUCTION("8FE96099CA"                    , vpshaw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("8FE9E8998C2B80000000"          , vpshaw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("8FE9E8998C2B80000000"          , vpshaw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("8FE958998C1A80000000"          , vpshaw(xmm1, ptr(edx, ebx, 0, 128), xmm4));
  TEST_INSTRUCTION("8FE958998C1A80000000"          , vpshaw(xmm1, xmmword_ptr(edx, ebx, 0, 128), xmm4));
  TEST_INSTRUCTION("8FE96094CA"                    , vpshlb(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("8FE9E8948C2B80000000"          , vpshlb(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("8FE9E8948C2B80000000"          , vpshlb(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("8FE958948C1A80000000"          , vpshlb(xmm1, ptr(edx, ebx, 0, 128), xmm4));
  TEST_INSTRUCTION("8FE958948C1A80000000"          , vpshlb(xmm1, xmmword_ptr(edx, ebx, 0, 128), xmm4));
  TEST_INSTRUCTION("8FE96096CA"                    , vpshld(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("8FE9E8968C2B80000000"          , vpshld(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("8FE9E8968C2B80000000"          , vpshld(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("8FE958968C1A80000000"          , vpshld(xmm1, ptr(edx, ebx, 0, 128), xmm4));
  TEST_INSTRUCTION("8FE958968C1A80000000"          , vpshld(xmm1, xmmword_ptr(edx, ebx, 0, 128), xmm4));
  TEST_INSTRUCTION("8FE96097CA"                    , vpshlq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("8FE9E8978C2B80000000"          , vpshlq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("8FE9E8978C2B80000000"          , vpshlq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("8FE958978C1A80000000"          , vpshlq(xmm1, ptr(edx, ebx, 0, 128), xmm4));
  TEST_INSTRUCTION("8FE958978C1A80000000"          , vpshlq(xmm1, xmmword_ptr(edx, ebx, 0, 128), xmm4));
  TEST_INSTRUCTION("8FE96095CA"                    , vpshlw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("8FE9E8958C2B80000000"          , vpshlw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("8FE9E8958C2B80000000"          , vpshlw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("8FE958958C1A80000000"          , vpshlw(xmm1, ptr(edx, ebx, 0, 128), xmm4));
  TEST_INSTRUCTION("8FE958958C1A80000000"          , vpshlw(xmm1, xmmword_ptr(edx, ebx, 0, 128), xmm4));
}

static void ASMJIT_NOINLINE testX86AssemblerAVX512(AssemblerTester<x86::Assembler>& tester) noexcept {
  using namespace x86;

  TEST_INSTRUCTION("C5ED4ACB"                      , kaddb(k1, k2, k3));
  TEST_INSTRUCTION("C4E1ED4ACB"                    , kaddd(k1, k2, k3));
  TEST_INSTRUCTION("C4E1EC4ACB"                    , kaddq(k1, k2, k3));
  TEST_INSTRUCTION("C5EC4ACB"                      , kaddw(k1, k2, k3));
  TEST_INSTRUCTION("C5ED41CB"                      , kandb(k1, k2, k3));
  TEST_INSTRUCTION("C4E1ED41CB"                    , kandd(k1, k2, k3));
  TEST_INSTRUCTION("C5ED42CB"                      , kandnb(k1, k2, k3));
  TEST_INSTRUCTION("C4E1ED42CB"                    , kandnd(k1, k2, k3));
  TEST_INSTRUCTION("C4E1EC42CB"                    , kandnq(k1, k2, k3));
  TEST_INSTRUCTION("C5EC42CB"                      , kandnw(k1, k2, k3));
  TEST_INSTRUCTION("C4E1EC41CB"                    , kandq(k1, k2, k3));
  TEST_INSTRUCTION("C5EC41CB"                      , kandw(k1, k2, k3));
  TEST_INSTRUCTION("C5F992CA"                      , kmovb(k1, edx));
  TEST_INSTRUCTION("C5F990CA"                      , kmovb(k1, k2));
  TEST_INSTRUCTION("C5F9908C1A80000000"            , kmovb(k1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F9908C1A80000000"            , kmovb(k1, byte_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F993CA"                      , kmovb(ecx, k2));
  TEST_INSTRUCTION("C5F9919C1180000000"            , kmovb(ptr(ecx, edx, 0, 128), k3));
  TEST_INSTRUCTION("C5F9919C1180000000"            , kmovb(byte_ptr(ecx, edx, 0, 128), k3));
  TEST_INSTRUCTION("C5FB92CA"                      , kmovd(k1, edx));
  TEST_INSTRUCTION("C4E1F990CA"                    , kmovd(k1, k2));
  TEST_INSTRUCTION("C4E1F9908C1A80000000"          , kmovd(k1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E1F9908C1A80000000"          , kmovd(k1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5FB93CA"                      , kmovd(ecx, k2));
  TEST_INSTRUCTION("C4E1F9919C1180000000"          , kmovd(ptr(ecx, edx, 0, 128), k3));
  TEST_INSTRUCTION("C4E1F9919C1180000000"          , kmovd(dword_ptr(ecx, edx, 0, 128), k3));
  TEST_INSTRUCTION("C4E1F890CA"                    , kmovq(k1, k2));
  TEST_INSTRUCTION("C4E1F8908C1A80000000"          , kmovq(k1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E1F8908C1A80000000"          , kmovq(k1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C4E1F8919C1180000000"          , kmovq(ptr(ecx, edx, 0, 128), k3));
  TEST_INSTRUCTION("C4E1F8919C1180000000"          , kmovq(qword_ptr(ecx, edx, 0, 128), k3));
  TEST_INSTRUCTION("C5F892CA"                      , kmovw(k1, edx));
  TEST_INSTRUCTION("C5F890CA"                      , kmovw(k1, k2));
  TEST_INSTRUCTION("C5F8908C1A80000000"            , kmovw(k1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F8908C1A80000000"            , kmovw(k1, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("C5F893CA"                      , kmovw(ecx, k2));
  TEST_INSTRUCTION("C5F8919C1180000000"            , kmovw(ptr(ecx, edx, 0, 128), k3));
  TEST_INSTRUCTION("C5F8919C1180000000"            , kmovw(word_ptr(ecx, edx, 0, 128), k3));
  TEST_INSTRUCTION("C5F944CA"                      , knotb(k1, k2));
  TEST_INSTRUCTION("C4E1F944CA"                    , knotd(k1, k2));
  TEST_INSTRUCTION("C4E1F844CA"                    , knotq(k1, k2));
  TEST_INSTRUCTION("C5F844CA"                      , knotw(k1, k2));
  TEST_INSTRUCTION("C5ED45CB"                      , korb(k1, k2, k3));
  TEST_INSTRUCTION("C4E1ED45CB"                    , kord(k1, k2, k3));
  TEST_INSTRUCTION("C4E1EC45CB"                    , korq(k1, k2, k3));
  TEST_INSTRUCTION("C5F998CA"                      , kortestb(k1, k2));
  TEST_INSTRUCTION("C4E1F998CA"                    , kortestd(k1, k2));
  TEST_INSTRUCTION("C4E1F898CA"                    , kortestq(k1, k2));
  TEST_INSTRUCTION("C5F898CA"                      , kortestw(k1, k2));
  TEST_INSTRUCTION("C5EC45CB"                      , korw(k1, k2, k3));
  TEST_INSTRUCTION("C4E37932CA01"                  , kshiftlb(k1, k2, 1));
  TEST_INSTRUCTION("C4E37933CA01"                  , kshiftld(k1, k2, 1));
  TEST_INSTRUCTION("C4E3F933CA01"                  , kshiftlq(k1, k2, 1));
  TEST_INSTRUCTION("C4E3F932CA01"                  , kshiftlw(k1, k2, 1));
  TEST_INSTRUCTION("C4E37930CA01"                  , kshiftrb(k1, k2, 1));
  TEST_INSTRUCTION("C4E37931CA01"                  , kshiftrd(k1, k2, 1));
  TEST_INSTRUCTION("C4E3F931CA01"                  , kshiftrq(k1, k2, 1));
  TEST_INSTRUCTION("C4E3F930CA01"                  , kshiftrw(k1, k2, 1));
  TEST_INSTRUCTION("C5F999CA"                      , ktestb(k1, k2));
  TEST_INSTRUCTION("C4E1F999CA"                    , ktestd(k1, k2));
  TEST_INSTRUCTION("C4E1F899CA"                    , ktestq(k1, k2));
  TEST_INSTRUCTION("C5F899CA"                      , ktestw(k1, k2));
  TEST_INSTRUCTION("C5ED4BCB"                      , kunpckbw(k1, k2, k3));
  TEST_INSTRUCTION("C4E1EC4BCB"                    , kunpckdq(k1, k2, k3));
  TEST_INSTRUCTION("C5EC4BCB"                      , kunpckwd(k1, k2, k3));
  TEST_INSTRUCTION("C5ED46CB"                      , kxnorb(k1, k2, k3));
  TEST_INSTRUCTION("C4E1ED46CB"                    , kxnord(k1, k2, k3));
  TEST_INSTRUCTION("C4E1EC46CB"                    , kxnorq(k1, k2, k3));
  TEST_INSTRUCTION("C5EC46CB"                      , kxnorw(k1, k2, k3));
  TEST_INSTRUCTION("C5ED47CB"                      , kxorb(k1, k2, k3));
  TEST_INSTRUCTION("C4E1ED47CB"                    , kxord(k1, k2, k3));
  TEST_INSTRUCTION("C4E1EC47CB"                    , kxorq(k1, k2, k3));
  TEST_INSTRUCTION("C5EC47CB"                      , kxorw(k1, k2, k3));
  TEST_INSTRUCTION("62F25F489A4C1A08"              , v4fmaddps(zmm1, zmm4, zmm5, zmm6, zmm7, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F25F489A4C1A08"              , v4fmaddps(zmm1, zmm4, zmm5, zmm6, zmm7, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F25F089B4C1A08"              , v4fmaddss(xmm1, xmm4, xmm5, xmm6, xmm7, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F25F089B4C1A08"              , v4fmaddss(xmm1, xmm4, xmm5, xmm6, xmm7, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F25F48AA4C1A08"              , v4fnmaddps(zmm1, zmm4, zmm5, zmm6, zmm7, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F25F48AA4C1A08"              , v4fnmaddps(zmm1, zmm4, zmm5, zmm6, zmm7, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F25F08AB4C1A08"              , v4fnmaddss(xmm1, xmm4, xmm5, xmm6, xmm7, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F25F08AB4C1A08"              , v4fnmaddss(xmm1, xmm4, xmm5, xmm6, xmm7, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1ED4858CB"                  , vaddpd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F1ED48584C2B02"              , vaddpd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED48584C2B02"              , vaddpd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16C4858CB"                  , vaddps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16C48584C2B02"              , vaddps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16C48584C2B02"              , vaddps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48DECB"                  , vaesdec(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48DE4C2B02"              , vaesdec(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48DE4C2B02"              , vaesdec(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48DFCB"                  , vaesdeclast(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48DF4C2B02"              , vaesdeclast(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48DF4C2B02"              , vaesdeclast(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48DCCB"                  , vaesenc(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48DC4C2B02"              , vaesenc(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48DC4C2B02"              , vaesenc(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48DDCB"                  , vaesenclast(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48DD4C2B02"              , vaesenclast(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48DD4C2B02"              , vaesenclast(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F36D0803CB01"                , valignd(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F36D08034C2B0801"            , valignd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D08034C2B0801"            , valignd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D2803CB01"                , valignd(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("62F36D28034C2B0401"            , valignd(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D28034C2B0401"            , valignd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D4803CB01"                , valignd(zmm1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F36D48034C2B0201"            , valignd(zmm1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D48034C2B0201"            , valignd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED0803CB01"                , valignq(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F3ED08034C2B0801"            , valignq(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED08034C2B0801"            , valignq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED2803CB01"                , valignq(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("62F3ED28034C2B0401"            , valignq(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED28034C2B0401"            , valignq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED4803CB01"                , valignq(zmm1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F3ED48034C2B0201"            , valignq(zmm1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED48034C2B0201"            , valignq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F1ED4855CB"                  , vandnpd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F1ED48554C2B02"              , vandnpd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED48554C2B02"              , vandnpd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16C4855CB"                  , vandnps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16C48554C2B02"              , vandnps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16C48554C2B02"              , vandnps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED4854CB"                  , vandpd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F1ED48544C2B02"              , vandpd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED48544C2B02"              , vandpd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16C4854CB"                  , vandps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16C48544C2B02"              , vandps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16C48544C2B02"              , vandps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED0865CB"                  , vblendmpd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED08654C2B08"              , vblendmpd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED08654C2B08"              , vblendmpd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED2865CB"                  , vblendmpd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED28654C2B04"              , vblendmpd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED28654C2B04"              , vblendmpd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED4865CB"                  , vblendmpd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48654C2B02"              , vblendmpd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48654C2B02"              , vblendmpd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D0865CB"                  , vblendmps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26D08654C2B08"              , vblendmps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D08654C2B08"              , vblendmps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D2865CB"                  , vblendmps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F26D28654C2B04"              , vblendmps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D28654C2B04"              , vblendmps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D4865CB"                  , vblendmps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48654C2B02"              , vblendmps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48654C2B02"              , vblendmps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F27D2819CA"                  , vbroadcastf32x2(ymm1, xmm2));
  TEST_INSTRUCTION("62F27D28194C1A10"              , vbroadcastf32x2(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D28194C1A10"              , vbroadcastf32x2(ymm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D4819CA"                  , vbroadcastf32x2(zmm1, xmm2));
  TEST_INSTRUCTION("62F27D48194C1A10"              , vbroadcastf32x2(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48194C1A10"              , vbroadcastf32x2(zmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D281A4C1A08"              , vbroadcastf32x4(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D281A4C1A08"              , vbroadcastf32x4(ymm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D481A4C1A08"              , vbroadcastf32x4(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D481A4C1A08"              , vbroadcastf32x4(zmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D481B4C1A04"              , vbroadcastf32x8(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D481B4C1A04"              , vbroadcastf32x8(zmm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD281A4C1A08"              , vbroadcastf64x2(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD281A4C1A08"              , vbroadcastf64x2(ymm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD481A4C1A08"              , vbroadcastf64x2(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD481A4C1A08"              , vbroadcastf64x2(zmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD481B4C1A04"              , vbroadcastf64x4(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD481B4C1A04"              , vbroadcastf64x4(zmm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D0859CA"                  , vbroadcasti32x2(xmm1, xmm2));
  TEST_INSTRUCTION("62F27D08594C1A10"              , vbroadcasti32x2(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D08594C1A10"              , vbroadcasti32x2(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D2859CA"                  , vbroadcasti32x2(ymm1, xmm2));
  TEST_INSTRUCTION("62F27D28594C1A10"              , vbroadcasti32x2(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D28594C1A10"              , vbroadcasti32x2(ymm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D4859CA"                  , vbroadcasti32x2(zmm1, xmm2));
  TEST_INSTRUCTION("62F27D48594C1A10"              , vbroadcasti32x2(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48594C1A10"              , vbroadcasti32x2(zmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D285A4C1A08"              , vbroadcasti32x4(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D285A4C1A08"              , vbroadcasti32x4(ymm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D485A4C1A08"              , vbroadcasti32x4(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D485A4C1A08"              , vbroadcasti32x4(zmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D485B4C1A04"              , vbroadcasti32x8(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D485B4C1A04"              , vbroadcasti32x8(zmm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD285A4C1A08"              , vbroadcasti64x2(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD285A4C1A08"              , vbroadcasti64x2(ymm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD485A4C1A08"              , vbroadcasti64x2(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD485A4C1A08"              , vbroadcasti64x2(zmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD485B4C1A04"              , vbroadcasti64x4(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD485B4C1A04"              , vbroadcasti64x4(zmm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD4819CA"                  , vbroadcastsd(zmm1, xmm2));
  TEST_INSTRUCTION("62F2FD48194C1A10"              , vbroadcastsd(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD48194C1A10"              , vbroadcastsd(zmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D4818CA"                  , vbroadcastss(zmm1, xmm2));
  TEST_INSTRUCTION("62F27D48184C1A20"              , vbroadcastss(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48184C1A20"              , vbroadcastss(zmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1ED08C2CB01"                , vcmppd(k1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F1ED08C24C2B0801"            , vcmppd(k1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F1ED08C24C2B0801"            , vcmppd(k1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F1ED28C2CB01"                , vcmppd(k1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("62F1ED28C24C2B0401"            , vcmppd(k1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F1ED28C24C2B0401"            , vcmppd(k1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F1ED48C2CB01"                , vcmppd(k1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F1ED48C24C2B0201"            , vcmppd(k1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F1ED48C24C2B0201"            , vcmppd(k1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F16C08C2CB01"                , vcmpps(k1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F16C08C24C2B0801"            , vcmpps(k1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F16C08C24C2B0801"            , vcmpps(k1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F16C28C2CB01"                , vcmpps(k1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("62F16C28C24C2B0401"            , vcmpps(k1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F16C28C24C2B0401"            , vcmpps(k1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F16C48C2CB01"                , vcmpps(k1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F16C48C24C2B0201"            , vcmpps(k1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F16C48C24C2B0201"            , vcmpps(k1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F1EF08C2CB01"                , vcmpsd(k1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F1EF08C24C2B1001"            , vcmpsd(k1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F1EF08C24C2B1001"            , vcmpsd(k1, xmm2, qword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F16E08C2CB01"                , vcmpss(k1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F16E08C24C2B2001"            , vcmpss(k1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F16E08C24C2B2001"            , vcmpss(k1, xmm2, dword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F2FD088AD1"                  , vcompresspd(xmm1, xmm2));
  TEST_INSTRUCTION("62F2FD088A5C1110"              , vcompresspd(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F2FD088A5C1110"              , vcompresspd(xmmword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F2FD288AD1"                  , vcompresspd(ymm1, ymm2));
  TEST_INSTRUCTION("62F2FD288A5C1110"              , vcompresspd(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F2FD288A5C1110"              , vcompresspd(ymmword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F2FD488AD1"                  , vcompresspd(zmm1, zmm2));
  TEST_INSTRUCTION("62F2FD488A5C1110"              , vcompresspd(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F2FD488A5C1110"              , vcompresspd(zmmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27D088AD1"                  , vcompressps(xmm1, xmm2));
  TEST_INSTRUCTION("62F27D088A5C1120"              , vcompressps(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27D088A5C1120"              , vcompressps(xmmword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27D288AD1"                  , vcompressps(ymm1, ymm2));
  TEST_INSTRUCTION("62F27D288A5C1120"              , vcompressps(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27D288A5C1120"              , vcompressps(ymmword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27D488AD1"                  , vcompressps(zmm1, zmm2));
  TEST_INSTRUCTION("62F27D488A5C1120"              , vcompressps(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27D488A5C1120"              , vcompressps(zmmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F17E48E6CA"                  , vcvtdq2pd(zmm1, ymm2));
  TEST_INSTRUCTION("62F17E48E64C1A04"              , vcvtdq2pd(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17E48E64C1A04"              , vcvtdq2pd(zmm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17C485BCA"                  , vcvtdq2ps(zmm1, zmm2));
  TEST_INSTRUCTION("62F17C485B4C1A02"              , vcvtdq2ps(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17C485B4C1A02"              , vcvtdq2ps(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F26F0872CB"                  , vcvtne2ps2bf16(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26F08724C2B08"              , vcvtne2ps2bf16(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26F08724C2B08"              , vcvtne2ps2bf16(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26F2872CB"                  , vcvtne2ps2bf16(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F26F28724C2B04"              , vcvtne2ps2bf16(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26F28724C2B04"              , vcvtne2ps2bf16(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26F4872CB"                  , vcvtne2ps2bf16(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26F48724C2B02"              , vcvtne2ps2bf16(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26F48724C2B02"              , vcvtne2ps2bf16(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F27E0872CA"                  , vcvtneps2bf16(xmm1, xmm2));
  TEST_INSTRUCTION("62F27E2872CA"                  , vcvtneps2bf16(xmm1, ymm2));
  TEST_INSTRUCTION("62F27E08724C1A08"              , vcvtneps2bf16(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27E28724C1A04"              , vcvtneps2bf16(xmm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27E4872CA"                  , vcvtneps2bf16(ymm1, zmm2));
  TEST_INSTRUCTION("62F27E48724C1A02"              , vcvtneps2bf16(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27E48724C1A02"              , vcvtneps2bf16(ymm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FF48E6CA"                  , vcvtpd2dq(ymm1, zmm2));
  TEST_INSTRUCTION("62F1FF48E64C1A02"              , vcvtpd2dq(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FF48E64C1A02"              , vcvtpd2dq(ymm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD485ACA"                  , vcvtpd2ps(ymm1, zmm2));
  TEST_INSTRUCTION("62F1FD485A4C1A02"              , vcvtpd2ps(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD485A4C1A02"              , vcvtpd2ps(ymm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD087BCA"                  , vcvtpd2qq(xmm1, xmm2));
  TEST_INSTRUCTION("62F1FD087B4C1A08"              , vcvtpd2qq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD087B4C1A08"              , vcvtpd2qq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD287BCA"                  , vcvtpd2qq(ymm1, ymm2));
  TEST_INSTRUCTION("62F1FD287B4C1A04"              , vcvtpd2qq(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD287B4C1A04"              , vcvtpd2qq(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD487BCA"                  , vcvtpd2qq(zmm1, zmm2));
  TEST_INSTRUCTION("62F1FD487B4C1A02"              , vcvtpd2qq(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD487B4C1A02"              , vcvtpd2qq(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FC0879CA"                  , vcvtpd2udq(xmm1, xmm2));
  TEST_INSTRUCTION("62F1FC2879CA"                  , vcvtpd2udq(xmm1, ymm2));
  TEST_INSTRUCTION("62F1FC08794C1A08"              , vcvtpd2udq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FC28794C1A04"              , vcvtpd2udq(xmm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FC4879CA"                  , vcvtpd2udq(ymm1, zmm2));
  TEST_INSTRUCTION("62F1FC48794C1A02"              , vcvtpd2udq(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FC48794C1A02"              , vcvtpd2udq(ymm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD0879CA"                  , vcvtpd2uqq(xmm1, xmm2));
  TEST_INSTRUCTION("62F1FD08794C1A08"              , vcvtpd2uqq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD08794C1A08"              , vcvtpd2uqq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD2879CA"                  , vcvtpd2uqq(ymm1, ymm2));
  TEST_INSTRUCTION("62F1FD28794C1A04"              , vcvtpd2uqq(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD28794C1A04"              , vcvtpd2uqq(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD4879CA"                  , vcvtpd2uqq(zmm1, zmm2));
  TEST_INSTRUCTION("62F1FD48794C1A02"              , vcvtpd2uqq(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD48794C1A02"              , vcvtpd2uqq(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D4813CA"                  , vcvtph2ps(zmm1, ymm2));
  TEST_INSTRUCTION("62F27D48134C1A04"              , vcvtph2ps(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48134C1A04"              , vcvtph2ps(zmm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D485BCA"                  , vcvtps2dq(zmm1, zmm2));
  TEST_INSTRUCTION("62F17D485B4C1A02"              , vcvtps2dq(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D485B4C1A02"              , vcvtps2dq(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17C485ACA"                  , vcvtps2pd(zmm1, ymm2));
  TEST_INSTRUCTION("62F17C485A4C1A04"              , vcvtps2pd(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17C485A4C1A04"              , vcvtps2pd(zmm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F37D481DD101"                , vcvtps2ph(ymm1, zmm2, 1));
  TEST_INSTRUCTION("62F37D481D5C110401"            , vcvtps2ph(ptr(ecx, edx, 0, 128), zmm3, 1));
  TEST_INSTRUCTION("62F37D481D5C110401"            , vcvtps2ph(ymmword_ptr(ecx, edx, 0, 128), zmm3, 1));
  TEST_INSTRUCTION("62F17D087BCA"                  , vcvtps2qq(xmm1, xmm2));
  TEST_INSTRUCTION("62F17D087B4C1A10"              , vcvtps2qq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D087B4C1A10"              , vcvtps2qq(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D287BCA"                  , vcvtps2qq(ymm1, xmm2));
  TEST_INSTRUCTION("62F17D287B4C1A08"              , vcvtps2qq(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D287B4C1A08"              , vcvtps2qq(ymm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D487BCA"                  , vcvtps2qq(zmm1, ymm2));
  TEST_INSTRUCTION("62F17D487B4C1A04"              , vcvtps2qq(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D487B4C1A04"              , vcvtps2qq(zmm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17C0879CA"                  , vcvtps2udq(xmm1, xmm2));
  TEST_INSTRUCTION("62F17C08794C1A08"              , vcvtps2udq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17C08794C1A08"              , vcvtps2udq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17C2879CA"                  , vcvtps2udq(ymm1, ymm2));
  TEST_INSTRUCTION("62F17C28794C1A04"              , vcvtps2udq(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17C28794C1A04"              , vcvtps2udq(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17C4879CA"                  , vcvtps2udq(zmm1, zmm2));
  TEST_INSTRUCTION("62F17C48794C1A02"              , vcvtps2udq(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17C48794C1A02"              , vcvtps2udq(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D0879CA"                  , vcvtps2uqq(xmm1, xmm2));
  TEST_INSTRUCTION("62F17D08794C1A10"              , vcvtps2uqq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D08794C1A10"              , vcvtps2uqq(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D2879CA"                  , vcvtps2uqq(ymm1, xmm2));
  TEST_INSTRUCTION("62F17D28794C1A08"              , vcvtps2uqq(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D28794C1A08"              , vcvtps2uqq(ymm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D4879CA"                  , vcvtps2uqq(zmm1, ymm2));
  TEST_INSTRUCTION("62F17D48794C1A04"              , vcvtps2uqq(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D48794C1A04"              , vcvtps2uqq(zmm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FE08E6CA"                  , vcvtqq2pd(xmm1, xmm2));
  TEST_INSTRUCTION("62F1FE08E64C1A08"              , vcvtqq2pd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FE08E64C1A08"              , vcvtqq2pd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FE28E6CA"                  , vcvtqq2pd(ymm1, ymm2));
  TEST_INSTRUCTION("62F1FE28E64C1A04"              , vcvtqq2pd(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FE28E64C1A04"              , vcvtqq2pd(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FE48E6CA"                  , vcvtqq2pd(zmm1, zmm2));
  TEST_INSTRUCTION("62F1FE48E64C1A02"              , vcvtqq2pd(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FE48E64C1A02"              , vcvtqq2pd(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FC085BCA"                  , vcvtqq2ps(xmm1, xmm2));
  TEST_INSTRUCTION("62F1FC285BCA"                  , vcvtqq2ps(xmm1, ymm2));
  TEST_INSTRUCTION("62F1FC085B4C1A08"              , vcvtqq2ps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FC285B4C1A04"              , vcvtqq2ps(xmm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FC485BCA"                  , vcvtqq2ps(ymm1, zmm2));
  TEST_INSTRUCTION("62F1FC485B4C1A02"              , vcvtqq2ps(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FC485B4C1A02"              , vcvtqq2ps(ymm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17F0879CA"                  , vcvtsd2usi(ecx, xmm2));
  TEST_INSTRUCTION("62F17F08794C1A10"              , vcvtsd2usi(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17F08794C1A10"              , vcvtsd2usi(ecx, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17E0879CA"                  , vcvtss2usi(ecx, xmm2));
  TEST_INSTRUCTION("62F17E08794C1A20"              , vcvtss2usi(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17E08794C1A20"              , vcvtss2usi(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD48E6CA"                  , vcvttpd2dq(ymm1, zmm2));
  TEST_INSTRUCTION("62F1FD48E64C1A02"              , vcvttpd2dq(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD48E64C1A02"              , vcvttpd2dq(ymm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD087ACA"                  , vcvttpd2qq(xmm1, xmm2));
  TEST_INSTRUCTION("62F1FD087A4C1A08"              , vcvttpd2qq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD087A4C1A08"              , vcvttpd2qq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD287ACA"                  , vcvttpd2qq(ymm1, ymm2));
  TEST_INSTRUCTION("62F1FD287A4C1A04"              , vcvttpd2qq(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD287A4C1A04"              , vcvttpd2qq(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD487ACA"                  , vcvttpd2qq(zmm1, zmm2));
  TEST_INSTRUCTION("62F1FD487A4C1A02"              , vcvttpd2qq(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD487A4C1A02"              , vcvttpd2qq(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FC0878CA"                  , vcvttpd2udq(xmm1, xmm2));
  TEST_INSTRUCTION("62F1FC2878CA"                  , vcvttpd2udq(xmm1, ymm2));
  TEST_INSTRUCTION("62F1FC08784C1A08"              , vcvttpd2udq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FC28784C1A04"              , vcvttpd2udq(xmm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FC4878CA"                  , vcvttpd2udq(ymm1, zmm2));
  TEST_INSTRUCTION("62F1FC48784C1A02"              , vcvttpd2udq(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FC48784C1A02"              , vcvttpd2udq(ymm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD0878CA"                  , vcvttpd2uqq(xmm1, xmm2));
  TEST_INSTRUCTION("62F1FD08784C1A08"              , vcvttpd2uqq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD08784C1A08"              , vcvttpd2uqq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD2878CA"                  , vcvttpd2uqq(ymm1, ymm2));
  TEST_INSTRUCTION("62F1FD28784C1A04"              , vcvttpd2uqq(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD28784C1A04"              , vcvttpd2uqq(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD4878CA"                  , vcvttpd2uqq(zmm1, zmm2));
  TEST_INSTRUCTION("62F1FD48784C1A02"              , vcvttpd2uqq(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD48784C1A02"              , vcvttpd2uqq(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17E485BCA"                  , vcvttps2dq(zmm1, zmm2));
  TEST_INSTRUCTION("62F17E485B4C1A02"              , vcvttps2dq(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17E485B4C1A02"              , vcvttps2dq(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D087ACA"                  , vcvttps2qq(xmm1, xmm2));
  TEST_INSTRUCTION("62F17D087A4C1A10"              , vcvttps2qq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D087A4C1A10"              , vcvttps2qq(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D287ACA"                  , vcvttps2qq(ymm1, xmm2));
  TEST_INSTRUCTION("62F17D287A4C1A08"              , vcvttps2qq(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D287A4C1A08"              , vcvttps2qq(ymm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D487ACA"                  , vcvttps2qq(zmm1, ymm2));
  TEST_INSTRUCTION("62F17D487A4C1A04"              , vcvttps2qq(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D487A4C1A04"              , vcvttps2qq(zmm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17C0878CA"                  , vcvttps2udq(xmm1, xmm2));
  TEST_INSTRUCTION("62F17C08784C1A08"              , vcvttps2udq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17C08784C1A08"              , vcvttps2udq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17C2878CA"                  , vcvttps2udq(ymm1, ymm2));
  TEST_INSTRUCTION("62F17C28784C1A04"              , vcvttps2udq(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17C28784C1A04"              , vcvttps2udq(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17C4878CA"                  , vcvttps2udq(zmm1, zmm2));
  TEST_INSTRUCTION("62F17C48784C1A02"              , vcvttps2udq(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17C48784C1A02"              , vcvttps2udq(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D0878CA"                  , vcvttps2uqq(xmm1, xmm2));
  TEST_INSTRUCTION("62F17D08784C1A10"              , vcvttps2uqq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D08784C1A10"              , vcvttps2uqq(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D2878CA"                  , vcvttps2uqq(ymm1, xmm2));
  TEST_INSTRUCTION("62F17D28784C1A08"              , vcvttps2uqq(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D28784C1A08"              , vcvttps2uqq(ymm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D4878CA"                  , vcvttps2uqq(zmm1, ymm2));
  TEST_INSTRUCTION("62F17D48784C1A04"              , vcvttps2uqq(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D48784C1A04"              , vcvttps2uqq(zmm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17F0878CA"                  , vcvttsd2usi(ecx, xmm2));
  TEST_INSTRUCTION("62F17F08784C1A10"              , vcvttsd2usi(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17F08784C1A10"              , vcvttsd2usi(ecx, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17E0878CA"                  , vcvttss2usi(ecx, xmm2));
  TEST_INSTRUCTION("62F17E08784C1A20"              , vcvttss2usi(ecx, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17E08784C1A20"              , vcvttss2usi(ecx, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17E087ACA"                  , vcvtudq2pd(xmm1, xmm2));
  TEST_INSTRUCTION("62F17E087A4C1A10"              , vcvtudq2pd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17E087A4C1A10"              , vcvtudq2pd(xmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17E287ACA"                  , vcvtudq2pd(ymm1, xmm2));
  TEST_INSTRUCTION("62F17E287A4C1A08"              , vcvtudq2pd(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17E287A4C1A08"              , vcvtudq2pd(ymm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17E487ACA"                  , vcvtudq2pd(zmm1, ymm2));
  TEST_INSTRUCTION("62F17E487A4C1A04"              , vcvtudq2pd(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17E487A4C1A04"              , vcvtudq2pd(zmm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17F087ACA"                  , vcvtudq2ps(xmm1, xmm2));
  TEST_INSTRUCTION("62F17F087A4C1A08"              , vcvtudq2ps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17F087A4C1A08"              , vcvtudq2ps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17F287ACA"                  , vcvtudq2ps(ymm1, ymm2));
  TEST_INSTRUCTION("62F17F287A4C1A04"              , vcvtudq2ps(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17F287A4C1A04"              , vcvtudq2ps(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17F487ACA"                  , vcvtudq2ps(zmm1, zmm2));
  TEST_INSTRUCTION("62F17F487A4C1A02"              , vcvtudq2ps(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17F487A4C1A02"              , vcvtudq2ps(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FE087ACA"                  , vcvtuqq2pd(xmm1, xmm2));
  TEST_INSTRUCTION("62F1FE087A4C1A08"              , vcvtuqq2pd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FE087A4C1A08"              , vcvtuqq2pd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FE287ACA"                  , vcvtuqq2pd(ymm1, ymm2));
  TEST_INSTRUCTION("62F1FE287A4C1A04"              , vcvtuqq2pd(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FE287A4C1A04"              , vcvtuqq2pd(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FE487ACA"                  , vcvtuqq2pd(zmm1, zmm2));
  TEST_INSTRUCTION("62F1FE487A4C1A02"              , vcvtuqq2pd(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FE487A4C1A02"              , vcvtuqq2pd(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FF087ACA"                  , vcvtuqq2ps(xmm1, xmm2));
  TEST_INSTRUCTION("62F1FF287ACA"                  , vcvtuqq2ps(xmm1, ymm2));
  TEST_INSTRUCTION("62F1FF087A4C1A08"              , vcvtuqq2ps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FF287A4C1A04"              , vcvtuqq2ps(xmm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FF487ACA"                  , vcvtuqq2ps(ymm1, zmm2));
  TEST_INSTRUCTION("62F1FF487A4C1A02"              , vcvtuqq2ps(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FF487A4C1A02"              , vcvtuqq2ps(ymm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F16F087BCB"                  , vcvtusi2sd(xmm1, xmm2, ebx));
  TEST_INSTRUCTION("62F16F087B4C2B20"              , vcvtusi2sd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16F087B4C2B20"              , vcvtusi2sd(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16E087BCB"                  , vcvtusi2ss(xmm1, xmm2, ebx));
  TEST_INSTRUCTION("62F16E087B4C2B20"              , vcvtusi2ss(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16E087B4C2B20"              , vcvtusi2ss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F36D0842CB01"                , vdbpsadbw(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F36D08424C2B0801"            , vdbpsadbw(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D08424C2B0801"            , vdbpsadbw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D2842CB01"                , vdbpsadbw(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("62F36D28424C2B0401"            , vdbpsadbw(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D28424C2B0401"            , vdbpsadbw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D4842CB01"                , vdbpsadbw(zmm1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F36D48424C2B0201"            , vdbpsadbw(zmm1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D48424C2B0201"            , vdbpsadbw(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F1ED485ECB"                  , vdivpd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F1ED485E4C2B02"              , vdivpd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED485E4C2B02"              , vdivpd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16C485ECB"                  , vdivps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16C485E4C2B02"              , vdivps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16C485E4C2B02"              , vdivps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26E0852CB"                  , vdpbf16ps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26E08524C2B08"              , vdpbf16ps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26E08524C2B08"              , vdpbf16ps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26E2852CB"                  , vdpbf16ps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F26E28524C2B04"              , vdpbf16ps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26E28524C2B04"              , vdpbf16ps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26E4852CB"                  , vdpbf16ps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26E48524C2B02"              , vdpbf16ps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26E48524C2B02"              , vdpbf16ps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2FD48C8CA"                  , vexp2pd(zmm1, zmm2));
  TEST_INSTRUCTION("62F2FD48C84C1A02"              , vexp2pd(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD48C84C1A02"              , vexp2pd(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48C8CA"                  , vexp2ps(zmm1, zmm2));
  TEST_INSTRUCTION("62F27D48C84C1A02"              , vexp2ps(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48C84C1A02"              , vexp2ps(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD0888CA"                  , vexpandpd(xmm1, xmm2));
  TEST_INSTRUCTION("62F2FD08884C1A10"              , vexpandpd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD08884C1A10"              , vexpandpd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD2888CA"                  , vexpandpd(ymm1, ymm2));
  TEST_INSTRUCTION("62F2FD28884C1A10"              , vexpandpd(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD28884C1A10"              , vexpandpd(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD4888CA"                  , vexpandpd(zmm1, zmm2));
  TEST_INSTRUCTION("62F2FD48884C1A10"              , vexpandpd(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD48884C1A10"              , vexpandpd(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D0888CA"                  , vexpandps(xmm1, xmm2));
  TEST_INSTRUCTION("62F27D08884C1A20"              , vexpandps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D08884C1A20"              , vexpandps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D2888CA"                  , vexpandps(ymm1, ymm2));
  TEST_INSTRUCTION("62F27D28884C1A20"              , vexpandps(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D28884C1A20"              , vexpandps(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D4888CA"                  , vexpandps(zmm1, zmm2));
  TEST_INSTRUCTION("62F27D48884C1A20"              , vexpandps(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48884C1A20"              , vexpandps(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F37D2819D101"                , vextractf32x4(xmm1, ymm2, 1));
  TEST_INSTRUCTION("62F37D4819D101"                , vextractf32x4(xmm1, zmm2, 1));
  TEST_INSTRUCTION("62F37D28195C110801"            , vextractf32x4(ptr(ecx, edx, 0, 128), ymm3, 1));
  TEST_INSTRUCTION("62F37D48195C110801"            , vextractf32x4(ptr(ecx, edx, 0, 128), zmm3, 1));
  TEST_INSTRUCTION("62F37D28195C110801"            , vextractf32x4(xmmword_ptr(ecx, edx, 0, 128), ymm3, 1));
  TEST_INSTRUCTION("62F37D48195C110801"            , vextractf32x4(xmmword_ptr(ecx, edx, 0, 128), zmm3, 1));
  TEST_INSTRUCTION("62F37D481BD101"                , vextractf32x8(ymm1, zmm2, 1));
  TEST_INSTRUCTION("62F37D481B5C110401"            , vextractf32x8(ptr(ecx, edx, 0, 128), zmm3, 1));
  TEST_INSTRUCTION("62F37D481B5C110401"            , vextractf32x8(ymmword_ptr(ecx, edx, 0, 128), zmm3, 1));
  TEST_INSTRUCTION("62F3FD2819D101"                , vextractf64x2(xmm1, ymm2, 1));
  TEST_INSTRUCTION("62F3FD4819D101"                , vextractf64x2(xmm1, zmm2, 1));
  TEST_INSTRUCTION("62F3FD28195C110801"            , vextractf64x2(ptr(ecx, edx, 0, 128), ymm3, 1));
  TEST_INSTRUCTION("62F3FD48195C110801"            , vextractf64x2(ptr(ecx, edx, 0, 128), zmm3, 1));
  TEST_INSTRUCTION("62F3FD28195C110801"            , vextractf64x2(xmmword_ptr(ecx, edx, 0, 128), ymm3, 1));
  TEST_INSTRUCTION("62F3FD48195C110801"            , vextractf64x2(xmmword_ptr(ecx, edx, 0, 128), zmm3, 1));
  TEST_INSTRUCTION("62F3FD481BD101"                , vextractf64x4(ymm1, zmm2, 1));
  TEST_INSTRUCTION("62F3FD481B5C110401"            , vextractf64x4(ptr(ecx, edx, 0, 128), zmm3, 1));
  TEST_INSTRUCTION("62F3FD481B5C110401"            , vextractf64x4(ymmword_ptr(ecx, edx, 0, 128), zmm3, 1));
  TEST_INSTRUCTION("62F37D2839D101"                , vextracti32x4(xmm1, ymm2, 1));
  TEST_INSTRUCTION("62F37D4839D101"                , vextracti32x4(xmm1, zmm2, 1));
  TEST_INSTRUCTION("62F37D28395C110801"            , vextracti32x4(ptr(ecx, edx, 0, 128), ymm3, 1));
  TEST_INSTRUCTION("62F37D48395C110801"            , vextracti32x4(ptr(ecx, edx, 0, 128), zmm3, 1));
  TEST_INSTRUCTION("62F37D28395C110801"            , vextracti32x4(xmmword_ptr(ecx, edx, 0, 128), ymm3, 1));
  TEST_INSTRUCTION("62F37D48395C110801"            , vextracti32x4(xmmword_ptr(ecx, edx, 0, 128), zmm3, 1));
  TEST_INSTRUCTION("62F37D483BD101"                , vextracti32x8(ymm1, zmm2, 1));
  TEST_INSTRUCTION("62F37D483B5C110401"            , vextracti32x8(ptr(ecx, edx, 0, 128), zmm3, 1));
  TEST_INSTRUCTION("62F37D483B5C110401"            , vextracti32x8(ymmword_ptr(ecx, edx, 0, 128), zmm3, 1));
  TEST_INSTRUCTION("62F3FD2839D101"                , vextracti64x2(xmm1, ymm2, 1));
  TEST_INSTRUCTION("62F3FD4839D101"                , vextracti64x2(xmm1, zmm2, 1));
  TEST_INSTRUCTION("62F3FD28395C110801"            , vextracti64x2(ptr(ecx, edx, 0, 128), ymm3, 1));
  TEST_INSTRUCTION("62F3FD48395C110801"            , vextracti64x2(ptr(ecx, edx, 0, 128), zmm3, 1));
  TEST_INSTRUCTION("62F3FD28395C110801"            , vextracti64x2(xmmword_ptr(ecx, edx, 0, 128), ymm3, 1));
  TEST_INSTRUCTION("62F3FD48395C110801"            , vextracti64x2(xmmword_ptr(ecx, edx, 0, 128), zmm3, 1));
  TEST_INSTRUCTION("62F3FD483BD101"                , vextracti64x4(ymm1, zmm2, 1));
  TEST_INSTRUCTION("62F3FD483B5C110401"            , vextracti64x4(ptr(ecx, edx, 0, 128), zmm3, 1));
  TEST_INSTRUCTION("62F3FD483B5C110401"            , vextracti64x4(ymmword_ptr(ecx, edx, 0, 128), zmm3, 1));
  TEST_INSTRUCTION("62F3ED0854CB01"                , vfixupimmpd(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F3ED08544C2B0801"            , vfixupimmpd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED08544C2B0801"            , vfixupimmpd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED2854CB01"                , vfixupimmpd(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("62F3ED28544C2B0401"            , vfixupimmpd(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED28544C2B0401"            , vfixupimmpd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED4854CB01"                , vfixupimmpd(zmm1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F3ED48544C2B0201"            , vfixupimmpd(zmm1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED48544C2B0201"            , vfixupimmpd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D0854CB01"                , vfixupimmps(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F36D08544C2B0801"            , vfixupimmps(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D08544C2B0801"            , vfixupimmps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D2854CB01"                , vfixupimmps(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("62F36D28544C2B0401"            , vfixupimmps(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D28544C2B0401"            , vfixupimmps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D4854CB01"                , vfixupimmps(zmm1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F36D48544C2B0201"            , vfixupimmps(zmm1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D48544C2B0201"            , vfixupimmps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED0855CB01"                , vfixupimmsd(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F3ED08554C2B1001"            , vfixupimmsd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED08554C2B1001"            , vfixupimmsd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D0855CB01"                , vfixupimmss(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F36D08554C2B2001"            , vfixupimmss(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D08554C2B2001"            , vfixupimmss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F2ED4898CB"                  , vfmadd132pd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48984C2B02"              , vfmadd132pd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48984C2B02"              , vfmadd132pd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D4898CB"                  , vfmadd132ps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48984C2B02"              , vfmadd132ps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48984C2B02"              , vfmadd132ps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48A8CB"                  , vfmadd213pd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48A84C2B02"              , vfmadd213pd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48A84C2B02"              , vfmadd213pd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48A8CB"                  , vfmadd213ps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48A84C2B02"              , vfmadd213ps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48A84C2B02"              , vfmadd213ps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48B8CB"                  , vfmadd231pd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48B84C2B02"              , vfmadd231pd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48B84C2B02"              , vfmadd231pd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48B8CB"                  , vfmadd231ps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48B84C2B02"              , vfmadd231ps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48B84C2B02"              , vfmadd231ps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED4896CB"                  , vfmaddsub132pd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48964C2B02"              , vfmaddsub132pd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48964C2B02"              , vfmaddsub132pd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D4896CB"                  , vfmaddsub132ps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48964C2B02"              , vfmaddsub132ps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48964C2B02"              , vfmaddsub132ps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48A6CB"                  , vfmaddsub213pd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48A64C2B02"              , vfmaddsub213pd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48A64C2B02"              , vfmaddsub213pd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48A6CB"                  , vfmaddsub213ps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48A64C2B02"              , vfmaddsub213ps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48A64C2B02"              , vfmaddsub213ps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48B6CB"                  , vfmaddsub231pd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48B64C2B02"              , vfmaddsub231pd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48B64C2B02"              , vfmaddsub231pd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48B6CB"                  , vfmaddsub231ps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48B64C2B02"              , vfmaddsub231ps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48B64C2B02"              , vfmaddsub231ps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED489ACB"                  , vfmsub132pd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED489A4C2B02"              , vfmsub132pd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED489A4C2B02"              , vfmsub132pd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D489ACB"                  , vfmsub132ps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D489A4C2B02"              , vfmsub132ps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D489A4C2B02"              , vfmsub132ps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48AACB"                  , vfmsub213pd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48AA4C2B02"              , vfmsub213pd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48AA4C2B02"              , vfmsub213pd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48AACB"                  , vfmsub213ps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48AA4C2B02"              , vfmsub213ps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48AA4C2B02"              , vfmsub213ps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48BACB"                  , vfmsub231pd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48BA4C2B02"              , vfmsub231pd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48BA4C2B02"              , vfmsub231pd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48BACB"                  , vfmsub231ps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48BA4C2B02"              , vfmsub231ps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48BA4C2B02"              , vfmsub231ps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED4897CB"                  , vfmsubadd132pd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48974C2B02"              , vfmsubadd132pd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48974C2B02"              , vfmsubadd132pd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D4897CB"                  , vfmsubadd132ps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48974C2B02"              , vfmsubadd132ps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48974C2B02"              , vfmsubadd132ps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48A7CB"                  , vfmsubadd213pd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48A74C2B02"              , vfmsubadd213pd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48A74C2B02"              , vfmsubadd213pd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48A7CB"                  , vfmsubadd213ps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48A74C2B02"              , vfmsubadd213ps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48A74C2B02"              , vfmsubadd213ps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48B7CB"                  , vfmsubadd231pd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48B74C2B02"              , vfmsubadd231pd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48B74C2B02"              , vfmsubadd231pd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48B7CB"                  , vfmsubadd231ps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48B74C2B02"              , vfmsubadd231ps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48B74C2B02"              , vfmsubadd231ps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED489CCB"                  , vfnmadd132pd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED489C4C2B02"              , vfnmadd132pd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED489C4C2B02"              , vfnmadd132pd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D489CCB"                  , vfnmadd132ps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D489C4C2B02"              , vfnmadd132ps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D489C4C2B02"              , vfnmadd132ps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48ACCB"                  , vfnmadd213pd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48AC4C2B02"              , vfnmadd213pd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48AC4C2B02"              , vfnmadd213pd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48ACCB"                  , vfnmadd213ps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48AC4C2B02"              , vfnmadd213ps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48AC4C2B02"              , vfnmadd213ps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48BCCB"                  , vfnmadd231pd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48BC4C2B02"              , vfnmadd231pd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48BC4C2B02"              , vfnmadd231pd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48BCCB"                  , vfnmadd231ps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48BC4C2B02"              , vfnmadd231ps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48BC4C2B02"              , vfnmadd231ps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED489ECB"                  , vfnmsub132pd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED489E4C2B02"              , vfnmsub132pd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED489E4C2B02"              , vfnmsub132pd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D489ECB"                  , vfnmsub132ps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D489E4C2B02"              , vfnmsub132ps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D489E4C2B02"              , vfnmsub132ps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48AECB"                  , vfnmsub213pd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48AE4C2B02"              , vfnmsub213pd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48AE4C2B02"              , vfnmsub213pd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48AECB"                  , vfnmsub213ps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48AE4C2B02"              , vfnmsub213ps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48AE4C2B02"              , vfnmsub213ps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48BECB"                  , vfnmsub231pd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48BE4C2B02"              , vfnmsub231pd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48BE4C2B02"              , vfnmsub231pd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48BECB"                  , vfnmsub231ps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48BE4C2B02"              , vfnmsub231ps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48BE4C2B02"              , vfnmsub231ps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F3FD0866CA01"                , vfpclasspd(k1, xmm2, 1));
  TEST_INSTRUCTION("62F3FD2866CA01"                , vfpclasspd(k1, ymm2, 1));
  TEST_INSTRUCTION("62F3FD4866CA01"                , vfpclasspd(k1, zmm2, 1));
  TEST_INSTRUCTION("62F3FD08664C1A0801"            , vfpclasspd(k1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F3FD28664C1A0401"            , vfpclasspd(k1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F3FD48664C1A0201"            , vfpclasspd(k1, zmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F37D0866CA01"                , vfpclassps(k1, xmm2, 1));
  TEST_INSTRUCTION("62F37D2866CA01"                , vfpclassps(k1, ymm2, 1));
  TEST_INSTRUCTION("62F37D4866CA01"                , vfpclassps(k1, zmm2, 1));
  TEST_INSTRUCTION("62F37D08664C1A0801"            , vfpclassps(k1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F37D28664C1A0401"            , vfpclassps(k1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F37D48664C1A0201"            , vfpclassps(k1, zmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F3FD0867CA01"                , vfpclasssd(k1, xmm2, 1));
  TEST_INSTRUCTION("62F3FD08674C1A1001"            , vfpclasssd(k1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F3FD08674C1A1001"            , vfpclasssd(k1, qword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F37D0867CA01"                , vfpclassss(k1, xmm2, 1));
  TEST_INSTRUCTION("62F37D08674C1A2001"            , vfpclassss(k1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F37D08674C1A2001"            , vfpclassss(k1, dword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F2FD09924C1A10"              , k(k1).vgatherdpd(xmm1, ptr(edx, xmm3, 0, 128)));
  TEST_INSTRUCTION("62F2FD29924C1A10"              , k(k1).vgatherdpd(ymm1, ptr(edx, xmm3, 0, 128)));
  TEST_INSTRUCTION("62F2FD49924C1A10"              , k(k1).vgatherdpd(zmm1, ptr(edx, ymm3, 0, 128)));
  TEST_INSTRUCTION("62F27D09924C1A20"              , k(k1).vgatherdps(xmm1, ptr(edx, xmm3, 0, 128)));
  TEST_INSTRUCTION("62F27D29924C1A20"              , k(k1).vgatherdps(ymm1, ptr(edx, ymm3, 0, 128)));
  TEST_INSTRUCTION("62F27D49924C1A20"              , k(k1).vgatherdps(zmm1, ptr(edx, zmm3, 0, 128)));
  TEST_INSTRUCTION("62F2FD49C64C1110"              , k(k1).vgatherpf0dpd(ptr(ecx, ymm2, 0, 128)));
  TEST_INSTRUCTION("62F27D49C64C1120"              , k(k1).vgatherpf0dps(ptr(ecx, zmm2, 0, 128)));
  TEST_INSTRUCTION("62F2FD49C74C1110"              , k(k1).vgatherpf0qpd(ptr(ecx, zmm2, 0, 128)));
  TEST_INSTRUCTION("62F27D49C74C1120"              , k(k1).vgatherpf0qps(ptr(ecx, zmm2, 0, 128)));
  TEST_INSTRUCTION("62F2FD49C6541110"              , k(k1).vgatherpf1dpd(ptr(ecx, ymm2, 0, 128)));
  TEST_INSTRUCTION("62F27D49C6541120"              , k(k1).vgatherpf1dps(ptr(ecx, zmm2, 0, 128)));
  TEST_INSTRUCTION("62F2FD49C7541110"              , k(k1).vgatherpf1qpd(ptr(ecx, zmm2, 0, 128)));
  TEST_INSTRUCTION("62F27D49C7541120"              , k(k1).vgatherpf1qps(ptr(ecx, zmm2, 0, 128)));
  TEST_INSTRUCTION("62F2FD09934C1A10"              , k(k1).vgatherqpd(xmm1, ptr(edx, xmm3, 0, 128)));
  TEST_INSTRUCTION("62F2FD29934C1A10"              , k(k1).vgatherqpd(ymm1, ptr(edx, ymm3, 0, 128)));
  TEST_INSTRUCTION("62F2FD49934C1A10"              , k(k1).vgatherqpd(zmm1, ptr(edx, zmm3, 0, 128)));
  TEST_INSTRUCTION("62F27D09934C1A20"              , k(k1).vgatherqps(xmm1, ptr(edx, xmm3, 0, 128)));
  TEST_INSTRUCTION("62F27D29934C1A20"              , k(k1).vgatherqps(xmm1, ptr(edx, ymm3, 0, 128)));
  TEST_INSTRUCTION("62F27D49934C1A20"              , k(k1).vgatherqps(ymm1, ptr(edx, zmm3, 0, 128)));
  TEST_INSTRUCTION("62F2FD0842CA"                  , vgetexppd(xmm1, xmm2));
  TEST_INSTRUCTION("62F2FD08424C1A08"              , vgetexppd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD08424C1A08"              , vgetexppd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD2842CA"                  , vgetexppd(ymm1, ymm2));
  TEST_INSTRUCTION("62F2FD28424C1A04"              , vgetexppd(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD28424C1A04"              , vgetexppd(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD4842CA"                  , vgetexppd(zmm1, zmm2));
  TEST_INSTRUCTION("62F2FD48424C1A02"              , vgetexppd(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD48424C1A02"              , vgetexppd(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D0842CA"                  , vgetexpps(xmm1, xmm2));
  TEST_INSTRUCTION("62F27D08424C1A08"              , vgetexpps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D08424C1A08"              , vgetexpps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D2842CA"                  , vgetexpps(ymm1, ymm2));
  TEST_INSTRUCTION("62F27D28424C1A04"              , vgetexpps(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D28424C1A04"              , vgetexpps(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D4842CA"                  , vgetexpps(zmm1, zmm2));
  TEST_INSTRUCTION("62F27D48424C1A02"              , vgetexpps(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48424C1A02"              , vgetexpps(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2ED0843CB"                  , vgetexpsd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED08434C2B10"              , vgetexpsd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED08434C2B10"              , vgetexpsd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D0843CB"                  , vgetexpss(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26D08434C2B20"              , vgetexpss(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D08434C2B20"              , vgetexpss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F3FD0826CA01"                , vgetmantpd(xmm1, xmm2, 1));
  TEST_INSTRUCTION("62F3FD08264C1A0801"            , vgetmantpd(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F3FD08264C1A0801"            , vgetmantpd(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F3FD2826CA01"                , vgetmantpd(ymm1, ymm2, 1));
  TEST_INSTRUCTION("62F3FD28264C1A0401"            , vgetmantpd(ymm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F3FD28264C1A0401"            , vgetmantpd(ymm1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F3FD4826CA01"                , vgetmantpd(zmm1, zmm2, 1));
  TEST_INSTRUCTION("62F3FD48264C1A0201"            , vgetmantpd(zmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F3FD48264C1A0201"            , vgetmantpd(zmm1, zmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F37D0826CA01"                , vgetmantps(xmm1, xmm2, 1));
  TEST_INSTRUCTION("62F37D08264C1A0801"            , vgetmantps(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F37D08264C1A0801"            , vgetmantps(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F37D2826CA01"                , vgetmantps(ymm1, ymm2, 1));
  TEST_INSTRUCTION("62F37D28264C1A0401"            , vgetmantps(ymm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F37D28264C1A0401"            , vgetmantps(ymm1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F37D4826CA01"                , vgetmantps(zmm1, zmm2, 1));
  TEST_INSTRUCTION("62F37D48264C1A0201"            , vgetmantps(zmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F37D48264C1A0201"            , vgetmantps(zmm1, zmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED0827CB01"                , vgetmantsd(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F3ED08274C2B1001"            , vgetmantsd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED08274C2B1001"            , vgetmantsd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D0827CB01"                , vgetmantss(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F36D08274C2B2001"            , vgetmantss(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D08274C2B2001"            , vgetmantss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED48CFCB01"                , vgf2p8affineinvqb(zmm1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F3ED48CF4C2B0201"            , vgf2p8affineinvqb(zmm1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED48CF4C2B0201"            , vgf2p8affineinvqb(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED48CECB01"                , vgf2p8affineqb(zmm1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F3ED48CE4C2B0201"            , vgf2p8affineqb(zmm1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED48CE4C2B0201"            , vgf2p8affineqb(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F26D48CFCB"                  , vgf2p8mulb(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48CF4C2B02"              , vgf2p8mulb(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48CF4C2B02"              , vgf2p8mulb(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F36D2818CB01"                , vinsertf32x4(ymm1, ymm2, xmm3, 1));
  TEST_INSTRUCTION("62F36D28184C2B0801"            , vinsertf32x4(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D28184C2B0801"            , vinsertf32x4(ymm1, ymm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D4818CB01"                , vinsertf32x4(zmm1, zmm2, xmm3, 1));
  TEST_INSTRUCTION("62F36D48184C2B0801"            , vinsertf32x4(zmm1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D48184C2B0801"            , vinsertf32x4(zmm1, zmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D481ACB01"                , vinsertf32x8(zmm1, zmm2, ymm3, 1));
  TEST_INSTRUCTION("62F36D481A4C2B0401"            , vinsertf32x8(zmm1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D481A4C2B0401"            , vinsertf32x8(zmm1, zmm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED2818CB01"                , vinsertf64x2(ymm1, ymm2, xmm3, 1));
  TEST_INSTRUCTION("62F3ED28184C2B0801"            , vinsertf64x2(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED28184C2B0801"            , vinsertf64x2(ymm1, ymm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED4818CB01"                , vinsertf64x2(zmm1, zmm2, xmm3, 1));
  TEST_INSTRUCTION("62F3ED48184C2B0801"            , vinsertf64x2(zmm1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED48184C2B0801"            , vinsertf64x2(zmm1, zmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED481ACB01"                , vinsertf64x4(zmm1, zmm2, ymm3, 1));
  TEST_INSTRUCTION("62F3ED481A4C2B0401"            , vinsertf64x4(zmm1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED481A4C2B0401"            , vinsertf64x4(zmm1, zmm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D2838CB01"                , vinserti32x4(ymm1, ymm2, xmm3, 1));
  TEST_INSTRUCTION("62F36D28384C2B0801"            , vinserti32x4(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D28384C2B0801"            , vinserti32x4(ymm1, ymm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D4838CB01"                , vinserti32x4(zmm1, zmm2, xmm3, 1));
  TEST_INSTRUCTION("62F36D48384C2B0801"            , vinserti32x4(zmm1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D48384C2B0801"            , vinserti32x4(zmm1, zmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D483ACB01"                , vinserti32x8(zmm1, zmm2, ymm3, 1));
  TEST_INSTRUCTION("62F36D483A4C2B0401"            , vinserti32x8(zmm1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D483A4C2B0401"            , vinserti32x8(zmm1, zmm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED2838CB01"                , vinserti64x2(ymm1, ymm2, xmm3, 1));
  TEST_INSTRUCTION("62F3ED28384C2B0801"            , vinserti64x2(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED28384C2B0801"            , vinserti64x2(ymm1, ymm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED4838CB01"                , vinserti64x2(zmm1, zmm2, xmm3, 1));
  TEST_INSTRUCTION("62F3ED48384C2B0801"            , vinserti64x2(zmm1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED48384C2B0801"            , vinserti64x2(zmm1, zmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED483ACB01"                , vinserti64x4(zmm1, zmm2, ymm3, 1));
  TEST_INSTRUCTION("62F3ED483A4C2B0401"            , vinserti64x4(zmm1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED483A4C2B0401"            , vinserti64x4(zmm1, zmm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F1ED485FCB"                  , vmaxpd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F1ED485F4C2B02"              , vmaxpd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED485F4C2B02"              , vmaxpd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16C485FCB"                  , vmaxps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16C485F4C2B02"              , vmaxps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16C485F4C2B02"              , vmaxps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED485DCB"                  , vminpd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F1ED485D4C2B02"              , vminpd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED485D4C2B02"              , vminpd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16C485DCB"                  , vminps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16C485D4C2B02"              , vminps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16C485D4C2B02"              , vminps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1FD4828CA"                  , vmovapd(zmm1, zmm2));
  TEST_INSTRUCTION("62F1FD48284C1A02"              , vmovapd(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD48284C1A02"              , vmovapd(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD48295C1102"              , vmovapd(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F1FD48295C1102"              , vmovapd(zmmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F17C4828CA"                  , vmovaps(zmm1, zmm2));
  TEST_INSTRUCTION("62F17C48284C1A02"              , vmovaps(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17C48284C1A02"              , vmovaps(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17C48295C1102"              , vmovaps(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F17C48295C1102"              , vmovaps(zmmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F1FF4812CA"                  , vmovddup(zmm1, zmm2));
  TEST_INSTRUCTION("62F1FF48124C1A02"              , vmovddup(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FF48124C1A02"              , vmovddup(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D086FCA"                  , vmovdqa32(xmm1, xmm2));
  TEST_INSTRUCTION("62F17D086F4C1A08"              , vmovdqa32(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D086F4C1A08"              , vmovdqa32(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D087F5C1108"              , vmovdqa32(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F17D087F5C1108"              , vmovdqa32(xmmword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F17D286FCA"                  , vmovdqa32(ymm1, ymm2));
  TEST_INSTRUCTION("62F17D286F4C1A04"              , vmovdqa32(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D286F4C1A04"              , vmovdqa32(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D287F5C1104"              , vmovdqa32(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F17D287F5C1104"              , vmovdqa32(ymmword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F17D486FCA"                  , vmovdqa32(zmm1, zmm2));
  TEST_INSTRUCTION("62F17D486F4C1A02"              , vmovdqa32(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D486F4C1A02"              , vmovdqa32(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17D487F5C1102"              , vmovdqa32(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F17D487F5C1102"              , vmovdqa32(zmmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F1FD086FCA"                  , vmovdqa64(xmm1, xmm2));
  TEST_INSTRUCTION("62F1FD086F4C1A08"              , vmovdqa64(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD086F4C1A08"              , vmovdqa64(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD087F5C1108"              , vmovdqa64(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F1FD087F5C1108"              , vmovdqa64(xmmword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F1FD286FCA"                  , vmovdqa64(ymm1, ymm2));
  TEST_INSTRUCTION("62F1FD286F4C1A04"              , vmovdqa64(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD286F4C1A04"              , vmovdqa64(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD287F5C1104"              , vmovdqa64(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F1FD287F5C1104"              , vmovdqa64(ymmword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F1FD486FCA"                  , vmovdqa64(zmm1, zmm2));
  TEST_INSTRUCTION("62F1FD486F4C1A02"              , vmovdqa64(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD486F4C1A02"              , vmovdqa64(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD487F5C1102"              , vmovdqa64(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F1FD487F5C1102"              , vmovdqa64(zmmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F1FF086FCA"                  , vmovdqu16(xmm1, xmm2));
  TEST_INSTRUCTION("62F1FF086F4C1A08"              , vmovdqu16(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FF086F4C1A08"              , vmovdqu16(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FF087F5C1108"              , vmovdqu16(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F1FF087F5C1108"              , vmovdqu16(xmmword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F1FF286FCA"                  , vmovdqu16(ymm1, ymm2));
  TEST_INSTRUCTION("62F1FF286F4C1A04"              , vmovdqu16(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FF286F4C1A04"              , vmovdqu16(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FF287F5C1104"              , vmovdqu16(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F1FF287F5C1104"              , vmovdqu16(ymmword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F1FF486FCA"                  , vmovdqu16(zmm1, zmm2));
  TEST_INSTRUCTION("62F1FF486F4C1A02"              , vmovdqu16(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FF486F4C1A02"              , vmovdqu16(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FF487F5C1102"              , vmovdqu16(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F1FF487F5C1102"              , vmovdqu16(zmmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F17E086FCA"                  , vmovdqu32(xmm1, xmm2));
  TEST_INSTRUCTION("62F17E086F4C1A08"              , vmovdqu32(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17E086F4C1A08"              , vmovdqu32(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17E087F5C1108"              , vmovdqu32(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F17E087F5C1108"              , vmovdqu32(xmmword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F17E286FCA"                  , vmovdqu32(ymm1, ymm2));
  TEST_INSTRUCTION("62F17E286F4C1A04"              , vmovdqu32(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17E286F4C1A04"              , vmovdqu32(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17E287F5C1104"              , vmovdqu32(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F17E287F5C1104"              , vmovdqu32(ymmword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F17E486FCA"                  , vmovdqu32(zmm1, zmm2));
  TEST_INSTRUCTION("62F17E486F4C1A02"              , vmovdqu32(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17E486F4C1A02"              , vmovdqu32(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17E487F5C1102"              , vmovdqu32(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F17E487F5C1102"              , vmovdqu32(zmmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F1FE086FCA"                  , vmovdqu64(xmm1, xmm2));
  TEST_INSTRUCTION("62F1FE086F4C1A08"              , vmovdqu64(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FE086F4C1A08"              , vmovdqu64(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FE087F5C1108"              , vmovdqu64(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F1FE087F5C1108"              , vmovdqu64(xmmword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F1FE286FCA"                  , vmovdqu64(ymm1, ymm2));
  TEST_INSTRUCTION("62F1FE286F4C1A04"              , vmovdqu64(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FE286F4C1A04"              , vmovdqu64(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FE287F5C1104"              , vmovdqu64(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F1FE287F5C1104"              , vmovdqu64(ymmword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F1FE486FCA"                  , vmovdqu64(zmm1, zmm2));
  TEST_INSTRUCTION("62F1FE486F4C1A02"              , vmovdqu64(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FE486F4C1A02"              , vmovdqu64(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FE487F5C1102"              , vmovdqu64(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F1FE487F5C1102"              , vmovdqu64(zmmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F17F086FCA"                  , vmovdqu8(xmm1, xmm2));
  TEST_INSTRUCTION("62F17F086F4C1A08"              , vmovdqu8(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17F086F4C1A08"              , vmovdqu8(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17F087F5C1108"              , vmovdqu8(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F17F087F5C1108"              , vmovdqu8(xmmword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F17F286FCA"                  , vmovdqu8(ymm1, ymm2));
  TEST_INSTRUCTION("62F17F286F4C1A04"              , vmovdqu8(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17F286F4C1A04"              , vmovdqu8(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17F287F5C1104"              , vmovdqu8(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F17F287F5C1104"              , vmovdqu8(ymmword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F17F486FCA"                  , vmovdqu8(zmm1, zmm2));
  TEST_INSTRUCTION("62F17F486F4C1A02"              , vmovdqu8(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17F486F4C1A02"              , vmovdqu8(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17F487F5C1102"              , vmovdqu8(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F17F487F5C1102"              , vmovdqu8(zmmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F17D48E75C1102"              , vmovntdq(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F17D48E75C1102"              , vmovntdq(zmmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27D482A4C1A02"              , vmovntdqa(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D482A4C1A02"              , vmovntdqa(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD482B5C1102"              , vmovntpd(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F1FD482B5C1102"              , vmovntpd(zmmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F17C482B5C1102"              , vmovntps(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F17C482B5C1102"              , vmovntps(zmmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F17E4816CA"                  , vmovshdup(zmm1, zmm2));
  TEST_INSTRUCTION("62F17E48164C1A02"              , vmovshdup(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17E48164C1A02"              , vmovshdup(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17E4812CA"                  , vmovsldup(zmm1, zmm2));
  TEST_INSTRUCTION("62F17E48124C1A02"              , vmovsldup(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17E48124C1A02"              , vmovsldup(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD4810CA"                  , vmovupd(zmm1, zmm2));
  TEST_INSTRUCTION("62F1FD48104C1A02"              , vmovupd(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD48104C1A02"              , vmovupd(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD48115C1102"              , vmovupd(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F1FD48115C1102"              , vmovupd(zmmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F17C4810CA"                  , vmovups(zmm1, zmm2));
  TEST_INSTRUCTION("62F17C48104C1A02"              , vmovups(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17C48104C1A02"              , vmovups(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17C48115C1102"              , vmovups(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F17C48115C1102"              , vmovups(zmmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F1ED4859CB"                  , vmulpd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F1ED48594C2B02"              , vmulpd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED48594C2B02"              , vmulpd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16C4859CB"                  , vmulps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16C48594C2B02"              , vmulps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16C48594C2B02"              , vmulps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED4856CB"                  , vorpd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F1ED48564C2B02"              , vorpd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED48564C2B02"              , vorpd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16C4856CB"                  , vorps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16C48564C2B02"              , vorps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16C48564C2B02"              , vorps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F25F0868D1"                  , vp2intersectd(k2, k3, xmm4, xmm1));
  TEST_INSTRUCTION("62F25F0868541108"              , vp2intersectd(k2, k3, xmm4, ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("62F25F0868541108"              , vp2intersectd(k2, k3, xmm4, xmmword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("62F25F2868D1"                  , vp2intersectd(k2, k3, ymm4, ymm1));
  TEST_INSTRUCTION("62F25F2868541104"              , vp2intersectd(k2, k3, ymm4, ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("62F25F2868541104"              , vp2intersectd(k2, k3, ymm4, ymmword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("62F25F4868D1"                  , vp2intersectd(k2, k3, zmm4, zmm1));
  TEST_INSTRUCTION("62F25F4868541102"              , vp2intersectd(k2, k3, zmm4, ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("62F25F4868541102"              , vp2intersectd(k2, k3, zmm4, zmmword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("62F2DF0868D1"                  , vp2intersectq(k2, k3, xmm4, xmm1));
  TEST_INSTRUCTION("62F2DF0868541108"              , vp2intersectq(k2, k3, xmm4, ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("62F2DF0868541108"              , vp2intersectq(k2, k3, xmm4, xmmword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("62F2DF2868D1"                  , vp2intersectq(k2, k3, ymm4, ymm1));
  TEST_INSTRUCTION("62F2DF2868541104"              , vp2intersectq(k2, k3, ymm4, ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("62F2DF2868541104"              , vp2intersectq(k2, k3, ymm4, ymmword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("62F2DF4868D1"                  , vp2intersectq(k2, k3, zmm4, zmm1));
  TEST_INSTRUCTION("62F2DF4868541102"              , vp2intersectq(k2, k3, zmm4, ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("62F2DF4868541102"              , vp2intersectq(k2, k3, zmm4, zmmword_ptr(ecx, edx, 0, 128)));
  TEST_INSTRUCTION("62F25F48524C1A08"              , vp4dpwssd(zmm1, zmm4, zmm5, zmm6, zmm7, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F25F48524C1A08"              , vp4dpwssd(zmm1, zmm4, zmm5, zmm6, zmm7, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F25F48534C1A08"              , vp4dpwssds(zmm1, zmm4, zmm5, zmm6, zmm7, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F25F48534C1A08"              , vp4dpwssds(zmm1, zmm4, zmm5, zmm6, zmm7, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D481CCA"                  , vpabsb(zmm1, zmm2));
  TEST_INSTRUCTION("62F27D481C4C1A02"              , vpabsb(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D481C4C1A02"              , vpabsb(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D481ECA"                  , vpabsd(zmm1, zmm2));
  TEST_INSTRUCTION("62F27D481E4C1A02"              , vpabsd(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D481E4C1A02"              , vpabsd(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD081FCA"                  , vpabsq(xmm1, xmm2));
  TEST_INSTRUCTION("62F2FD081F4C1A08"              , vpabsq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD081F4C1A08"              , vpabsq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD281FCA"                  , vpabsq(ymm1, ymm2));
  TEST_INSTRUCTION("62F2FD281F4C1A04"              , vpabsq(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD281F4C1A04"              , vpabsq(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD481FCA"                  , vpabsq(zmm1, zmm2));
  TEST_INSTRUCTION("62F2FD481F4C1A02"              , vpabsq(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD481F4C1A02"              , vpabsq(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D481DCA"                  , vpabsw(zmm1, zmm2));
  TEST_INSTRUCTION("62F27D481D4C1A02"              , vpabsw(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D481D4C1A02"              , vpabsw(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F16D486BCB"                  , vpackssdw(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D486B4C2B02"              , vpackssdw(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D486B4C2B02"              , vpackssdw(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D4863CB"                  , vpacksswb(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48634C2B02"              , vpacksswb(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48634C2B02"              , vpacksswb(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D482BCB"                  , vpackusdw(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D482B4C2B02"              , vpackusdw(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D482B4C2B02"              , vpackusdw(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D4867CB"                  , vpackuswb(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48674C2B02"              , vpackuswb(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48674C2B02"              , vpackuswb(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48FCCB"                  , vpaddb(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48FC4C2B02"              , vpaddb(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48FC4C2B02"              , vpaddb(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48FECB"                  , vpaddd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48FE4C2B02"              , vpaddd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48FE4C2B02"              , vpaddd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED48D4CB"                  , vpaddq(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F1ED48D44C2B02"              , vpaddq(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED48D44C2B02"              , vpaddq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48ECCB"                  , vpaddsb(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48EC4C2B02"              , vpaddsb(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48EC4C2B02"              , vpaddsb(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48EDCB"                  , vpaddsw(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48ED4C2B02"              , vpaddsw(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48ED4C2B02"              , vpaddsw(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48DCCB"                  , vpaddusb(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48DC4C2B02"              , vpaddusb(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48DC4C2B02"              , vpaddusb(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48DDCB"                  , vpaddusw(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48DD4C2B02"              , vpaddusw(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48DD4C2B02"              , vpaddusw(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48FDCB"                  , vpaddw(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48FD4C2B02"              , vpaddw(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48FD4C2B02"              , vpaddw(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F36D480FCB01"                , vpalignr(zmm1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F36D480F4C2B0201"            , vpalignr(zmm1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D480F4C2B0201"            , vpalignr(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F16D08DBCB"                  , vpandd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F16D08DB4C2B08"              , vpandd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D08DB4C2B08"              , vpandd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D28DBCB"                  , vpandd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F16D28DB4C2B04"              , vpandd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D28DB4C2B04"              , vpandd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48DBCB"                  , vpandd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48DB4C2B02"              , vpandd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48DB4C2B02"              , vpandd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D08DFCB"                  , vpandnd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F16D08DF4C2B08"              , vpandnd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D08DF4C2B08"              , vpandnd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D28DFCB"                  , vpandnd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F16D28DF4C2B04"              , vpandnd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D28DF4C2B04"              , vpandnd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48DFCB"                  , vpandnd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48DF4C2B02"              , vpandnd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48DF4C2B02"              , vpandnd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED08DFCB"                  , vpandnq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F1ED08DF4C2B08"              , vpandnq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED08DF4C2B08"              , vpandnq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED28DFCB"                  , vpandnq(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F1ED28DF4C2B04"              , vpandnq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED28DF4C2B04"              , vpandnq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED48DFCB"                  , vpandnq(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F1ED48DF4C2B02"              , vpandnq(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED48DF4C2B02"              , vpandnq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED08DBCB"                  , vpandq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F1ED08DB4C2B08"              , vpandq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED08DB4C2B08"              , vpandq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED28DBCB"                  , vpandq(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F1ED28DB4C2B04"              , vpandq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED28DB4C2B04"              , vpandq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED48DBCB"                  , vpandq(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F1ED48DB4C2B02"              , vpandq(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED48DB4C2B02"              , vpandq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48E0CB"                  , vpavgb(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48E04C2B02"              , vpavgb(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48E04C2B02"              , vpavgb(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48E3CB"                  , vpavgw(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48E34C2B02"              , vpavgw(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48E34C2B02"              , vpavgw(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D0866CB"                  , vpblendmb(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26D08664C2B08"              , vpblendmb(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D08664C2B08"              , vpblendmb(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D2866CB"                  , vpblendmb(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F26D28664C2B04"              , vpblendmb(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D28664C2B04"              , vpblendmb(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D4866CB"                  , vpblendmb(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48664C2B02"              , vpblendmb(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48664C2B02"              , vpblendmb(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D0864CB"                  , vpblendmd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26D08644C2B08"              , vpblendmd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D08644C2B08"              , vpblendmd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D2864CB"                  , vpblendmd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F26D28644C2B04"              , vpblendmd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D28644C2B04"              , vpblendmd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D4864CB"                  , vpblendmd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48644C2B02"              , vpblendmd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48644C2B02"              , vpblendmd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED0864CB"                  , vpblendmq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED08644C2B08"              , vpblendmq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED08644C2B08"              , vpblendmq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED2864CB"                  , vpblendmq(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED28644C2B04"              , vpblendmq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED28644C2B04"              , vpblendmq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED4864CB"                  , vpblendmq(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48644C2B02"              , vpblendmq(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48644C2B02"              , vpblendmq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED0866CB"                  , vpblendmw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED08664C2B08"              , vpblendmw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED08664C2B08"              , vpblendmw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED2866CB"                  , vpblendmw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED28664C2B04"              , vpblendmw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED28664C2B04"              , vpblendmw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED4866CB"                  , vpblendmw(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48664C2B02"              , vpblendmw(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48664C2B02"              , vpblendmw(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F27D087ACA"                  , vpbroadcastb(xmm1, edx));
  TEST_INSTRUCTION("62F27D287ACA"                  , vpbroadcastb(ymm1, edx));
  TEST_INSTRUCTION("62F27D487ACA"                  , vpbroadcastb(zmm1, edx));
  TEST_INSTRUCTION("62F27D4878CA"                  , vpbroadcastb(zmm1, xmm2));
  TEST_INSTRUCTION("62F27D48788C1A80000000"        , vpbroadcastb(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48788C1A80000000"        , vpbroadcastb(zmm1, byte_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D087CCA"                  , vpbroadcastd(xmm1, edx));
  TEST_INSTRUCTION("62F27D287CCA"                  , vpbroadcastd(ymm1, edx));
  TEST_INSTRUCTION("62F27D487CCA"                  , vpbroadcastd(zmm1, edx));
  TEST_INSTRUCTION("62F27D4858CA"                  , vpbroadcastd(zmm1, xmm2));
  TEST_INSTRUCTION("62F27D48584C1A20"              , vpbroadcastd(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48584C1A20"              , vpbroadcastd(zmm1, dword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FE082ACA"                  , vpbroadcastmb2q(xmm1, k2));
  TEST_INSTRUCTION("62F2FE282ACA"                  , vpbroadcastmb2q(ymm1, k2));
  TEST_INSTRUCTION("62F2FE482ACA"                  , vpbroadcastmb2q(zmm1, k2));
  TEST_INSTRUCTION("62F27E083ACA"                  , vpbroadcastmw2d(xmm1, k2));
  TEST_INSTRUCTION("62F27E283ACA"                  , vpbroadcastmw2d(ymm1, k2));
  TEST_INSTRUCTION("62F27E483ACA"                  , vpbroadcastmw2d(zmm1, k2));
  TEST_INSTRUCTION("62F2FD4859CA"                  , vpbroadcastq(zmm1, xmm2));
  TEST_INSTRUCTION("62F2FD48594C1A10"              , vpbroadcastq(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD48594C1A10"              , vpbroadcastq(zmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D087BCA"                  , vpbroadcastw(xmm1, edx));
  TEST_INSTRUCTION("62F27D287BCA"                  , vpbroadcastw(ymm1, edx));
  TEST_INSTRUCTION("62F27D487BCA"                  , vpbroadcastw(zmm1, edx));
  TEST_INSTRUCTION("62F27D4879CA"                  , vpbroadcastw(zmm1, xmm2));
  TEST_INSTRUCTION("62F27D48794C1A40"              , vpbroadcastw(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48794C1A40"              , vpbroadcastw(zmm1, word_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F36D4844CB01"                , vpclmulqdq(zmm1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F36D48444C2B0201"            , vpclmulqdq(zmm1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D48444C2B0201"            , vpclmulqdq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D083FCB01"                , vpcmpb(k1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F36D083F4C2B0801"            , vpcmpb(k1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D083F4C2B0801"            , vpcmpb(k1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D283FCB01"                , vpcmpb(k1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("62F36D283F4C2B0401"            , vpcmpb(k1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D283F4C2B0401"            , vpcmpb(k1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D483FCB01"                , vpcmpb(k1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F36D483F4C2B0201"            , vpcmpb(k1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D483F4C2B0201"            , vpcmpb(k1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D081FCB01"                , vpcmpd(k1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F36D081F4C2B0801"            , vpcmpd(k1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D081F4C2B0801"            , vpcmpd(k1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D281FCB01"                , vpcmpd(k1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("62F36D281F4C2B0401"            , vpcmpd(k1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D281F4C2B0401"            , vpcmpd(k1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D481FCB01"                , vpcmpd(k1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F36D481F4C2B0201"            , vpcmpd(k1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D481F4C2B0201"            , vpcmpd(k1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F16D0874CB"                  , vpcmpeqb(k1, xmm2, xmm3));
  TEST_INSTRUCTION("62F16D08744C2B08"              , vpcmpeqb(k1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D08744C2B08"              , vpcmpeqb(k1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D2874CB"                  , vpcmpeqb(k1, ymm2, ymm3));
  TEST_INSTRUCTION("62F16D28744C2B04"              , vpcmpeqb(k1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D28744C2B04"              , vpcmpeqb(k1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D4874CB"                  , vpcmpeqb(k1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48744C2B02"              , vpcmpeqb(k1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48744C2B02"              , vpcmpeqb(k1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D0876CB"                  , vpcmpeqd(k1, xmm2, xmm3));
  TEST_INSTRUCTION("62F16D08764C2B08"              , vpcmpeqd(k1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D08764C2B08"              , vpcmpeqd(k1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D2876CB"                  , vpcmpeqd(k1, ymm2, ymm3));
  TEST_INSTRUCTION("62F16D28764C2B04"              , vpcmpeqd(k1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D28764C2B04"              , vpcmpeqd(k1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D4876CB"                  , vpcmpeqd(k1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48764C2B02"              , vpcmpeqd(k1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48764C2B02"              , vpcmpeqd(k1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED0829CB"                  , vpcmpeqq(k1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED08294C2B08"              , vpcmpeqq(k1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED08294C2B08"              , vpcmpeqq(k1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED2829CB"                  , vpcmpeqq(k1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED28294C2B04"              , vpcmpeqq(k1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED28294C2B04"              , vpcmpeqq(k1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED4829CB"                  , vpcmpeqq(k1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48294C2B02"              , vpcmpeqq(k1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48294C2B02"              , vpcmpeqq(k1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D0875CB"                  , vpcmpeqw(k1, xmm2, xmm3));
  TEST_INSTRUCTION("62F16D08754C2B08"              , vpcmpeqw(k1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D08754C2B08"              , vpcmpeqw(k1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D2875CB"                  , vpcmpeqw(k1, ymm2, ymm3));
  TEST_INSTRUCTION("62F16D28754C2B04"              , vpcmpeqw(k1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D28754C2B04"              , vpcmpeqw(k1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D4875CB"                  , vpcmpeqw(k1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48754C2B02"              , vpcmpeqw(k1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48754C2B02"              , vpcmpeqw(k1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D0864CB"                  , vpcmpgtb(k1, xmm2, xmm3));
  TEST_INSTRUCTION("62F16D08644C2B08"              , vpcmpgtb(k1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D08644C2B08"              , vpcmpgtb(k1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D2864CB"                  , vpcmpgtb(k1, ymm2, ymm3));
  TEST_INSTRUCTION("62F16D28644C2B04"              , vpcmpgtb(k1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D28644C2B04"              , vpcmpgtb(k1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D4864CB"                  , vpcmpgtb(k1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48644C2B02"              , vpcmpgtb(k1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48644C2B02"              , vpcmpgtb(k1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D0866CB"                  , vpcmpgtd(k1, xmm2, xmm3));
  TEST_INSTRUCTION("62F16D08664C2B08"              , vpcmpgtd(k1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D08664C2B08"              , vpcmpgtd(k1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D2866CB"                  , vpcmpgtd(k1, ymm2, ymm3));
  TEST_INSTRUCTION("62F16D28664C2B04"              , vpcmpgtd(k1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D28664C2B04"              , vpcmpgtd(k1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D4866CB"                  , vpcmpgtd(k1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48664C2B02"              , vpcmpgtd(k1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48664C2B02"              , vpcmpgtd(k1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED0837CB"                  , vpcmpgtq(k1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED08374C2B08"              , vpcmpgtq(k1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED08374C2B08"              , vpcmpgtq(k1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED2837CB"                  , vpcmpgtq(k1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED28374C2B04"              , vpcmpgtq(k1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED28374C2B04"              , vpcmpgtq(k1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED4837CB"                  , vpcmpgtq(k1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48374C2B02"              , vpcmpgtq(k1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48374C2B02"              , vpcmpgtq(k1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D0865CB"                  , vpcmpgtw(k1, xmm2, xmm3));
  TEST_INSTRUCTION("62F16D08654C2B08"              , vpcmpgtw(k1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D08654C2B08"              , vpcmpgtw(k1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D2865CB"                  , vpcmpgtw(k1, ymm2, ymm3));
  TEST_INSTRUCTION("62F16D28654C2B04"              , vpcmpgtw(k1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D28654C2B04"              , vpcmpgtw(k1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D4865CB"                  , vpcmpgtw(k1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48654C2B02"              , vpcmpgtw(k1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48654C2B02"              , vpcmpgtw(k1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F3ED081FCB01"                , vpcmpq(k1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F3ED081F4C2B0801"            , vpcmpq(k1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED081F4C2B0801"            , vpcmpq(k1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED281FCB01"                , vpcmpq(k1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("62F3ED281F4C2B0401"            , vpcmpq(k1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED281F4C2B0401"            , vpcmpq(k1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED481FCB01"                , vpcmpq(k1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F3ED481F4C2B0201"            , vpcmpq(k1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED481F4C2B0201"            , vpcmpq(k1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D083ECB01"                , vpcmpub(k1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F36D083E4C2B0801"            , vpcmpub(k1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D083E4C2B0801"            , vpcmpub(k1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D283ECB01"                , vpcmpub(k1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("62F36D283E4C2B0401"            , vpcmpub(k1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D283E4C2B0401"            , vpcmpub(k1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D483ECB01"                , vpcmpub(k1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F36D483E4C2B0201"            , vpcmpub(k1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D483E4C2B0201"            , vpcmpub(k1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D081ECB01"                , vpcmpud(k1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F36D081E4C2B0801"            , vpcmpud(k1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D081E4C2B0801"            , vpcmpud(k1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D281ECB01"                , vpcmpud(k1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("62F36D281E4C2B0401"            , vpcmpud(k1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D281E4C2B0401"            , vpcmpud(k1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D481ECB01"                , vpcmpud(k1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F36D481E4C2B0201"            , vpcmpud(k1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D481E4C2B0201"            , vpcmpud(k1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED081ECB01"                , vpcmpuq(k1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F3ED081E4C2B0801"            , vpcmpuq(k1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED081E4C2B0801"            , vpcmpuq(k1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED281ECB01"                , vpcmpuq(k1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("62F3ED281E4C2B0401"            , vpcmpuq(k1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED281E4C2B0401"            , vpcmpuq(k1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED481ECB01"                , vpcmpuq(k1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F3ED481E4C2B0201"            , vpcmpuq(k1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED481E4C2B0201"            , vpcmpuq(k1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED083ECB01"                , vpcmpuw(k1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F3ED083E4C2B0801"            , vpcmpuw(k1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED083E4C2B0801"            , vpcmpuw(k1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED283ECB01"                , vpcmpuw(k1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("62F3ED283E4C2B0401"            , vpcmpuw(k1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED283E4C2B0401"            , vpcmpuw(k1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED483ECB01"                , vpcmpuw(k1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F3ED483E4C2B0201"            , vpcmpuw(k1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED483E4C2B0201"            , vpcmpuw(k1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED083FCB01"                , vpcmpw(k1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F3ED083F4C2B0801"            , vpcmpw(k1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED083F4C2B0801"            , vpcmpw(k1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED283FCB01"                , vpcmpw(k1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("62F3ED283F4C2B0401"            , vpcmpw(k1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED283F4C2B0401"            , vpcmpw(k1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED483FCB01"                , vpcmpw(k1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F3ED483F4C2B0201"            , vpcmpw(k1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED483F4C2B0201"            , vpcmpw(k1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F27D0863D1"                  , vpcompressb(xmm1, xmm2));
  TEST_INSTRUCTION("62F27D08639C1180000000"        , vpcompressb(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27D08639C1180000000"        , vpcompressb(xmmword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27D2863D1"                  , vpcompressb(ymm1, ymm2));
  TEST_INSTRUCTION("62F27D28639C1180000000"        , vpcompressb(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27D28639C1180000000"        , vpcompressb(ymmword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27D4863D1"                  , vpcompressb(zmm1, zmm2));
  TEST_INSTRUCTION("62F27D48639C1180000000"        , vpcompressb(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27D48639C1180000000"        , vpcompressb(zmmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27D088BD1"                  , vpcompressd(xmm1, xmm2));
  TEST_INSTRUCTION("62F27D088B5C1120"              , vpcompressd(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27D088B5C1120"              , vpcompressd(xmmword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27D288BD1"                  , vpcompressd(ymm1, ymm2));
  TEST_INSTRUCTION("62F27D288B5C1120"              , vpcompressd(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27D288B5C1120"              , vpcompressd(ymmword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27D488BD1"                  , vpcompressd(zmm1, zmm2));
  TEST_INSTRUCTION("62F27D488B5C1120"              , vpcompressd(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27D488B5C1120"              , vpcompressd(zmmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F2FD088BD1"                  , vpcompressq(xmm1, xmm2));
  TEST_INSTRUCTION("62F2FD088B5C1110"              , vpcompressq(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F2FD088B5C1110"              , vpcompressq(xmmword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F2FD288BD1"                  , vpcompressq(ymm1, ymm2));
  TEST_INSTRUCTION("62F2FD288B5C1110"              , vpcompressq(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F2FD288B5C1110"              , vpcompressq(ymmword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F2FD488BD1"                  , vpcompressq(zmm1, zmm2));
  TEST_INSTRUCTION("62F2FD488B5C1110"              , vpcompressq(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F2FD488B5C1110"              , vpcompressq(zmmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F2FD0863D1"                  , vpcompressw(xmm1, xmm2));
  TEST_INSTRUCTION("62F2FD08635C1140"              , vpcompressw(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F2FD08635C1140"              , vpcompressw(xmmword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F2FD2863D1"                  , vpcompressw(ymm1, ymm2));
  TEST_INSTRUCTION("62F2FD28635C1140"              , vpcompressw(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F2FD28635C1140"              , vpcompressw(ymmword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F2FD4863D1"                  , vpcompressw(zmm1, zmm2));
  TEST_INSTRUCTION("62F2FD48635C1140"              , vpcompressw(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F2FD48635C1140"              , vpcompressw(zmmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27D08C4CA"                  , vpconflictd(xmm1, xmm2));
  TEST_INSTRUCTION("62F27D08C44C1A08"              , vpconflictd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D08C44C1A08"              , vpconflictd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D28C4CA"                  , vpconflictd(ymm1, ymm2));
  TEST_INSTRUCTION("62F27D28C44C1A04"              , vpconflictd(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D28C44C1A04"              , vpconflictd(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48C4CA"                  , vpconflictd(zmm1, zmm2));
  TEST_INSTRUCTION("62F27D48C44C1A02"              , vpconflictd(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48C44C1A02"              , vpconflictd(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD08C4CA"                  , vpconflictq(xmm1, xmm2));
  TEST_INSTRUCTION("62F2FD08C44C1A08"              , vpconflictq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD08C44C1A08"              , vpconflictq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD28C4CA"                  , vpconflictq(ymm1, ymm2));
  TEST_INSTRUCTION("62F2FD28C44C1A04"              , vpconflictq(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD28C44C1A04"              , vpconflictq(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD48C4CA"                  , vpconflictq(zmm1, zmm2));
  TEST_INSTRUCTION("62F2FD48C44C1A02"              , vpconflictq(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD48C44C1A02"              , vpconflictq(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F26D0850CB"                  , vpdpbusd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26D08504C2B08"              , vpdpbusd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D08504C2B08"              , vpdpbusd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D2850CB"                  , vpdpbusd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F26D28504C2B04"              , vpdpbusd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D28504C2B04"              , vpdpbusd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D4850CB"                  , vpdpbusd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48504C2B02"              , vpdpbusd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48504C2B02"              , vpdpbusd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D0851CB"                  , vpdpbusds(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26D08514C2B08"              , vpdpbusds(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D08514C2B08"              , vpdpbusds(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D2851CB"                  , vpdpbusds(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F26D28514C2B04"              , vpdpbusds(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D28514C2B04"              , vpdpbusds(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D4851CB"                  , vpdpbusds(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48514C2B02"              , vpdpbusds(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48514C2B02"              , vpdpbusds(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D0852CB"                  , vpdpwssd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26D08524C2B08"              , vpdpwssd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D08524C2B08"              , vpdpwssd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D2852CB"                  , vpdpwssd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F26D28524C2B04"              , vpdpwssd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D28524C2B04"              , vpdpwssd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D4852CB"                  , vpdpwssd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48524C2B02"              , vpdpwssd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48524C2B02"              , vpdpwssd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D0853CB"                  , vpdpwssds(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26D08534C2B08"              , vpdpwssds(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D08534C2B08"              , vpdpwssds(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D2853CB"                  , vpdpwssds(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F26D28534C2B04"              , vpdpwssds(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D28534C2B04"              , vpdpwssds(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D4853CB"                  , vpdpwssds(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48534C2B02"              , vpdpwssds(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48534C2B02"              , vpdpwssds(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D088DCB"                  , vpermb(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26D088D4C2B08"              , vpermb(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D088D4C2B08"              , vpermb(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D288DCB"                  , vpermb(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F26D288D4C2B04"              , vpermb(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D288D4C2B04"              , vpermb(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D488DCB"                  , vpermb(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D488D4C2B02"              , vpermb(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D488D4C2B02"              , vpermb(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D4836CB"                  , vpermd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48364C2B02"              , vpermd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48364C2B02"              , vpermd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D0875CB"                  , vpermi2b(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26D08754C2B08"              , vpermi2b(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D08754C2B08"              , vpermi2b(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D2875CB"                  , vpermi2b(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F26D28754C2B04"              , vpermi2b(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D28754C2B04"              , vpermi2b(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D4875CB"                  , vpermi2b(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48754C2B02"              , vpermi2b(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48754C2B02"              , vpermi2b(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D0876CB"                  , vpermi2d(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26D08764C2B08"              , vpermi2d(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D08764C2B08"              , vpermi2d(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D2876CB"                  , vpermi2d(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F26D28764C2B04"              , vpermi2d(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D28764C2B04"              , vpermi2d(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D4876CB"                  , vpermi2d(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48764C2B02"              , vpermi2d(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48764C2B02"              , vpermi2d(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED0877CB"                  , vpermi2pd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED08774C2B08"              , vpermi2pd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED08774C2B08"              , vpermi2pd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED2877CB"                  , vpermi2pd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED28774C2B04"              , vpermi2pd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED28774C2B04"              , vpermi2pd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED4877CB"                  , vpermi2pd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48774C2B02"              , vpermi2pd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48774C2B02"              , vpermi2pd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D0877CB"                  , vpermi2ps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26D08774C2B08"              , vpermi2ps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D08774C2B08"              , vpermi2ps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D2877CB"                  , vpermi2ps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F26D28774C2B04"              , vpermi2ps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D28774C2B04"              , vpermi2ps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D4877CB"                  , vpermi2ps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48774C2B02"              , vpermi2ps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48774C2B02"              , vpermi2ps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED0876CB"                  , vpermi2q(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED08764C2B08"              , vpermi2q(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED08764C2B08"              , vpermi2q(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED2876CB"                  , vpermi2q(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED28764C2B04"              , vpermi2q(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED28764C2B04"              , vpermi2q(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED4876CB"                  , vpermi2q(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48764C2B02"              , vpermi2q(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48764C2B02"              , vpermi2q(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED0875CB"                  , vpermi2w(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED08754C2B08"              , vpermi2w(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED08754C2B08"              , vpermi2w(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED2875CB"                  , vpermi2w(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED28754C2B04"              , vpermi2w(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED28754C2B04"              , vpermi2w(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED4875CB"                  , vpermi2w(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48754C2B02"              , vpermi2w(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48754C2B02"              , vpermi2w(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED480DCB"                  , vpermilpd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F3FD4805CA01"                , vpermilpd(zmm1, zmm2, 1));
  TEST_INSTRUCTION("62F2ED480D4C2B02"              , vpermilpd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED480D4C2B02"              , vpermilpd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F3FD48054C1A0201"            , vpermilpd(zmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F3FD48054C1A0201"            , vpermilpd(zmm1, zmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F26D480CCB"                  , vpermilps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F37D4804CA01"                , vpermilps(zmm1, zmm2, 1));
  TEST_INSTRUCTION("62F26D480C4C2B02"              , vpermilps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D480C4C2B02"              , vpermilps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F37D48044C1A0201"            , vpermilps(zmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F37D48044C1A0201"            , vpermilps(zmm1, zmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F2ED2816CB"                  , vpermpd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED28164C2B04"              , vpermpd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED28164C2B04"              , vpermpd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED4816CB"                  , vpermpd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F3FD4801CA01"                , vpermpd(zmm1, zmm2, 1));
  TEST_INSTRUCTION("62F2ED48164C2B02"              , vpermpd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48164C2B02"              , vpermpd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F3FD48014C1A0201"            , vpermpd(zmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F3FD48014C1A0201"            , vpermpd(zmm1, zmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F26D4816CB"                  , vpermps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48164C2B02"              , vpermps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48164C2B02"              , vpermps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED2836CB"                  , vpermq(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED28364C2B04"              , vpermq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED28364C2B04"              , vpermq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED4836CB"                  , vpermq(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F3FD4800CA01"                , vpermq(zmm1, zmm2, 1));
  TEST_INSTRUCTION("62F2ED48364C2B02"              , vpermq(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48364C2B02"              , vpermq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F3FD48004C1A0201"            , vpermq(zmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F3FD48004C1A0201"            , vpermq(zmm1, zmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F26D087DCB"                  , vpermt2b(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26D087D4C2B08"              , vpermt2b(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D087D4C2B08"              , vpermt2b(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D287DCB"                  , vpermt2b(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F26D287D4C2B04"              , vpermt2b(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D287D4C2B04"              , vpermt2b(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D487DCB"                  , vpermt2b(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D487D4C2B02"              , vpermt2b(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D487D4C2B02"              , vpermt2b(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D087ECB"                  , vpermt2d(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26D087E4C2B08"              , vpermt2d(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D087E4C2B08"              , vpermt2d(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D287ECB"                  , vpermt2d(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F26D287E4C2B04"              , vpermt2d(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D287E4C2B04"              , vpermt2d(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D487ECB"                  , vpermt2d(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D487E4C2B02"              , vpermt2d(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D487E4C2B02"              , vpermt2d(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED087FCB"                  , vpermt2pd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED087F4C2B08"              , vpermt2pd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED087F4C2B08"              , vpermt2pd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED287FCB"                  , vpermt2pd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED287F4C2B04"              , vpermt2pd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED287F4C2B04"              , vpermt2pd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED487FCB"                  , vpermt2pd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED487F4C2B02"              , vpermt2pd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED487F4C2B02"              , vpermt2pd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D087FCB"                  , vpermt2ps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26D087F4C2B08"              , vpermt2ps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D087F4C2B08"              , vpermt2ps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D287FCB"                  , vpermt2ps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F26D287F4C2B04"              , vpermt2ps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D287F4C2B04"              , vpermt2ps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D487FCB"                  , vpermt2ps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D487F4C2B02"              , vpermt2ps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D487F4C2B02"              , vpermt2ps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED087ECB"                  , vpermt2q(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED087E4C2B08"              , vpermt2q(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED087E4C2B08"              , vpermt2q(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED287ECB"                  , vpermt2q(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED287E4C2B04"              , vpermt2q(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED287E4C2B04"              , vpermt2q(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED487ECB"                  , vpermt2q(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED487E4C2B02"              , vpermt2q(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED487E4C2B02"              , vpermt2q(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED087DCB"                  , vpermt2w(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED087D4C2B08"              , vpermt2w(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED087D4C2B08"              , vpermt2w(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED287DCB"                  , vpermt2w(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED287D4C2B04"              , vpermt2w(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED287D4C2B04"              , vpermt2w(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED487DCB"                  , vpermt2w(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED487D4C2B02"              , vpermt2w(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED487D4C2B02"              , vpermt2w(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED088DCB"                  , vpermw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED088D4C2B08"              , vpermw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED088D4C2B08"              , vpermw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED288DCB"                  , vpermw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED288D4C2B04"              , vpermw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED288D4C2B04"              , vpermw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED488DCB"                  , vpermw(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED488D4C2B02"              , vpermw(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED488D4C2B02"              , vpermw(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F27D0862CA"                  , vpexpandb(xmm1, xmm2));
  TEST_INSTRUCTION("62F27D08628C1A80000000"        , vpexpandb(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D08628C1A80000000"        , vpexpandb(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D2862CA"                  , vpexpandb(ymm1, ymm2));
  TEST_INSTRUCTION("62F27D28628C1A80000000"        , vpexpandb(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D28628C1A80000000"        , vpexpandb(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D4862CA"                  , vpexpandb(zmm1, zmm2));
  TEST_INSTRUCTION("62F27D48628C1A80000000"        , vpexpandb(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48628C1A80000000"        , vpexpandb(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D0889CA"                  , vpexpandd(xmm1, xmm2));
  TEST_INSTRUCTION("62F27D08894C1A20"              , vpexpandd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D08894C1A20"              , vpexpandd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D2889CA"                  , vpexpandd(ymm1, ymm2));
  TEST_INSTRUCTION("62F27D28894C1A20"              , vpexpandd(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D28894C1A20"              , vpexpandd(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D4889CA"                  , vpexpandd(zmm1, zmm2));
  TEST_INSTRUCTION("62F27D48894C1A20"              , vpexpandd(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48894C1A20"              , vpexpandd(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD0889CA"                  , vpexpandq(xmm1, xmm2));
  TEST_INSTRUCTION("62F2FD08894C1A10"              , vpexpandq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD08894C1A10"              , vpexpandq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD2889CA"                  , vpexpandq(ymm1, ymm2));
  TEST_INSTRUCTION("62F2FD28894C1A10"              , vpexpandq(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD28894C1A10"              , vpexpandq(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD4889CA"                  , vpexpandq(zmm1, zmm2));
  TEST_INSTRUCTION("62F2FD48894C1A10"              , vpexpandq(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD48894C1A10"              , vpexpandq(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD0862CA"                  , vpexpandw(xmm1, xmm2));
  TEST_INSTRUCTION("62F2FD08624C1A40"              , vpexpandw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD08624C1A40"              , vpexpandw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD2862CA"                  , vpexpandw(ymm1, ymm2));
  TEST_INSTRUCTION("62F2FD28624C1A40"              , vpexpandw(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD28624C1A40"              , vpexpandw(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD4862CA"                  , vpexpandw(zmm1, zmm2));
  TEST_INSTRUCTION("62F2FD48624C1A40"              , vpexpandw(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD48624C1A40"              , vpexpandw(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D09904C1A20"              , k(k1).vpgatherdd(xmm1, ptr(edx, xmm3, 0, 128)));
  TEST_INSTRUCTION("62F27D29904C1A20"              , k(k1).vpgatherdd(ymm1, ptr(edx, ymm3, 0, 128)));
  TEST_INSTRUCTION("62F27D49904C1A20"              , k(k1).vpgatherdd(zmm1, ptr(edx, zmm3, 0, 128)));
  TEST_INSTRUCTION("62F2FD09904C1A10"              , k(k1).vpgatherdq(xmm1, ptr(edx, xmm3, 0, 128)));
  TEST_INSTRUCTION("62F2FD29904C1A10"              , k(k1).vpgatherdq(ymm1, ptr(edx, xmm3, 0, 128)));
  TEST_INSTRUCTION("62F2FD49904C1A10"              , k(k1).vpgatherdq(zmm1, ptr(edx, ymm3, 0, 128)));
  TEST_INSTRUCTION("62F27D09914C1A20"              , k(k1).vpgatherqd(xmm1, ptr(edx, xmm3, 0, 128)));
  TEST_INSTRUCTION("62F27D29914C1A20"              , k(k1).vpgatherqd(xmm1, ptr(edx, ymm3, 0, 128)));
  TEST_INSTRUCTION("62F27D49914C1A20"              , k(k1).vpgatherqd(ymm1, ptr(edx, zmm3, 0, 128)));
  TEST_INSTRUCTION("62F2FD09914C1A10"              , k(k1).vpgatherqq(xmm1, ptr(edx, xmm3, 0, 128)));
  TEST_INSTRUCTION("62F2FD29914C1A10"              , k(k1).vpgatherqq(ymm1, ptr(edx, ymm3, 0, 128)));
  TEST_INSTRUCTION("62F2FD49914C1A10"              , k(k1).vpgatherqq(zmm1, ptr(edx, zmm3, 0, 128)));
  TEST_INSTRUCTION("62F27D0844CA"                  , vplzcntd(xmm1, xmm2));
  TEST_INSTRUCTION("62F27D08444C1A08"              , vplzcntd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D08444C1A08"              , vplzcntd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D2844CA"                  , vplzcntd(ymm1, ymm2));
  TEST_INSTRUCTION("62F27D28444C1A04"              , vplzcntd(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D28444C1A04"              , vplzcntd(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D4844CA"                  , vplzcntd(zmm1, zmm2));
  TEST_INSTRUCTION("62F27D48444C1A02"              , vplzcntd(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48444C1A02"              , vplzcntd(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD0844CA"                  , vplzcntq(xmm1, xmm2));
  TEST_INSTRUCTION("62F2FD08444C1A08"              , vplzcntq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD08444C1A08"              , vplzcntq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD2844CA"                  , vplzcntq(ymm1, ymm2));
  TEST_INSTRUCTION("62F2FD28444C1A04"              , vplzcntq(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD28444C1A04"              , vplzcntq(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD4844CA"                  , vplzcntq(zmm1, zmm2));
  TEST_INSTRUCTION("62F2FD48444C1A02"              , vplzcntq(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD48444C1A02"              , vplzcntq(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2ED08B5CB"                  , vpmadd52huq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED08B54C2B08"              , vpmadd52huq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED08B54C2B08"              , vpmadd52huq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED28B5CB"                  , vpmadd52huq(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED28B54C2B04"              , vpmadd52huq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED28B54C2B04"              , vpmadd52huq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48B5CB"                  , vpmadd52huq(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48B54C2B02"              , vpmadd52huq(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48B54C2B02"              , vpmadd52huq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED08B4CB"                  , vpmadd52luq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED08B44C2B08"              , vpmadd52luq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED08B44C2B08"              , vpmadd52luq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED28B4CB"                  , vpmadd52luq(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED28B44C2B04"              , vpmadd52luq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED28B44C2B04"              , vpmadd52luq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48B4CB"                  , vpmadd52luq(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48B44C2B02"              , vpmadd52luq(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48B44C2B02"              , vpmadd52luq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D4804CB"                  , vpmaddubsw(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48044C2B02"              , vpmaddubsw(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48044C2B02"              , vpmaddubsw(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48F5CB"                  , vpmaddwd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48F54C2B02"              , vpmaddwd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48F54C2B02"              , vpmaddwd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D483CCB"                  , vpmaxsb(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D483C4C2B02"              , vpmaxsb(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D483C4C2B02"              , vpmaxsb(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D483DCB"                  , vpmaxsd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D483D4C2B02"              , vpmaxsd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D483D4C2B02"              , vpmaxsd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED083DCB"                  , vpmaxsq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED083D4C2B08"              , vpmaxsq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED083D4C2B08"              , vpmaxsq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED283DCB"                  , vpmaxsq(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED283D4C2B04"              , vpmaxsq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED283D4C2B04"              , vpmaxsq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED483DCB"                  , vpmaxsq(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED483D4C2B02"              , vpmaxsq(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED483D4C2B02"              , vpmaxsq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48EECB"                  , vpmaxsw(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48EE4C2B02"              , vpmaxsw(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48EE4C2B02"              , vpmaxsw(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48DECB"                  , vpmaxub(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48DE4C2B02"              , vpmaxub(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48DE4C2B02"              , vpmaxub(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D483FCB"                  , vpmaxud(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D483F4C2B02"              , vpmaxud(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D483F4C2B02"              , vpmaxud(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED083FCB"                  , vpmaxuq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED083F4C2B08"              , vpmaxuq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED083F4C2B08"              , vpmaxuq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED283FCB"                  , vpmaxuq(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED283F4C2B04"              , vpmaxuq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED283F4C2B04"              , vpmaxuq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED483FCB"                  , vpmaxuq(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED483F4C2B02"              , vpmaxuq(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED483F4C2B02"              , vpmaxuq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D483ECB"                  , vpmaxuw(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D483E4C2B02"              , vpmaxuw(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D483E4C2B02"              , vpmaxuw(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D4838CB"                  , vpminsb(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48384C2B02"              , vpminsb(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48384C2B02"              , vpminsb(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D4839CB"                  , vpminsd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48394C2B02"              , vpminsd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48394C2B02"              , vpminsd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED0839CB"                  , vpminsq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED08394C2B08"              , vpminsq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED08394C2B08"              , vpminsq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED2839CB"                  , vpminsq(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED28394C2B04"              , vpminsq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED28394C2B04"              , vpminsq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED4839CB"                  , vpminsq(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48394C2B02"              , vpminsq(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48394C2B02"              , vpminsq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48EACB"                  , vpminsw(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48EA4C2B02"              , vpminsw(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48EA4C2B02"              , vpminsw(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48DACB"                  , vpminub(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48DA4C2B02"              , vpminub(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48DA4C2B02"              , vpminub(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D483BCB"                  , vpminud(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D483B4C2B02"              , vpminud(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D483B4C2B02"              , vpminud(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED083BCB"                  , vpminuq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED083B4C2B08"              , vpminuq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED083B4C2B08"              , vpminuq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED283BCB"                  , vpminuq(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED283B4C2B04"              , vpminuq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED283B4C2B04"              , vpminuq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED483BCB"                  , vpminuq(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED483B4C2B02"              , vpminuq(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED483B4C2B02"              , vpminuq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D483ACB"                  , vpminuw(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D483A4C2B02"              , vpminuw(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D483A4C2B02"              , vpminuw(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F27E0829CA"                  , vpmovb2m(k1, xmm2));
  TEST_INSTRUCTION("62F27E2829CA"                  , vpmovb2m(k1, ymm2));
  TEST_INSTRUCTION("62F27E4829CA"                  , vpmovb2m(k1, zmm2));
  TEST_INSTRUCTION("62F27E0839CA"                  , vpmovd2m(k1, xmm2));
  TEST_INSTRUCTION("62F27E2839CA"                  , vpmovd2m(k1, ymm2));
  TEST_INSTRUCTION("62F27E4839CA"                  , vpmovd2m(k1, zmm2));
  TEST_INSTRUCTION("62F27E0831D1"                  , vpmovdb(xmm1, xmm2));
  TEST_INSTRUCTION("62F27E08315C1120"              , vpmovdb(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E08315C1120"              , vpmovdb(dword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E2831D1"                  , vpmovdb(xmm1, ymm2));
  TEST_INSTRUCTION("62F27E28315C1110"              , vpmovdb(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E28315C1110"              , vpmovdb(qword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E4831D1"                  , vpmovdb(xmm1, zmm2));
  TEST_INSTRUCTION("62F27E48315C1108"              , vpmovdb(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E48315C1108"              , vpmovdb(xmmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E0833D1"                  , vpmovdw(xmm1, xmm2));
  TEST_INSTRUCTION("62F27E08335C1110"              , vpmovdw(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E08335C1110"              , vpmovdw(qword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E2833D1"                  , vpmovdw(xmm1, ymm2));
  TEST_INSTRUCTION("62F27E28335C1108"              , vpmovdw(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E28335C1108"              , vpmovdw(xmmword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E4833D1"                  , vpmovdw(ymm1, zmm2));
  TEST_INSTRUCTION("62F27E48335C1104"              , vpmovdw(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E48335C1104"              , vpmovdw(ymmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E0828CA"                  , vpmovm2b(xmm1, k2));
  TEST_INSTRUCTION("62F27E2828CA"                  , vpmovm2b(ymm1, k2));
  TEST_INSTRUCTION("62F27E4828CA"                  , vpmovm2b(zmm1, k2));
  TEST_INSTRUCTION("62F27E0838CA"                  , vpmovm2d(xmm1, k2));
  TEST_INSTRUCTION("62F27E2838CA"                  , vpmovm2d(ymm1, k2));
  TEST_INSTRUCTION("62F27E4838CA"                  , vpmovm2d(zmm1, k2));
  TEST_INSTRUCTION("62F2FE0838CA"                  , vpmovm2q(xmm1, k2));
  TEST_INSTRUCTION("62F2FE2838CA"                  , vpmovm2q(ymm1, k2));
  TEST_INSTRUCTION("62F2FE4838CA"                  , vpmovm2q(zmm1, k2));
  TEST_INSTRUCTION("62F2FE0828CA"                  , vpmovm2w(xmm1, k2));
  TEST_INSTRUCTION("62F2FE2828CA"                  , vpmovm2w(ymm1, k2));
  TEST_INSTRUCTION("62F2FE4828CA"                  , vpmovm2w(zmm1, k2));
  TEST_INSTRUCTION("62F2FE0839CA"                  , vpmovq2m(k1, xmm2));
  TEST_INSTRUCTION("62F2FE2839CA"                  , vpmovq2m(k1, ymm2));
  TEST_INSTRUCTION("62F2FE4839CA"                  , vpmovq2m(k1, zmm2));
  TEST_INSTRUCTION("62F27E0832D1"                  , vpmovqb(xmm1, xmm2));
  TEST_INSTRUCTION("62F27E08325C1140"              , vpmovqb(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E08325C1140"              , vpmovqb(word_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E2832D1"                  , vpmovqb(xmm1, ymm2));
  TEST_INSTRUCTION("62F27E28325C1120"              , vpmovqb(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E28325C1120"              , vpmovqb(dword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E4832D1"                  , vpmovqb(xmm1, zmm2));
  TEST_INSTRUCTION("62F27E48325C1110"              , vpmovqb(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E48325C1110"              , vpmovqb(qword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E0835D1"                  , vpmovqd(xmm1, xmm2));
  TEST_INSTRUCTION("62F27E08355C1110"              , vpmovqd(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E08355C1110"              , vpmovqd(qword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E2835D1"                  , vpmovqd(xmm1, ymm2));
  TEST_INSTRUCTION("62F27E28355C1108"              , vpmovqd(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E28355C1108"              , vpmovqd(xmmword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E4835D1"                  , vpmovqd(ymm1, zmm2));
  TEST_INSTRUCTION("62F27E48355C1104"              , vpmovqd(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E48355C1104"              , vpmovqd(ymmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E0834D1"                  , vpmovqw(xmm1, xmm2));
  TEST_INSTRUCTION("62F27E08345C1120"              , vpmovqw(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E08345C1120"              , vpmovqw(dword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E2834D1"                  , vpmovqw(xmm1, ymm2));
  TEST_INSTRUCTION("62F27E28345C1110"              , vpmovqw(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E28345C1110"              , vpmovqw(qword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E4834D1"                  , vpmovqw(xmm1, zmm2));
  TEST_INSTRUCTION("62F27E48345C1108"              , vpmovqw(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E48345C1108"              , vpmovqw(xmmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E0821D1"                  , vpmovsdb(xmm1, xmm2));
  TEST_INSTRUCTION("62F27E08215C1120"              , vpmovsdb(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E08215C1120"              , vpmovsdb(dword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E2821D1"                  , vpmovsdb(xmm1, ymm2));
  TEST_INSTRUCTION("62F27E28215C1110"              , vpmovsdb(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E28215C1110"              , vpmovsdb(qword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E4821D1"                  , vpmovsdb(xmm1, zmm2));
  TEST_INSTRUCTION("62F27E48215C1108"              , vpmovsdb(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E48215C1108"              , vpmovsdb(xmmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E0823D1"                  , vpmovsdw(xmm1, xmm2));
  TEST_INSTRUCTION("62F27E08235C1110"              , vpmovsdw(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E08235C1110"              , vpmovsdw(qword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E2823D1"                  , vpmovsdw(xmm1, ymm2));
  TEST_INSTRUCTION("62F27E28235C1108"              , vpmovsdw(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E28235C1108"              , vpmovsdw(xmmword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E4823D1"                  , vpmovsdw(ymm1, zmm2));
  TEST_INSTRUCTION("62F27E48235C1104"              , vpmovsdw(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E48235C1104"              , vpmovsdw(ymmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E0822D1"                  , vpmovsqb(xmm1, xmm2));
  TEST_INSTRUCTION("62F27E08225C1140"              , vpmovsqb(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E08225C1140"              , vpmovsqb(word_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E2822D1"                  , vpmovsqb(xmm1, ymm2));
  TEST_INSTRUCTION("62F27E28225C1120"              , vpmovsqb(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E28225C1120"              , vpmovsqb(dword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E4822D1"                  , vpmovsqb(xmm1, zmm2));
  TEST_INSTRUCTION("62F27E48225C1110"              , vpmovsqb(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E48225C1110"              , vpmovsqb(qword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E0825D1"                  , vpmovsqd(xmm1, xmm2));
  TEST_INSTRUCTION("62F27E08255C1110"              , vpmovsqd(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E08255C1110"              , vpmovsqd(qword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E2825D1"                  , vpmovsqd(xmm1, ymm2));
  TEST_INSTRUCTION("62F27E28255C1108"              , vpmovsqd(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E28255C1108"              , vpmovsqd(xmmword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E4825D1"                  , vpmovsqd(ymm1, zmm2));
  TEST_INSTRUCTION("62F27E48255C1104"              , vpmovsqd(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E48255C1104"              , vpmovsqd(ymmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E0824D1"                  , vpmovsqw(xmm1, xmm2));
  TEST_INSTRUCTION("62F27E08245C1120"              , vpmovsqw(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E08245C1120"              , vpmovsqw(dword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E2824D1"                  , vpmovsqw(xmm1, ymm2));
  TEST_INSTRUCTION("62F27E28245C1110"              , vpmovsqw(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E28245C1110"              , vpmovsqw(qword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E4824D1"                  , vpmovsqw(xmm1, zmm2));
  TEST_INSTRUCTION("62F27E48245C1108"              , vpmovsqw(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E48245C1108"              , vpmovsqw(xmmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E0820D1"                  , vpmovswb(xmm1, xmm2));
  TEST_INSTRUCTION("62F27E08205C1110"              , vpmovswb(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E08205C1110"              , vpmovswb(qword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E2820D1"                  , vpmovswb(xmm1, ymm2));
  TEST_INSTRUCTION("62F27E28205C1108"              , vpmovswb(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E28205C1108"              , vpmovswb(xmmword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E4820D1"                  , vpmovswb(ymm1, zmm2));
  TEST_INSTRUCTION("62F27E48205C1104"              , vpmovswb(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E48205C1104"              , vpmovswb(ymmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27D4821CA"                  , vpmovsxbd(zmm1, xmm2));
  TEST_INSTRUCTION("62F27D48214C1A08"              , vpmovsxbd(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48214C1A08"              , vpmovsxbd(zmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D4822CA"                  , vpmovsxbq(zmm1, xmm2));
  TEST_INSTRUCTION("62F27D48224C1A10"              , vpmovsxbq(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48224C1A10"              , vpmovsxbq(zmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D4820CA"                  , vpmovsxbw(zmm1, ymm2));
  TEST_INSTRUCTION("62F27D48204C1A04"              , vpmovsxbw(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48204C1A04"              , vpmovsxbw(zmm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D4825CA"                  , vpmovsxdq(zmm1, ymm2));
  TEST_INSTRUCTION("62F27D48254C1A04"              , vpmovsxdq(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48254C1A04"              , vpmovsxdq(zmm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D4823CA"                  , vpmovsxwd(zmm1, ymm2));
  TEST_INSTRUCTION("62F27D48234C1A04"              , vpmovsxwd(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48234C1A04"              , vpmovsxwd(zmm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D4824CA"                  , vpmovsxwq(zmm1, xmm2));
  TEST_INSTRUCTION("62F27D48244C1A08"              , vpmovsxwq(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48244C1A08"              , vpmovsxwq(zmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27E0811D1"                  , vpmovusdb(xmm1, xmm2));
  TEST_INSTRUCTION("62F27E08115C1120"              , vpmovusdb(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E08115C1120"              , vpmovusdb(dword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E2811D1"                  , vpmovusdb(xmm1, ymm2));
  TEST_INSTRUCTION("62F27E28115C1110"              , vpmovusdb(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E28115C1110"              , vpmovusdb(qword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E4811D1"                  , vpmovusdb(xmm1, zmm2));
  TEST_INSTRUCTION("62F27E48115C1108"              , vpmovusdb(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E48115C1108"              , vpmovusdb(xmmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E0813D1"                  , vpmovusdw(xmm1, xmm2));
  TEST_INSTRUCTION("62F27E08135C1110"              , vpmovusdw(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E08135C1110"              , vpmovusdw(qword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E2813D1"                  , vpmovusdw(xmm1, ymm2));
  TEST_INSTRUCTION("62F27E28135C1108"              , vpmovusdw(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E28135C1108"              , vpmovusdw(xmmword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E4813D1"                  , vpmovusdw(ymm1, zmm2));
  TEST_INSTRUCTION("62F27E48135C1104"              , vpmovusdw(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E48135C1104"              , vpmovusdw(ymmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E0812D1"                  , vpmovusqb(xmm1, xmm2));
  TEST_INSTRUCTION("62F27E08125C1140"              , vpmovusqb(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E08125C1140"              , vpmovusqb(word_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E2812D1"                  , vpmovusqb(xmm1, ymm2));
  TEST_INSTRUCTION("62F27E28125C1120"              , vpmovusqb(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E28125C1120"              , vpmovusqb(dword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E4812D1"                  , vpmovusqb(xmm1, zmm2));
  TEST_INSTRUCTION("62F27E48125C1110"              , vpmovusqb(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E48125C1110"              , vpmovusqb(qword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E0815D1"                  , vpmovusqd(xmm1, xmm2));
  TEST_INSTRUCTION("62F27E08155C1110"              , vpmovusqd(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E08155C1110"              , vpmovusqd(qword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E2815D1"                  , vpmovusqd(xmm1, ymm2));
  TEST_INSTRUCTION("62F27E28155C1108"              , vpmovusqd(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E28155C1108"              , vpmovusqd(xmmword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E4815D1"                  , vpmovusqd(ymm1, zmm2));
  TEST_INSTRUCTION("62F27E48155C1104"              , vpmovusqd(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E48155C1104"              , vpmovusqd(ymmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E0814D1"                  , vpmovusqw(xmm1, xmm2));
  TEST_INSTRUCTION("62F27E08145C1120"              , vpmovusqw(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E08145C1120"              , vpmovusqw(dword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E2814D1"                  , vpmovusqw(xmm1, ymm2));
  TEST_INSTRUCTION("62F27E28145C1110"              , vpmovusqw(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E28145C1110"              , vpmovusqw(qword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E4814D1"                  , vpmovusqw(xmm1, zmm2));
  TEST_INSTRUCTION("62F27E48145C1108"              , vpmovusqw(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E48145C1108"              , vpmovusqw(xmmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E0810D1"                  , vpmovuswb(xmm1, xmm2));
  TEST_INSTRUCTION("62F27E08105C1110"              , vpmovuswb(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E08105C1110"              , vpmovuswb(qword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E2810D1"                  , vpmovuswb(xmm1, ymm2));
  TEST_INSTRUCTION("62F27E28105C1108"              , vpmovuswb(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E28105C1108"              , vpmovuswb(xmmword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E4810D1"                  , vpmovuswb(ymm1, zmm2));
  TEST_INSTRUCTION("62F27E48105C1104"              , vpmovuswb(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E48105C1104"              , vpmovuswb(ymmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F2FE0829CA"                  , vpmovw2m(k1, xmm2));
  TEST_INSTRUCTION("62F2FE2829CA"                  , vpmovw2m(k1, ymm2));
  TEST_INSTRUCTION("62F2FE4829CA"                  , vpmovw2m(k1, zmm2));
  TEST_INSTRUCTION("62F27E0830D1"                  , vpmovwb(xmm1, xmm2));
  TEST_INSTRUCTION("62F27E08305C1110"              , vpmovwb(ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E08305C1110"              , vpmovwb(qword_ptr(ecx, edx, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27E2830D1"                  , vpmovwb(xmm1, ymm2));
  TEST_INSTRUCTION("62F27E28305C1108"              , vpmovwb(ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E28305C1108"              , vpmovwb(xmmword_ptr(ecx, edx, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27E4830D1"                  , vpmovwb(ymm1, zmm2));
  TEST_INSTRUCTION("62F27E48305C1104"              , vpmovwb(ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27E48305C1104"              , vpmovwb(ymmword_ptr(ecx, edx, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27D4831CA"                  , vpmovzxbd(zmm1, xmm2));
  TEST_INSTRUCTION("62F27D48314C1A08"              , vpmovzxbd(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48314C1A08"              , vpmovzxbd(zmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D4832CA"                  , vpmovzxbq(zmm1, xmm2));
  TEST_INSTRUCTION("62F27D48324C1A10"              , vpmovzxbq(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48324C1A10"              , vpmovzxbq(zmm1, qword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D4830CA"                  , vpmovzxbw(zmm1, ymm2));
  TEST_INSTRUCTION("62F27D48304C1A04"              , vpmovzxbw(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48304C1A04"              , vpmovzxbw(zmm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D4835CA"                  , vpmovzxdq(zmm1, ymm2));
  TEST_INSTRUCTION("62F27D48354C1A04"              , vpmovzxdq(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48354C1A04"              , vpmovzxdq(zmm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D4833CA"                  , vpmovzxwd(zmm1, ymm2));
  TEST_INSTRUCTION("62F27D48334C1A04"              , vpmovzxwd(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48334C1A04"              , vpmovzxwd(zmm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D4834CA"                  , vpmovzxwq(zmm1, xmm2));
  TEST_INSTRUCTION("62F27D48344C1A08"              , vpmovzxwq(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48344C1A08"              , vpmovzxwq(zmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2ED4828CB"                  , vpmuldq(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48284C2B02"              , vpmuldq(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48284C2B02"              , vpmuldq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D480BCB"                  , vpmulhrsw(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D480B4C2B02"              , vpmulhrsw(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D480B4C2B02"              , vpmulhrsw(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48E4CB"                  , vpmulhuw(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48E44C2B02"              , vpmulhuw(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48E44C2B02"              , vpmulhuw(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48E5CB"                  , vpmulhw(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48E54C2B02"              , vpmulhw(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48E54C2B02"              , vpmulhw(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D4840CB"                  , vpmulld(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48404C2B02"              , vpmulld(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48404C2B02"              , vpmulld(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED0840CB"                  , vpmullq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED08404C2B08"              , vpmullq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED08404C2B08"              , vpmullq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED2840CB"                  , vpmullq(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED28404C2B04"              , vpmullq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED28404C2B04"              , vpmullq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED4840CB"                  , vpmullq(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48404C2B02"              , vpmullq(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48404C2B02"              , vpmullq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48D5CB"                  , vpmullw(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48D54C2B02"              , vpmullw(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48D54C2B02"              , vpmullw(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED0883CB"                  , vpmultishiftqb(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED08834C2B08"              , vpmultishiftqb(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED08834C2B08"              , vpmultishiftqb(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED2883CB"                  , vpmultishiftqb(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED28834C2B04"              , vpmultishiftqb(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED28834C2B04"              , vpmultishiftqb(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED4883CB"                  , vpmultishiftqb(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48834C2B02"              , vpmultishiftqb(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48834C2B02"              , vpmultishiftqb(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED48F4CB"                  , vpmuludq(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F1ED48F44C2B02"              , vpmuludq(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED48F44C2B02"              , vpmuludq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F27D0854CA"                  , vpopcntb(xmm1, xmm2));
  TEST_INSTRUCTION("62F27D08544C1A08"              , vpopcntb(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D08544C1A08"              , vpopcntb(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D2854CA"                  , vpopcntb(ymm1, ymm2));
  TEST_INSTRUCTION("62F27D28544C1A04"              , vpopcntb(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D28544C1A04"              , vpopcntb(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D4854CA"                  , vpopcntb(zmm1, zmm2));
  TEST_INSTRUCTION("62F27D48544C1A02"              , vpopcntb(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48544C1A02"              , vpopcntb(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D0855CA"                  , vpopcntd(xmm1, xmm2));
  TEST_INSTRUCTION("62F27D08554C1A08"              , vpopcntd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D08554C1A08"              , vpopcntd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D2855CA"                  , vpopcntd(ymm1, ymm2));
  TEST_INSTRUCTION("62F27D28554C1A04"              , vpopcntd(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D28554C1A04"              , vpopcntd(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D4855CA"                  , vpopcntd(zmm1, zmm2));
  TEST_INSTRUCTION("62F27D48554C1A02"              , vpopcntd(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48554C1A02"              , vpopcntd(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD0855CA"                  , vpopcntq(xmm1, xmm2));
  TEST_INSTRUCTION("62F2FD08554C1A08"              , vpopcntq(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD08554C1A08"              , vpopcntq(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD2855CA"                  , vpopcntq(ymm1, ymm2));
  TEST_INSTRUCTION("62F2FD28554C1A04"              , vpopcntq(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD28554C1A04"              , vpopcntq(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD4855CA"                  , vpopcntq(zmm1, zmm2));
  TEST_INSTRUCTION("62F2FD48554C1A02"              , vpopcntq(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD48554C1A02"              , vpopcntq(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD0854CA"                  , vpopcntw(xmm1, xmm2));
  TEST_INSTRUCTION("62F2FD08544C1A08"              , vpopcntw(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD08544C1A08"              , vpopcntw(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD2854CA"                  , vpopcntw(ymm1, ymm2));
  TEST_INSTRUCTION("62F2FD28544C1A04"              , vpopcntw(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD28544C1A04"              , vpopcntw(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD4854CA"                  , vpopcntw(zmm1, zmm2));
  TEST_INSTRUCTION("62F2FD48544C1A02"              , vpopcntw(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD48544C1A02"              , vpopcntw(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F16D08EBCB"                  , vpord(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F16D08EB4C2B08"              , vpord(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D08EB4C2B08"              , vpord(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D28EBCB"                  , vpord(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F16D28EB4C2B04"              , vpord(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D28EB4C2B04"              , vpord(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48EBCB"                  , vpord(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48EB4C2B02"              , vpord(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48EB4C2B02"              , vpord(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED08EBCB"                  , vporq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F1ED08EB4C2B08"              , vporq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED08EB4C2B08"              , vporq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED28EBCB"                  , vporq(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F1ED28EB4C2B04"              , vporq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED28EB4C2B04"              , vporq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED48EBCB"                  , vporq(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F1ED48EB4C2B02"              , vporq(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED48EB4C2B02"              , vporq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1750872CA01"                , vprold(xmm1, xmm2, 1));
  TEST_INSTRUCTION("62F17508724C1A0801"            , vprold(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F17508724C1A0801"            , vprold(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1752872CA01"                , vprold(ymm1, ymm2, 1));
  TEST_INSTRUCTION("62F17528724C1A0401"            , vprold(ymm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F17528724C1A0401"            , vprold(ymm1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1754872CA01"                , vprold(zmm1, zmm2, 1));
  TEST_INSTRUCTION("62F17548724C1A0201"            , vprold(zmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F17548724C1A0201"            , vprold(zmm1, zmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1F50872CA01"                , vprolq(xmm1, xmm2, 1));
  TEST_INSTRUCTION("62F1F508724C1A0801"            , vprolq(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1F508724C1A0801"            , vprolq(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1F52872CA01"                , vprolq(ymm1, ymm2, 1));
  TEST_INSTRUCTION("62F1F528724C1A0401"            , vprolq(ymm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1F528724C1A0401"            , vprolq(ymm1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1F54872CA01"                , vprolq(zmm1, zmm2, 1));
  TEST_INSTRUCTION("62F1F548724C1A0201"            , vprolq(zmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1F548724C1A0201"            , vprolq(zmm1, zmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F26D0815CB"                  , vprolvd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26D08154C2B08"              , vprolvd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D08154C2B08"              , vprolvd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D2815CB"                  , vprolvd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F26D28154C2B04"              , vprolvd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D28154C2B04"              , vprolvd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D4815CB"                  , vprolvd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48154C2B02"              , vprolvd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48154C2B02"              , vprolvd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED0815CB"                  , vprolvq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED08154C2B08"              , vprolvq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED08154C2B08"              , vprolvq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED2815CB"                  , vprolvq(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED28154C2B04"              , vprolvq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED28154C2B04"              , vprolvq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED4815CB"                  , vprolvq(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48154C2B02"              , vprolvq(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48154C2B02"              , vprolvq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1750872C201"                , vprord(xmm1, xmm2, 1));
  TEST_INSTRUCTION("62F1750872441A0801"            , vprord(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1750872441A0801"            , vprord(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1752872C201"                , vprord(ymm1, ymm2, 1));
  TEST_INSTRUCTION("62F1752872441A0401"            , vprord(ymm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1752872441A0401"            , vprord(ymm1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1754872C201"                , vprord(zmm1, zmm2, 1));
  TEST_INSTRUCTION("62F1754872441A0201"            , vprord(zmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1754872441A0201"            , vprord(zmm1, zmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1F50872C201"                , vprorq(xmm1, xmm2, 1));
  TEST_INSTRUCTION("62F1F50872441A0801"            , vprorq(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1F50872441A0801"            , vprorq(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1F52872C201"                , vprorq(ymm1, ymm2, 1));
  TEST_INSTRUCTION("62F1F52872441A0401"            , vprorq(ymm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1F52872441A0401"            , vprorq(ymm1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1F54872C201"                , vprorq(zmm1, zmm2, 1));
  TEST_INSTRUCTION("62F1F54872441A0201"            , vprorq(zmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1F54872441A0201"            , vprorq(zmm1, zmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F26D0814CB"                  , vprorvd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26D08144C2B08"              , vprorvd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D08144C2B08"              , vprorvd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D2814CB"                  , vprorvd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F26D28144C2B04"              , vprorvd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D28144C2B04"              , vprorvd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D4814CB"                  , vprorvd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48144C2B02"              , vprorvd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48144C2B02"              , vprorvd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED0814CB"                  , vprorvq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED08144C2B08"              , vprorvq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED08144C2B08"              , vprorvq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED2814CB"                  , vprorvq(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED28144C2B04"              , vprorvq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED28144C2B04"              , vprorvq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED4814CB"                  , vprorvq(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48144C2B02"              , vprorvq(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48144C2B02"              , vprorvq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48F6CB"                  , vpsadbw(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48F64C2B02"              , vpsadbw(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48F64C2B02"              , vpsadbw(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F27D09A05C1120"              , k(k1).vpscatterdd(ptr(ecx, xmm2, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27D29A05C1120"              , k(k1).vpscatterdd(ptr(ecx, ymm2, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27D49A05C1120"              , k(k1).vpscatterdd(ptr(ecx, zmm2, 0, 128), zmm3));
  TEST_INSTRUCTION("62F2FD09A05C1110"              , k(k1).vpscatterdq(ptr(ecx, xmm2, 0, 128), xmm3));
  TEST_INSTRUCTION("62F2FD29A05C1110"              , k(k1).vpscatterdq(ptr(ecx, xmm2, 0, 128), ymm3));
  TEST_INSTRUCTION("62F2FD49A05C1110"              , k(k1).vpscatterdq(ptr(ecx, ymm2, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27D09A15C1120"              , k(k1).vpscatterqd(ptr(ecx, xmm2, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27D29A15C1120"              , k(k1).vpscatterqd(ptr(ecx, ymm2, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27D49A15C1120"              , k(k1).vpscatterqd(ptr(ecx, zmm2, 0, 128), ymm3));
  TEST_INSTRUCTION("62F2FD09A15C1110"              , k(k1).vpscatterqq(ptr(ecx, xmm2, 0, 128), xmm3));
  TEST_INSTRUCTION("62F2FD29A15C1110"              , k(k1).vpscatterqq(ptr(ecx, ymm2, 0, 128), ymm3));
  TEST_INSTRUCTION("62F2FD49A15C1110"              , k(k1).vpscatterqq(ptr(ecx, zmm2, 0, 128), zmm3));
  TEST_INSTRUCTION("62F36D0871CB01"                , vpshldd(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F36D08714C2B0801"            , vpshldd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D08714C2B0801"            , vpshldd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D2871CB01"                , vpshldd(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("62F36D28714C2B0401"            , vpshldd(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D28714C2B0401"            , vpshldd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D4871CB01"                , vpshldd(zmm1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F36D48714C2B0201"            , vpshldd(zmm1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D48714C2B0201"            , vpshldd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED0871CB01"                , vpshldq(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F3ED08714C2B0801"            , vpshldq(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED08714C2B0801"            , vpshldq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED2871CB01"                , vpshldq(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("62F3ED28714C2B0401"            , vpshldq(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED28714C2B0401"            , vpshldq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED4871CB01"                , vpshldq(zmm1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F3ED48714C2B0201"            , vpshldq(zmm1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED48714C2B0201"            , vpshldq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F26D0871CB"                  , vpshldvd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26D08714C2B08"              , vpshldvd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D08714C2B08"              , vpshldvd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D2871CB"                  , vpshldvd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F26D28714C2B04"              , vpshldvd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D28714C2B04"              , vpshldvd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D4871CB"                  , vpshldvd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48714C2B02"              , vpshldvd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48714C2B02"              , vpshldvd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED0871CB"                  , vpshldvq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED08714C2B08"              , vpshldvq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED08714C2B08"              , vpshldvq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED2871CB"                  , vpshldvq(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED28714C2B04"              , vpshldvq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED28714C2B04"              , vpshldvq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED4871CB"                  , vpshldvq(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48714C2B02"              , vpshldvq(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48714C2B02"              , vpshldvq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED0870CB"                  , vpshldvw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED08704C2B08"              , vpshldvw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED08704C2B08"              , vpshldvw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED2870CB"                  , vpshldvw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED28704C2B04"              , vpshldvw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED28704C2B04"              , vpshldvw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED4870CB"                  , vpshldvw(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48704C2B02"              , vpshldvw(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48704C2B02"              , vpshldvw(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F3ED0870CB01"                , vpshldw(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F3ED08704C2B0801"            , vpshldw(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED08704C2B0801"            , vpshldw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED2870CB01"                , vpshldw(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("62F3ED28704C2B0401"            , vpshldw(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED28704C2B0401"            , vpshldw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED4870CB01"                , vpshldw(zmm1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F3ED48704C2B0201"            , vpshldw(zmm1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED48704C2B0201"            , vpshldw(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D0873CB01"                , vpshrdd(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F36D08734C2B0801"            , vpshrdd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D08734C2B0801"            , vpshrdd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D2873CB01"                , vpshrdd(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("62F36D28734C2B0401"            , vpshrdd(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D28734C2B0401"            , vpshrdd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D4873CB01"                , vpshrdd(zmm1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F36D48734C2B0201"            , vpshrdd(zmm1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D48734C2B0201"            , vpshrdd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED0873CB01"                , vpshrdq(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F3ED08734C2B0801"            , vpshrdq(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED08734C2B0801"            , vpshrdq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED2873CB01"                , vpshrdq(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("62F3ED28734C2B0401"            , vpshrdq(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED28734C2B0401"            , vpshrdq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED4873CB01"                , vpshrdq(zmm1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F3ED48734C2B0201"            , vpshrdq(zmm1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED48734C2B0201"            , vpshrdq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F26D0873CB"                  , vpshrdvd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26D08734C2B08"              , vpshrdvd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D08734C2B08"              , vpshrdvd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D2873CB"                  , vpshrdvd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F26D28734C2B04"              , vpshrdvd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D28734C2B04"              , vpshrdvd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D4873CB"                  , vpshrdvd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48734C2B02"              , vpshrdvd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48734C2B02"              , vpshrdvd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED0873CB"                  , vpshrdvq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED08734C2B08"              , vpshrdvq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED08734C2B08"              , vpshrdvq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED2873CB"                  , vpshrdvq(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED28734C2B04"              , vpshrdvq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED28734C2B04"              , vpshrdvq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED4873CB"                  , vpshrdvq(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48734C2B02"              , vpshrdvq(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48734C2B02"              , vpshrdvq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED0872CB"                  , vpshrdvw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED08724C2B08"              , vpshrdvw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED08724C2B08"              , vpshrdvw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED2872CB"                  , vpshrdvw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED28724C2B04"              , vpshrdvw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED28724C2B04"              , vpshrdvw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED4872CB"                  , vpshrdvw(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48724C2B02"              , vpshrdvw(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48724C2B02"              , vpshrdvw(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F3ED0872CB01"                , vpshrdw(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F3ED08724C2B0801"            , vpshrdw(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED08724C2B0801"            , vpshrdw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED2872CB01"                , vpshrdw(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("62F3ED28724C2B0401"            , vpshrdw(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED28724C2B0401"            , vpshrdw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED4872CB01"                , vpshrdw(zmm1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F3ED48724C2B0201"            , vpshrdw(zmm1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED48724C2B0201"            , vpshrdw(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F26D4800CB"                  , vpshufb(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48004C2B02"              , vpshufb(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48004C2B02"              , vpshufb(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D088FCB"                  , vpshufbitqmb(k1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26D088F4C2B08"              , vpshufbitqmb(k1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D088F4C2B08"              , vpshufbitqmb(k1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D288FCB"                  , vpshufbitqmb(k1, ymm2, ymm3));
  TEST_INSTRUCTION("62F26D288F4C2B04"              , vpshufbitqmb(k1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D288F4C2B04"              , vpshufbitqmb(k1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D488FCB"                  , vpshufbitqmb(k1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D488F4C2B02"              , vpshufbitqmb(k1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D488F4C2B02"              , vpshufbitqmb(k1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F17D4870CA01"                , vpshufd(zmm1, zmm2, 1));
  TEST_INSTRUCTION("62F17D48704C1A0201"            , vpshufd(zmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F17D48704C1A0201"            , vpshufd(zmm1, zmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F17E4870CA01"                , vpshufhw(zmm1, zmm2, 1));
  TEST_INSTRUCTION("62F17E48704C1A0201"            , vpshufhw(zmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F17E48704C1A0201"            , vpshufhw(zmm1, zmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F17F4870CA01"                , vpshuflw(zmm1, zmm2, 1));
  TEST_INSTRUCTION("62F17F48704C1A0201"            , vpshuflw(zmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F17F48704C1A0201"            , vpshuflw(zmm1, zmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1750872741A0801"            , vpslld(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1750872741A0801"            , vpslld(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1752872741A0401"            , vpslld(ymm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1752872741A0401"            , vpslld(ymm1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F16D48F2CB"                  , vpslld(zmm1, zmm2, xmm3));
  TEST_INSTRUCTION("62F1754872F201"                , vpslld(zmm1, zmm2, 1));
  TEST_INSTRUCTION("62F16D48F24C2B08"              , vpslld(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48F24C2B08"              , vpslld(zmm1, zmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1754872741A0201"            , vpslld(zmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1754872741A0201"            , vpslld(zmm1, zmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F17508737C1A0801"            , vpslldq(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F17508737C1A0801"            , vpslldq(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F17528737C1A0401"            , vpslldq(ymm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F17528737C1A0401"            , vpslldq(ymm1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1754873FA01"                , vpslldq(zmm1, zmm2, 1));
  TEST_INSTRUCTION("62F17548737C1A0201"            , vpslldq(zmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F17548737C1A0201"            , vpslldq(zmm1, zmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1F50873741A0801"            , vpsllq(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1F50873741A0801"            , vpsllq(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1F52873741A0401"            , vpsllq(ymm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1F52873741A0401"            , vpsllq(ymm1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1ED48F3CB"                  , vpsllq(zmm1, zmm2, xmm3));
  TEST_INSTRUCTION("62F1F54873F201"                , vpsllq(zmm1, zmm2, 1));
  TEST_INSTRUCTION("62F1ED48F34C2B08"              , vpsllq(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED48F34C2B08"              , vpsllq(zmm1, zmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1F54873741A0201"            , vpsllq(zmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1F54873741A0201"            , vpsllq(zmm1, zmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F26D4847CB"                  , vpsllvd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48474C2B02"              , vpsllvd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48474C2B02"              , vpsllvd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED4847CB"                  , vpsllvq(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48474C2B02"              , vpsllvq(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48474C2B02"              , vpsllvq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED0812CB"                  , vpsllvw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED08124C2B08"              , vpsllvw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED08124C2B08"              , vpsllvw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED2812CB"                  , vpsllvw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED28124C2B04"              , vpsllvw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED28124C2B04"              , vpsllvw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED4812CB"                  , vpsllvw(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48124C2B02"              , vpsllvw(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48124C2B02"              , vpsllvw(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1750871741A0801"            , vpsllw(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1750871741A0801"            , vpsllw(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1752871741A0401"            , vpsllw(ymm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1752871741A0401"            , vpsllw(ymm1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F16D48F1CB"                  , vpsllw(zmm1, zmm2, xmm3));
  TEST_INSTRUCTION("62F1754871F201"                , vpsllw(zmm1, zmm2, 1));
  TEST_INSTRUCTION("62F16D48F14C2B08"              , vpsllw(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48F14C2B08"              , vpsllw(zmm1, zmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1754871741A0201"            , vpsllw(zmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1754871741A0201"            , vpsllw(zmm1, zmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1750872641A0801"            , vpsrad(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1750872641A0801"            , vpsrad(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1752872641A0401"            , vpsrad(ymm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1752872641A0401"            , vpsrad(ymm1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F16D48E2CB"                  , vpsrad(zmm1, zmm2, xmm3));
  TEST_INSTRUCTION("62F1754872E201"                , vpsrad(zmm1, zmm2, 1));
  TEST_INSTRUCTION("62F16D48E24C2B08"              , vpsrad(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48E24C2B08"              , vpsrad(zmm1, zmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1754872641A0201"            , vpsrad(zmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1754872641A0201"            , vpsrad(zmm1, zmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1ED08E2CB"                  , vpsraq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F1F50872E201"                , vpsraq(xmm1, xmm2, 1));
  TEST_INSTRUCTION("62F1ED08E24C2B08"              , vpsraq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED08E24C2B08"              , vpsraq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1F50872641A0801"            , vpsraq(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1F50872641A0801"            , vpsraq(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1ED28E2CB"                  , vpsraq(ymm1, ymm2, xmm3));
  TEST_INSTRUCTION("62F1F52872E201"                , vpsraq(ymm1, ymm2, 1));
  TEST_INSTRUCTION("62F1ED28E24C2B08"              , vpsraq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED28E24C2B08"              , vpsraq(ymm1, ymm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1F52872641A0401"            , vpsraq(ymm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1F52872641A0401"            , vpsraq(ymm1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1ED48E2CB"                  , vpsraq(zmm1, zmm2, xmm3));
  TEST_INSTRUCTION("62F1F54872E201"                , vpsraq(zmm1, zmm2, 1));
  TEST_INSTRUCTION("62F1ED48E24C2B08"              , vpsraq(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED48E24C2B08"              , vpsraq(zmm1, zmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1F54872641A0201"            , vpsraq(zmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1F54872641A0201"            , vpsraq(zmm1, zmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F26D4846CB"                  , vpsravd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48464C2B02"              , vpsravd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48464C2B02"              , vpsravd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED0846CB"                  , vpsravq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED08464C2B08"              , vpsravq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED08464C2B08"              , vpsravq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED2846CB"                  , vpsravq(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED28464C2B04"              , vpsravq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED28464C2B04"              , vpsravq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED4846CB"                  , vpsravq(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48464C2B02"              , vpsravq(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48464C2B02"              , vpsravq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED0811CB"                  , vpsravw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED08114C2B08"              , vpsravw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED08114C2B08"              , vpsravw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED2811CB"                  , vpsravw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED28114C2B04"              , vpsravw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED28114C2B04"              , vpsravw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED4811CB"                  , vpsravw(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48114C2B02"              , vpsravw(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48114C2B02"              , vpsravw(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1750871641A0801"            , vpsraw(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1750871641A0801"            , vpsraw(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1752871641A0401"            , vpsraw(ymm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1752871641A0401"            , vpsraw(ymm1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F16D48E1CB"                  , vpsraw(zmm1, zmm2, xmm3));
  TEST_INSTRUCTION("62F1754871E201"                , vpsraw(zmm1, zmm2, 1));
  TEST_INSTRUCTION("62F16D48E14C2B08"              , vpsraw(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48E14C2B08"              , vpsraw(zmm1, zmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1754871641A0201"            , vpsraw(zmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1754871641A0201"            , vpsraw(zmm1, zmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1750872541A0801"            , vpsrld(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1750872541A0801"            , vpsrld(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1752872541A0401"            , vpsrld(ymm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1752872541A0401"            , vpsrld(ymm1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F16D48D2CB"                  , vpsrld(zmm1, zmm2, xmm3));
  TEST_INSTRUCTION("62F1754872D201"                , vpsrld(zmm1, zmm2, 1));
  TEST_INSTRUCTION("62F16D48D24C2B08"              , vpsrld(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48D24C2B08"              , vpsrld(zmm1, zmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1754872541A0201"            , vpsrld(zmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1754872541A0201"            , vpsrld(zmm1, zmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F17508735C1A0801"            , vpsrldq(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F17508735C1A0801"            , vpsrldq(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F17528735C1A0401"            , vpsrldq(ymm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F17528735C1A0401"            , vpsrldq(ymm1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1754873DA01"                , vpsrldq(zmm1, zmm2, 1));
  TEST_INSTRUCTION("62F17548735C1A0201"            , vpsrldq(zmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F17548735C1A0201"            , vpsrldq(zmm1, zmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1F50873541A0801"            , vpsrlq(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1F50873541A0801"            , vpsrlq(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1F52873541A0401"            , vpsrlq(ymm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1F52873541A0401"            , vpsrlq(ymm1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1ED48D3CB"                  , vpsrlq(zmm1, zmm2, xmm3));
  TEST_INSTRUCTION("62F1F54873D201"                , vpsrlq(zmm1, zmm2, 1));
  TEST_INSTRUCTION("62F1ED48D34C2B08"              , vpsrlq(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED48D34C2B08"              , vpsrlq(zmm1, zmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1F54873541A0201"            , vpsrlq(zmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1F54873541A0201"            , vpsrlq(zmm1, zmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F26D4845CB"                  , vpsrlvd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48454C2B02"              , vpsrlvd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48454C2B02"              , vpsrlvd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED4845CB"                  , vpsrlvq(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48454C2B02"              , vpsrlvq(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48454C2B02"              , vpsrlvq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED0810CB"                  , vpsrlvw(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED08104C2B08"              , vpsrlvw(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED08104C2B08"              , vpsrlvw(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED2810CB"                  , vpsrlvw(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED28104C2B04"              , vpsrlvw(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED28104C2B04"              , vpsrlvw(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED4810CB"                  , vpsrlvw(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48104C2B02"              , vpsrlvw(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48104C2B02"              , vpsrlvw(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1750871541A0801"            , vpsrlw(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1750871541A0801"            , vpsrlw(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1752871541A0401"            , vpsrlw(ymm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1752871541A0401"            , vpsrlw(ymm1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F16D48D1CB"                  , vpsrlw(zmm1, zmm2, xmm3));
  TEST_INSTRUCTION("62F1754871D201"                , vpsrlw(zmm1, zmm2, 1));
  TEST_INSTRUCTION("62F16D48D14C2B08"              , vpsrlw(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48D14C2B08"              , vpsrlw(zmm1, zmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1754871541A0201"            , vpsrlw(zmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F1754871541A0201"            , vpsrlw(zmm1, zmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F16D48F8CB"                  , vpsubb(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48F84C2B02"              , vpsubb(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48F84C2B02"              , vpsubb(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48FACB"                  , vpsubd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48FA4C2B02"              , vpsubd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48FA4C2B02"              , vpsubd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED48FBCB"                  , vpsubq(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F1ED48FB4C2B02"              , vpsubq(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED48FB4C2B02"              , vpsubq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48E8CB"                  , vpsubsb(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48E84C2B02"              , vpsubsb(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48E84C2B02"              , vpsubsb(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48E9CB"                  , vpsubsw(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48E94C2B02"              , vpsubsw(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48E94C2B02"              , vpsubsw(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48D8CB"                  , vpsubusb(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48D84C2B02"              , vpsubusb(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48D84C2B02"              , vpsubusb(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48D9CB"                  , vpsubusw(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48D94C2B02"              , vpsubusw(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48D94C2B02"              , vpsubusw(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48F9CB"                  , vpsubw(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48F94C2B02"              , vpsubw(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48F94C2B02"              , vpsubw(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F36D0825CB01"                , vpternlogd(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F36D08254C2B0801"            , vpternlogd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D08254C2B0801"            , vpternlogd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D2825CB01"                , vpternlogd(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("62F36D28254C2B0401"            , vpternlogd(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D28254C2B0401"            , vpternlogd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D4825CB01"                , vpternlogd(zmm1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F36D48254C2B0201"            , vpternlogd(zmm1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D48254C2B0201"            , vpternlogd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED0825CB01"                , vpternlogq(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F3ED08254C2B0801"            , vpternlogq(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED08254C2B0801"            , vpternlogq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED2825CB01"                , vpternlogq(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("62F3ED28254C2B0401"            , vpternlogq(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED28254C2B0401"            , vpternlogq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED4825CB01"                , vpternlogq(zmm1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F3ED48254C2B0201"            , vpternlogq(zmm1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED48254C2B0201"            , vpternlogq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F26D0826CB"                  , vptestmb(k1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26D08264C2B08"              , vptestmb(k1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D08264C2B08"              , vptestmb(k1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D2826CB"                  , vptestmb(k1, ymm2, ymm3));
  TEST_INSTRUCTION("62F26D28264C2B04"              , vptestmb(k1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D28264C2B04"              , vptestmb(k1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D4826CB"                  , vptestmb(k1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48264C2B02"              , vptestmb(k1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48264C2B02"              , vptestmb(k1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D0827CB"                  , vptestmd(k1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26D08274C2B08"              , vptestmd(k1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D08274C2B08"              , vptestmd(k1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D2827CB"                  , vptestmd(k1, ymm2, ymm3));
  TEST_INSTRUCTION("62F26D28274C2B04"              , vptestmd(k1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D28274C2B04"              , vptestmd(k1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D4827CB"                  , vptestmd(k1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D48274C2B02"              , vptestmd(k1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D48274C2B02"              , vptestmd(k1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED0827CB"                  , vptestmq(k1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED08274C2B08"              , vptestmq(k1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED08274C2B08"              , vptestmq(k1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED2827CB"                  , vptestmq(k1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED28274C2B04"              , vptestmq(k1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED28274C2B04"              , vptestmq(k1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED4827CB"                  , vptestmq(k1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48274C2B02"              , vptestmq(k1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48274C2B02"              , vptestmq(k1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED0826CB"                  , vptestmw(k1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED08264C2B08"              , vptestmw(k1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED08264C2B08"              , vptestmw(k1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED2826CB"                  , vptestmw(k1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED28264C2B04"              , vptestmw(k1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED28264C2B04"              , vptestmw(k1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED4826CB"                  , vptestmw(k1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED48264C2B02"              , vptestmw(k1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED48264C2B02"              , vptestmw(k1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26E0826CB"                  , vptestnmb(k1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26E08264C2B08"              , vptestnmb(k1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26E08264C2B08"              , vptestnmb(k1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26E2826CB"                  , vptestnmb(k1, ymm2, ymm3));
  TEST_INSTRUCTION("62F26E28264C2B04"              , vptestnmb(k1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26E28264C2B04"              , vptestnmb(k1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26E4826CB"                  , vptestnmb(k1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26E48264C2B02"              , vptestnmb(k1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26E48264C2B02"              , vptestnmb(k1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26E0827CB"                  , vptestnmd(k1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26E08274C2B08"              , vptestnmd(k1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26E08274C2B08"              , vptestnmd(k1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26E2827CB"                  , vptestnmd(k1, ymm2, ymm3));
  TEST_INSTRUCTION("62F26E28274C2B04"              , vptestnmd(k1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26E28274C2B04"              , vptestnmd(k1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26E4827CB"                  , vptestnmd(k1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26E48274C2B02"              , vptestnmd(k1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26E48274C2B02"              , vptestnmd(k1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2EE0827CB"                  , vptestnmq(k1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2EE08274C2B08"              , vptestnmq(k1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2EE08274C2B08"              , vptestnmq(k1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2EE2827CB"                  , vptestnmq(k1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2EE28274C2B04"              , vptestnmq(k1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2EE28274C2B04"              , vptestnmq(k1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2EE4827CB"                  , vptestnmq(k1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2EE48274C2B02"              , vptestnmq(k1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2EE48274C2B02"              , vptestnmq(k1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2EE0826CB"                  , vptestnmw(k1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2EE08264C2B08"              , vptestnmw(k1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2EE08264C2B08"              , vptestnmw(k1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2EE2826CB"                  , vptestnmw(k1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2EE28264C2B04"              , vptestnmw(k1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2EE28264C2B04"              , vptestnmw(k1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2EE4826CB"                  , vptestnmw(k1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2EE48264C2B02"              , vptestnmw(k1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2EE48264C2B02"              , vptestnmw(k1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D4868CB"                  , vpunpckhbw(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48684C2B02"              , vpunpckhbw(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48684C2B02"              , vpunpckhbw(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D486ACB"                  , vpunpckhdq(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D486A4C2B02"              , vpunpckhdq(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D486A4C2B02"              , vpunpckhdq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED486DCB"                  , vpunpckhqdq(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F1ED486D4C2B02"              , vpunpckhqdq(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED486D4C2B02"              , vpunpckhqdq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D4869CB"                  , vpunpckhwd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48694C2B02"              , vpunpckhwd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48694C2B02"              , vpunpckhwd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D4860CB"                  , vpunpcklbw(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48604C2B02"              , vpunpcklbw(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48604C2B02"              , vpunpcklbw(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D4862CB"                  , vpunpckldq(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48624C2B02"              , vpunpckldq(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48624C2B02"              , vpunpckldq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED486CCB"                  , vpunpcklqdq(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F1ED486C4C2B02"              , vpunpcklqdq(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED486C4C2B02"              , vpunpcklqdq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D4861CB"                  , vpunpcklwd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48614C2B02"              , vpunpcklwd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48614C2B02"              , vpunpcklwd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D08EFCB"                  , vpxord(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F16D08EF4C2B08"              , vpxord(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D08EF4C2B08"              , vpxord(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D28EFCB"                  , vpxord(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F16D28EF4C2B04"              , vpxord(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D28EF4C2B04"              , vpxord(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48EFCB"                  , vpxord(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16D48EF4C2B02"              , vpxord(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16D48EF4C2B02"              , vpxord(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED08EFCB"                  , vpxorq(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F1ED08EF4C2B08"              , vpxorq(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED08EF4C2B08"              , vpxorq(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED28EFCB"                  , vpxorq(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F1ED28EF4C2B04"              , vpxorq(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED28EF4C2B04"              , vpxorq(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED48EFCB"                  , vpxorq(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F1ED48EF4C2B02"              , vpxorq(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED48EF4C2B02"              , vpxorq(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F3ED0850CB01"                , vrangepd(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F3ED08504C2B0801"            , vrangepd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED08504C2B0801"            , vrangepd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED2850CB01"                , vrangepd(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("62F3ED28504C2B0401"            , vrangepd(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED28504C2B0401"            , vrangepd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED4850CB01"                , vrangepd(zmm1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F3ED48504C2B0201"            , vrangepd(zmm1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED48504C2B0201"            , vrangepd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D0850CB01"                , vrangeps(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F36D08504C2B0801"            , vrangeps(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D08504C2B0801"            , vrangeps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D2850CB01"                , vrangeps(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("62F36D28504C2B0401"            , vrangeps(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D28504C2B0401"            , vrangeps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D4850CB01"                , vrangeps(zmm1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F36D48504C2B0201"            , vrangeps(zmm1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D48504C2B0201"            , vrangeps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED0851CB01"                , vrangesd(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F3ED08514C2B1001"            , vrangesd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED08514C2B1001"            , vrangesd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D0851CB01"                , vrangess(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F36D08514C2B2001"            , vrangess(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D08514C2B2001"            , vrangess(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F2FD084CCA"                  , vrcp14pd(xmm1, xmm2));
  TEST_INSTRUCTION("62F2FD084C4C1A08"              , vrcp14pd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD084C4C1A08"              , vrcp14pd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD284CCA"                  , vrcp14pd(ymm1, ymm2));
  TEST_INSTRUCTION("62F2FD284C4C1A04"              , vrcp14pd(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD284C4C1A04"              , vrcp14pd(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD484CCA"                  , vrcp14pd(zmm1, zmm2));
  TEST_INSTRUCTION("62F2FD484C4C1A02"              , vrcp14pd(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD484C4C1A02"              , vrcp14pd(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D084CCA"                  , vrcp14ps(xmm1, xmm2));
  TEST_INSTRUCTION("62F27D084C4C1A08"              , vrcp14ps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D084C4C1A08"              , vrcp14ps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D284CCA"                  , vrcp14ps(ymm1, ymm2));
  TEST_INSTRUCTION("62F27D284C4C1A04"              , vrcp14ps(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D284C4C1A04"              , vrcp14ps(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D484CCA"                  , vrcp14ps(zmm1, zmm2));
  TEST_INSTRUCTION("62F27D484C4C1A02"              , vrcp14ps(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D484C4C1A02"              , vrcp14ps(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2ED084DCB"                  , vrcp14sd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED084D4C2B10"              , vrcp14sd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED084D4C2B10"              , vrcp14sd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D084DCB"                  , vrcp14ss(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26D084D4C2B20"              , vrcp14ss(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D084D4C2B20"              , vrcp14ss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2FD48CACA"                  , vrcp28pd(zmm1, zmm2));
  TEST_INSTRUCTION("62F2FD48CA4C1A02"              , vrcp28pd(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD48CA4C1A02"              , vrcp28pd(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48CACA"                  , vrcp28ps(zmm1, zmm2));
  TEST_INSTRUCTION("62F27D48CA4C1A02"              , vrcp28ps(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48CA4C1A02"              , vrcp28ps(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2ED08CBCB"                  , vrcp28sd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED08CB4C2B10"              , vrcp28sd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED08CB4C2B10"              , vrcp28sd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D08CBCB"                  , vrcp28ss(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26D08CB4C2B20"              , vrcp28ss(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D08CB4C2B20"              , vrcp28ss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F3FD0856CA01"                , vreducepd(xmm1, xmm2, 1));
  TEST_INSTRUCTION("62F3FD08564C1A0801"            , vreducepd(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F3FD08564C1A0801"            , vreducepd(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F3FD2856CA01"                , vreducepd(ymm1, ymm2, 1));
  TEST_INSTRUCTION("62F3FD28564C1A0401"            , vreducepd(ymm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F3FD28564C1A0401"            , vreducepd(ymm1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F3FD4856CA01"                , vreducepd(zmm1, zmm2, 1));
  TEST_INSTRUCTION("62F3FD48564C1A0201"            , vreducepd(zmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F3FD48564C1A0201"            , vreducepd(zmm1, zmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F37D0856CA01"                , vreduceps(xmm1, xmm2, 1));
  TEST_INSTRUCTION("62F37D08564C1A0801"            , vreduceps(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F37D08564C1A0801"            , vreduceps(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F37D2856CA01"                , vreduceps(ymm1, ymm2, 1));
  TEST_INSTRUCTION("62F37D28564C1A0401"            , vreduceps(ymm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F37D28564C1A0401"            , vreduceps(ymm1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F37D4856CA01"                , vreduceps(zmm1, zmm2, 1));
  TEST_INSTRUCTION("62F37D48564C1A0201"            , vreduceps(zmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F37D48564C1A0201"            , vreduceps(zmm1, zmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED0857CB01"                , vreducesd(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F3ED08574C2B1001"            , vreducesd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED08574C2B1001"            , vreducesd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D0857CB01"                , vreducess(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F36D08574C2B2001"            , vreducess(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D08574C2B2001"            , vreducess(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3FD0809CA01"                , vrndscalepd(xmm1, xmm2, 1));
  TEST_INSTRUCTION("62F3FD08094C1A0801"            , vrndscalepd(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F3FD08094C1A0801"            , vrndscalepd(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F3FD2809CA01"                , vrndscalepd(ymm1, ymm2, 1));
  TEST_INSTRUCTION("62F3FD28094C1A0401"            , vrndscalepd(ymm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F3FD28094C1A0401"            , vrndscalepd(ymm1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F3FD4809CA01"                , vrndscalepd(zmm1, zmm2, 1));
  TEST_INSTRUCTION("62F3FD48094C1A0201"            , vrndscalepd(zmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F3FD48094C1A0201"            , vrndscalepd(zmm1, zmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F37D0808CA01"                , vrndscaleps(xmm1, xmm2, 1));
  TEST_INSTRUCTION("62F37D08084C1A0801"            , vrndscaleps(xmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F37D08084C1A0801"            , vrndscaleps(xmm1, xmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F37D2808CA01"                , vrndscaleps(ymm1, ymm2, 1));
  TEST_INSTRUCTION("62F37D28084C1A0401"            , vrndscaleps(ymm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F37D28084C1A0401"            , vrndscaleps(ymm1, ymmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F37D4808CA01"                , vrndscaleps(zmm1, zmm2, 1));
  TEST_INSTRUCTION("62F37D48084C1A0201"            , vrndscaleps(zmm1, ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F37D48084C1A0201"            , vrndscaleps(zmm1, zmmword_ptr(edx, ebx, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED080BCB01"                , vrndscalesd(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F3ED080B4C2B1001"            , vrndscalesd(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED080B4C2B1001"            , vrndscalesd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D080ACB01"                , vrndscaless(xmm1, xmm2, xmm3, 1));
  TEST_INSTRUCTION("62F36D080A4C2B2001"            , vrndscaless(xmm1, xmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D080A4C2B2001"            , vrndscaless(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F2FD084ECA"                  , vrsqrt14pd(xmm1, xmm2));
  TEST_INSTRUCTION("62F2FD084E4C1A08"              , vrsqrt14pd(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD084E4C1A08"              , vrsqrt14pd(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD284ECA"                  , vrsqrt14pd(ymm1, ymm2));
  TEST_INSTRUCTION("62F2FD284E4C1A04"              , vrsqrt14pd(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD284E4C1A04"              , vrsqrt14pd(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD484ECA"                  , vrsqrt14pd(zmm1, zmm2));
  TEST_INSTRUCTION("62F2FD484E4C1A02"              , vrsqrt14pd(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD484E4C1A02"              , vrsqrt14pd(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D084ECA"                  , vrsqrt14ps(xmm1, xmm2));
  TEST_INSTRUCTION("62F27D084E4C1A08"              , vrsqrt14ps(xmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D084E4C1A08"              , vrsqrt14ps(xmm1, xmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D284ECA"                  , vrsqrt14ps(ymm1, ymm2));
  TEST_INSTRUCTION("62F27D284E4C1A04"              , vrsqrt14ps(ymm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D284E4C1A04"              , vrsqrt14ps(ymm1, ymmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D484ECA"                  , vrsqrt14ps(zmm1, zmm2));
  TEST_INSTRUCTION("62F27D484E4C1A02"              , vrsqrt14ps(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D484E4C1A02"              , vrsqrt14ps(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2ED084FCB"                  , vrsqrt14sd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED084F4C2B10"              , vrsqrt14sd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED084F4C2B10"              , vrsqrt14sd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D084FCB"                  , vrsqrt14ss(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26D084F4C2B20"              , vrsqrt14ss(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D084F4C2B20"              , vrsqrt14ss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2FD48CCCA"                  , vrsqrt28pd(zmm1, zmm2));
  TEST_INSTRUCTION("62F2FD48CC4C1A02"              , vrsqrt28pd(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2FD48CC4C1A02"              , vrsqrt28pd(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48CCCA"                  , vrsqrt28ps(zmm1, zmm2));
  TEST_INSTRUCTION("62F27D48CC4C1A02"              , vrsqrt28ps(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F27D48CC4C1A02"              , vrsqrt28ps(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F2ED08CDCB"                  , vrsqrt28sd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED08CD4C2B10"              , vrsqrt28sd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED08CD4C2B10"              , vrsqrt28sd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D08CDCB"                  , vrsqrt28ss(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26D08CD4C2B20"              , vrsqrt28ss(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D08CD4C2B20"              , vrsqrt28ss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED082CCB"                  , vscalefpd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED082C4C2B08"              , vscalefpd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED082C4C2B08"              , vscalefpd(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED282CCB"                  , vscalefpd(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F2ED282C4C2B04"              , vscalefpd(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED282C4C2B04"              , vscalefpd(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED482CCB"                  , vscalefpd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F2ED482C4C2B02"              , vscalefpd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED482C4C2B02"              , vscalefpd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D082CCB"                  , vscalefps(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26D082C4C2B08"              , vscalefps(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D082C4C2B08"              , vscalefps(xmm1, xmm2, xmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D282CCB"                  , vscalefps(ymm1, ymm2, ymm3));
  TEST_INSTRUCTION("62F26D282C4C2B04"              , vscalefps(ymm1, ymm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D282C4C2B04"              , vscalefps(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D482CCB"                  , vscalefps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F26D482C4C2B02"              , vscalefps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D482C4C2B02"              , vscalefps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED082DCB"                  , vscalefsd(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F2ED082D4C2B10"              , vscalefsd(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2ED082D4C2B10"              , vscalefsd(xmm1, xmm2, qword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D082DCB"                  , vscalefss(xmm1, xmm2, xmm3));
  TEST_INSTRUCTION("62F26D082D4C2B20"              , vscalefss(xmm1, xmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F26D082D4C2B20"              , vscalefss(xmm1, xmm2, dword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F2FD09A25C1110"              , k(k1).vscatterdpd(ptr(ecx, xmm2, 0, 128), xmm3));
  TEST_INSTRUCTION("62F2FD29A25C1110"              , k(k1).vscatterdpd(ptr(ecx, xmm2, 0, 128), ymm3));
  TEST_INSTRUCTION("62F2FD49A25C1110"              , k(k1).vscatterdpd(ptr(ecx, ymm2, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27D09A25C1120"              , k(k1).vscatterdps(ptr(ecx, xmm2, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27D29A25C1120"              , k(k1).vscatterdps(ptr(ecx, ymm2, 0, 128), ymm3));
  TEST_INSTRUCTION("62F27D49A25C1120"              , k(k1).vscatterdps(ptr(ecx, zmm2, 0, 128), zmm3));
  TEST_INSTRUCTION("62F2FD49C66C1110"              , k(k1).vscatterpf0dpd(ptr(ecx, ymm2, 0, 128)));
  TEST_INSTRUCTION("62F27D49C66C1120"              , k(k1).vscatterpf0dps(ptr(ecx, zmm2, 0, 128)));
  TEST_INSTRUCTION("62F2FD49C76C1110"              , k(k1).vscatterpf0qpd(ptr(ecx, zmm2, 0, 128)));
  TEST_INSTRUCTION("62F27D49C76C1120"              , k(k1).vscatterpf0qps(ptr(ecx, zmm2, 0, 128)));
  TEST_INSTRUCTION("62F2FD49C6741110"              , k(k1).vscatterpf1dpd(ptr(ecx, ymm2, 0, 128)));
  TEST_INSTRUCTION("62F27D49C6741120"              , k(k1).vscatterpf1dps(ptr(ecx, zmm2, 0, 128)));
  TEST_INSTRUCTION("62F2FD49C7741110"              , k(k1).vscatterpf1qpd(ptr(ecx, zmm2, 0, 128)));
  TEST_INSTRUCTION("62F27D49C7741120"              , k(k1).vscatterpf1qps(ptr(ecx, zmm2, 0, 128)));
  TEST_INSTRUCTION("62F2FD09A35C1110"              , k(k1).vscatterqpd(ptr(ecx, xmm2, 0, 128), xmm3));
  TEST_INSTRUCTION("62F2FD29A35C1110"              , k(k1).vscatterqpd(ptr(ecx, ymm2, 0, 128), ymm3));
  TEST_INSTRUCTION("62F2FD49A35C1110"              , k(k1).vscatterqpd(ptr(ecx, zmm2, 0, 128), zmm3));
  TEST_INSTRUCTION("62F27D09A35C1120"              , k(k1).vscatterqps(ptr(ecx, xmm2, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27D29A35C1120"              , k(k1).vscatterqps(ptr(ecx, ymm2, 0, 128), xmm3));
  TEST_INSTRUCTION("62F27D49A35C1120"              , k(k1).vscatterqps(ptr(ecx, zmm2, 0, 128), ymm3));
  TEST_INSTRUCTION("62F36D2823CB01"                , vshuff32x4(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("62F36D28234C2B0401"            , vshuff32x4(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D28234C2B0401"            , vshuff32x4(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D4823CB01"                , vshuff32x4(zmm1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F36D48234C2B0201"            , vshuff32x4(zmm1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D48234C2B0201"            , vshuff32x4(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED2823CB01"                , vshuff64x2(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("62F3ED28234C2B0401"            , vshuff64x2(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED28234C2B0401"            , vshuff64x2(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED4823CB01"                , vshuff64x2(zmm1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F3ED48234C2B0201"            , vshuff64x2(zmm1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED48234C2B0201"            , vshuff64x2(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D2843CB01"                , vshufi32x4(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("62F36D28434C2B0401"            , vshufi32x4(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D28434C2B0401"            , vshufi32x4(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D4843CB01"                , vshufi32x4(zmm1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F36D48434C2B0201"            , vshufi32x4(zmm1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F36D48434C2B0201"            , vshufi32x4(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED2843CB01"                , vshufi64x2(ymm1, ymm2, ymm3, 1));
  TEST_INSTRUCTION("62F3ED28434C2B0401"            , vshufi64x2(ymm1, ymm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED28434C2B0401"            , vshufi64x2(ymm1, ymm2, ymmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED4843CB01"                , vshufi64x2(zmm1, zmm2, zmm3, 1));
  TEST_INSTRUCTION("62F3ED48434C2B0201"            , vshufi64x2(zmm1, zmm2, ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F3ED48434C2B0201"            , vshufi64x2(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128), 1));
  TEST_INSTRUCTION("62F1FD4851CA"                  , vsqrtpd(zmm1, zmm2));
  TEST_INSTRUCTION("62F1FD48514C1A02"              , vsqrtpd(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1FD48514C1A02"              , vsqrtpd(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17C4851CA"                  , vsqrtps(zmm1, zmm2));
  TEST_INSTRUCTION("62F17C48514C1A02"              , vsqrtps(zmm1, ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F17C48514C1A02"              , vsqrtps(zmm1, zmmword_ptr(edx, ebx, 0, 128)));
  TEST_INSTRUCTION("62F1ED485CCB"                  , vsubpd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F1ED485C4C2B02"              , vsubpd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED485C4C2B02"              , vsubpd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16C485CCB"                  , vsubps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16C485C4C2B02"              , vsubps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16C485C4C2B02"              , vsubps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED4815CB"                  , vunpckhpd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F1ED48154C2B02"              , vunpckhpd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED48154C2B02"              , vunpckhpd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16C4815CB"                  , vunpckhps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16C48154C2B02"              , vunpckhps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16C48154C2B02"              , vunpckhps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED4814CB"                  , vunpcklpd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F1ED48144C2B02"              , vunpcklpd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED48144C2B02"              , vunpcklpd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16C4814CB"                  , vunpcklps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16C48144C2B02"              , vunpcklps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16C48144C2B02"              , vunpcklps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED4857CB"                  , vxorpd(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F1ED48574C2B02"              , vxorpd(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F1ED48574C2B02"              , vxorpd(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16C4857CB"                  , vxorps(zmm1, zmm2, zmm3));
  TEST_INSTRUCTION("62F16C48574C2B02"              , vxorps(zmm1, zmm2, ptr(ebx, ebp, 0, 128)));
  TEST_INSTRUCTION("62F16C48574C2B02"              , vxorps(zmm1, zmm2, zmmword_ptr(ebx, ebp, 0, 128)));
}

static void ASMJIT_NOINLINE testX86AssemblerAVX512_FP16(AssemblerTester<x86::Assembler>& tester) noexcept {
  using namespace x86;

  TEST_INSTRUCTION("62F5560810F4"                  , vmovsh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F57E0F10B4F400000010"        , k(k7).vmovsh(xmm6, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F57E081031"                  , vmovsh(xmm6, word_ptr(ecx)));
  TEST_INSTRUCTION("62F57E0810717F"                , vmovsh(xmm6, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F57E8F107280"                , k(k7).z().vmovsh(xmm6, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F57E0F11B4F400000010"        , k(k7).vmovsh(word_ptr(esp, esi, 3, 268435456), xmm6));
  TEST_INSTRUCTION("62F57E081131"                  , vmovsh(word_ptr(ecx), xmm6));
  TEST_INSTRUCTION("62F57E0811717F"                , vmovsh(word_ptr(ecx, 254), xmm6));
  TEST_INSTRUCTION("62F57E0F117280"                , k(k7).vmovsh(word_ptr(edx, -256), xmm6));
  TEST_INSTRUCTION("62F57D086EF2"                  , vmovw(xmm6, edx));
  TEST_INSTRUCTION("62F57D087EF2"                  , vmovw(edx, xmm6));
  TEST_INSTRUCTION("62F57D086EB4F400000010"        , vmovw(xmm6, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F57D086E31"                  , vmovw(xmm6, word_ptr(ecx)));
  TEST_INSTRUCTION("62F57D086E717F"                , vmovw(xmm6, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F57D086E717F"                , vmovw(xmm6, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F57D086E7280"                , vmovw(xmm6, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F57D087EB4F400000010"        , vmovw(word_ptr(esp, esi, 3, 268435456), xmm6));
  TEST_INSTRUCTION("62F57D087E31"                  , vmovw(word_ptr(ecx), xmm6));
  TEST_INSTRUCTION("62F57D087E717F"                , vmovw(word_ptr(ecx, 254), xmm6));
  TEST_INSTRUCTION("62F57D087E7280"                , vmovw(word_ptr(edx, -256), xmm6));

  TEST_INSTRUCTION("62F5544858F4"                  , vaddph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F5541858F4"                  , rn_sae().vaddph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F5544F58B4F400000010"        , k(k7).vaddph(zmm6, zmm5, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F554585831"                  , vaddph(zmm6, zmm5, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F5544858717F"                , vaddph(zmm6, zmm5, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F554DF587280"                , k(k7).z().vaddph(zmm6, zmm5, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F5560858F4"                  , vaddsh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F5561858F4"                  , rn_sae().vaddsh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F5560F58B4F400000010"        , k(k7).vaddsh(xmm6, xmm5, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F556085831"                  , vaddsh(xmm6, xmm5, word_ptr(ecx)));
  TEST_INSTRUCTION("62F5560858717F"                , vaddsh(xmm6, xmm5, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F5568F587280"                , k(k7).z().vaddsh(xmm6, xmm5, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F35448C2EC7B"                , vcmpph(k5, zmm5, zmm4, 123));
  TEST_INSTRUCTION("62F35418C2EC7B"                , sae().vcmpph(k5, zmm5, zmm4, 123));
  TEST_INSTRUCTION("62F3544FC2ACF4000000107B"      , k(k7).vcmpph(k5, zmm5, zmmword_ptr(esp, esi, 3, 268435456), 123));
  TEST_INSTRUCTION("62F35458C2297B"                , vcmpph(k5, zmm5, word_ptr(ecx)._1to32(), 123));
  TEST_INSTRUCTION("62F35448C2697F7B"              , vcmpph(k5, zmm5, zmmword_ptr(ecx, 8128), 123));
  TEST_INSTRUCTION("62F3545FC26A807B"              , k(k7).vcmpph(k5, zmm5, word_ptr(edx, -256)._1to32(), 123));
  TEST_INSTRUCTION("62F35608C2EC7B"                , vcmpsh(k5, xmm5, xmm4, 123));
  TEST_INSTRUCTION("62F35618C2EC7B"                , sae().vcmpsh(k5, xmm5, xmm4, 123));
  TEST_INSTRUCTION("62F3560FC2ACF4000000107B"      , k(k7).vcmpsh(k5, xmm5, word_ptr(esp, esi, 3, 268435456), 123));
  TEST_INSTRUCTION("62F35608C2297B"                , vcmpsh(k5, xmm5, word_ptr(ecx), 123));
  TEST_INSTRUCTION("62F35608C2697F7B"              , vcmpsh(k5, xmm5, word_ptr(ecx, 254), 123));
  TEST_INSTRUCTION("62F3560FC26A807B"              , k(k7).vcmpsh(k5, xmm5, word_ptr(edx, -256), 123));
  TEST_INSTRUCTION("62F57C082FF5"                  , vcomish(xmm6, xmm5));
  TEST_INSTRUCTION("62F57C182FF5"                  , sae().vcomish(xmm6, xmm5));
  TEST_INSTRUCTION("62F57C082FB4F400000010"        , vcomish(xmm6, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F57C082F31"                  , vcomish(xmm6, word_ptr(ecx)));
  TEST_INSTRUCTION("62F57C082F717F"                , vcomish(xmm6, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F57C082F7280"                , vcomish(xmm6, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F57C485BF5"                  , vcvtdq2ph(ymm6, zmm5));
  TEST_INSTRUCTION("62F57C185BF5"                  , rn_sae().vcvtdq2ph(ymm6, zmm5));
  TEST_INSTRUCTION("62F57C4F5BB4F400000010"        , k(k7).vcvtdq2ph(ymm6, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F57C585B31"                  , vcvtdq2ph(ymm6, dword_ptr(ecx)._1to16()));
  TEST_INSTRUCTION("62F57C485B717F"                , vcvtdq2ph(ymm6, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F57CDF5B7280"                , k(k7).z().vcvtdq2ph(ymm6, dword_ptr(edx, -512)._1to16()));
  TEST_INSTRUCTION("62F5FD485AF5"                  , vcvtpd2ph(xmm6, zmm5));
  TEST_INSTRUCTION("62F5FD185AF5"                  , rn_sae().vcvtpd2ph(xmm6, zmm5));
  TEST_INSTRUCTION("62F5FD4F5AB4F400000010"        , k(k7).vcvtpd2ph(xmm6, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F5FD585A31"                  , vcvtpd2ph(xmm6, qword_ptr(ecx)._1to8()));
  TEST_INSTRUCTION("62F5FD485A717F"                , vcvtpd2ph(xmm6, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F5FDDF5A7280"                , k(k7).z().vcvtpd2ph(xmm6, qword_ptr(edx, -1024)._1to8()));
  TEST_INSTRUCTION("62F57D485BF5"                  , vcvtph2dq(zmm6, ymm5));
  TEST_INSTRUCTION("62F57D185BF5"                  , rn_sae().vcvtph2dq(zmm6, ymm5));
  TEST_INSTRUCTION("62F57D4F5BB4F400000010"        , k(k7).vcvtph2dq(zmm6, ymmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F57D585B31"                  , vcvtph2dq(zmm6, word_ptr(ecx)._1to16()));
  TEST_INSTRUCTION("62F57D485B717F"                , vcvtph2dq(zmm6, ymmword_ptr(ecx, 4064)));
  TEST_INSTRUCTION("62F57DDF5B7280"                , k(k7).z().vcvtph2dq(zmm6, word_ptr(edx, -256)._1to16()));
  TEST_INSTRUCTION("62F57C485AF5"                  , vcvtph2pd(zmm6, xmm5));
  TEST_INSTRUCTION("62F57C185AF5"                  , sae().vcvtph2pd(zmm6, xmm5));
  TEST_INSTRUCTION("62F57C4F5AB4F400000010"        , k(k7).vcvtph2pd(zmm6, xmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F57C585A31"                  , vcvtph2pd(zmm6, word_ptr(ecx)._1to8()));
  TEST_INSTRUCTION("62F57C485A717F"                , vcvtph2pd(zmm6, xmmword_ptr(ecx, 2032)));
  TEST_INSTRUCTION("62F57CDF5A7280"                , k(k7).z().vcvtph2pd(zmm6, word_ptr(edx, -256)._1to8()));
  TEST_INSTRUCTION("62F67D4813F5"                  , vcvtph2psx(zmm6, ymm5));
  TEST_INSTRUCTION("62F67D1813F5"                  , sae().vcvtph2psx(zmm6, ymm5));
  TEST_INSTRUCTION("62F67D4F13B4F400000010"        , k(k7).vcvtph2psx(zmm6, ymmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F67D581331"                  , vcvtph2psx(zmm6, word_ptr(ecx)._1to16()));
  TEST_INSTRUCTION("62F67D4813717F"                , vcvtph2psx(zmm6, ymmword_ptr(ecx, 4064)));
  TEST_INSTRUCTION("62F67DDF137280"                , k(k7).z().vcvtph2psx(zmm6, word_ptr(edx, -256)._1to16()));
  TEST_INSTRUCTION("62F57D487BF5"                  , vcvtph2qq(zmm6, xmm5));
  TEST_INSTRUCTION("62F57D187BF5"                  , rn_sae().vcvtph2qq(zmm6, xmm5));
  TEST_INSTRUCTION("62F57D4F7BB4F400000010"        , k(k7).vcvtph2qq(zmm6, xmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F57D587B31"                  , vcvtph2qq(zmm6, word_ptr(ecx)._1to8()));
  TEST_INSTRUCTION("62F57D487B717F"                , vcvtph2qq(zmm6, xmmword_ptr(ecx, 2032)));
  TEST_INSTRUCTION("62F57DDF7B7280"                , k(k7).z().vcvtph2qq(zmm6, word_ptr(edx, -256)._1to8()));
  TEST_INSTRUCTION("62F57C4879F5"                  , vcvtph2udq(zmm6, ymm5));
  TEST_INSTRUCTION("62F57C1879F5"                  , rn_sae().vcvtph2udq(zmm6, ymm5));
  TEST_INSTRUCTION("62F57C4F79B4F400000010"        , k(k7).vcvtph2udq(zmm6, ymmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F57C587931"                  , vcvtph2udq(zmm6, word_ptr(ecx)._1to16()));
  TEST_INSTRUCTION("62F57C4879717F"                , vcvtph2udq(zmm6, ymmword_ptr(ecx, 4064)));
  TEST_INSTRUCTION("62F57CDF797280"                , k(k7).z().vcvtph2udq(zmm6, word_ptr(edx, -256)._1to16()));
  TEST_INSTRUCTION("62F57D4879F5"                  , vcvtph2uqq(zmm6, xmm5));
  TEST_INSTRUCTION("62F57D1879F5"                  , rn_sae().vcvtph2uqq(zmm6, xmm5));
  TEST_INSTRUCTION("62F57D4F79B4F400000010"        , k(k7).vcvtph2uqq(zmm6, xmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F57D587931"                  , vcvtph2uqq(zmm6, word_ptr(ecx)._1to8()));
  TEST_INSTRUCTION("62F57D4879717F"                , vcvtph2uqq(zmm6, xmmword_ptr(ecx, 2032)));
  TEST_INSTRUCTION("62F57DDF797280"                , k(k7).z().vcvtph2uqq(zmm6, word_ptr(edx, -256)._1to8()));
  TEST_INSTRUCTION("62F57C487DF5"                  , vcvtph2uw(zmm6, zmm5));
  TEST_INSTRUCTION("62F57C187DF5"                  , rn_sae().vcvtph2uw(zmm6, zmm5));
  TEST_INSTRUCTION("62F57C4F7DB4F400000010"        , k(k7).vcvtph2uw(zmm6, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F57C587D31"                  , vcvtph2uw(zmm6, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F57C487D717F"                , vcvtph2uw(zmm6, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F57CDF7D7280"                , k(k7).z().vcvtph2uw(zmm6, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F57D487DF5"                  , vcvtph2w(zmm6, zmm5));
  TEST_INSTRUCTION("62F57D187DF5"                  , rn_sae().vcvtph2w(zmm6, zmm5));
  TEST_INSTRUCTION("62F57D4F7DB4F400000010"        , k(k7).vcvtph2w(zmm6, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F57D587D31"                  , vcvtph2w(zmm6, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F57D487D717F"                , vcvtph2w(zmm6, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F57DDF7D7280"                , k(k7).z().vcvtph2w(zmm6, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F57D481DF5"                  , vcvtps2phx(ymm6, zmm5));
  TEST_INSTRUCTION("62F57D181DF5"                  , rn_sae().vcvtps2phx(ymm6, zmm5));
  TEST_INSTRUCTION("62F57D4F1DB4F400000010"        , k(k7).vcvtps2phx(ymm6, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F57D581D31"                  , vcvtps2phx(ymm6, dword_ptr(ecx)._1to16()));
  TEST_INSTRUCTION("62F57D481D717F"                , vcvtps2phx(ymm6, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F57DDF1D7280"                , k(k7).z().vcvtps2phx(ymm6, dword_ptr(edx, -512)._1to16()));
  TEST_INSTRUCTION("62F5FC485BF5"                  , vcvtqq2ph(xmm6, zmm5));
  TEST_INSTRUCTION("62F5FC185BF5"                  , rn_sae().vcvtqq2ph(xmm6, zmm5));
  TEST_INSTRUCTION("62F5FC4F5BB4F400000010"        , k(k7).vcvtqq2ph(xmm6, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F5FC585B31"                  , vcvtqq2ph(xmm6, qword_ptr(ecx)._1to8()));
  TEST_INSTRUCTION("62F5FC485B717F"                , vcvtqq2ph(xmm6, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F5FCDF5B7280"                , k(k7).z().vcvtqq2ph(xmm6, qword_ptr(edx, -1024)._1to8()));
  TEST_INSTRUCTION("62F5D7085AF4"                  , vcvtsd2sh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F5D7185AF4"                  , rn_sae().vcvtsd2sh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F5D70F5AB4F400000010"        , k(k7).vcvtsd2sh(xmm6, xmm5, qword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F5D7085A31"                  , vcvtsd2sh(xmm6, xmm5, qword_ptr(ecx)));
  TEST_INSTRUCTION("62F5D7085A717F"                , vcvtsd2sh(xmm6, xmm5, qword_ptr(ecx, 1016)));
  TEST_INSTRUCTION("62F5D78F5A7280"                , k(k7).z().vcvtsd2sh(xmm6, xmm5, qword_ptr(edx, -1024)));
  TEST_INSTRUCTION("62F556085AF4"                  , vcvtsh2sd(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F556185AF4"                  , sae().vcvtsh2sd(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F5560F5AB4F400000010"        , k(k7).vcvtsh2sd(xmm6, xmm5, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F556085A31"                  , vcvtsh2sd(xmm6, xmm5, word_ptr(ecx)));
  TEST_INSTRUCTION("62F556085A717F"                , vcvtsh2sd(xmm6, xmm5, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F5568F5A7280"                , k(k7).z().vcvtsh2sd(xmm6, xmm5, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F57E082DD6"                  , vcvtsh2si(edx, xmm6));
  TEST_INSTRUCTION("62F57E182DD6"                  , rn_sae().vcvtsh2si(edx, xmm6));
  TEST_INSTRUCTION("62F57E082D94F400000010"        , vcvtsh2si(edx, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F57E082D11"                  , vcvtsh2si(edx, word_ptr(ecx)));
  TEST_INSTRUCTION("62F57E082D517F"                , vcvtsh2si(edx, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F57E082D5280"                , vcvtsh2si(edx, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F6540813F4"                  , vcvtsh2ss(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F6541813F4"                  , sae().vcvtsh2ss(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F6540F13B4F400000010"        , k(k7).vcvtsh2ss(xmm6, xmm5, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F654081331"                  , vcvtsh2ss(xmm6, xmm5, word_ptr(ecx)));
  TEST_INSTRUCTION("62F6540813717F"                , vcvtsh2ss(xmm6, xmm5, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F6548F137280"                , k(k7).z().vcvtsh2ss(xmm6, xmm5, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F57E0879D6"                  , vcvtsh2usi(edx, xmm6));
  TEST_INSTRUCTION("62F57E1879D6"                  , rn_sae().vcvtsh2usi(edx, xmm6));
  TEST_INSTRUCTION("62F57E087994F400000010"        , vcvtsh2usi(edx, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F57E087911"                  , vcvtsh2usi(edx, word_ptr(ecx)));
  TEST_INSTRUCTION("62F57E0879517F"                , vcvtsh2usi(edx, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F57E08795280"                , vcvtsh2usi(edx, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F556082AF2"                  , vcvtsi2sh(xmm6, xmm5, edx));
  TEST_INSTRUCTION("62F556182AF2"                  , rn_sae().vcvtsi2sh(xmm6, xmm5, edx));
  TEST_INSTRUCTION("62F556082AB4F400000010"        , vcvtsi2sh(xmm6, xmm5, dword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F556082A31"                  , vcvtsi2sh(xmm6, xmm5, dword_ptr(ecx)));
  TEST_INSTRUCTION("62F556082A717F"                , vcvtsi2sh(xmm6, xmm5, dword_ptr(ecx, 508)));
  TEST_INSTRUCTION("62F556082A7280"                , vcvtsi2sh(xmm6, xmm5, dword_ptr(edx, -512)));
  TEST_INSTRUCTION("62F554081DF4"                  , vcvtss2sh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F554181DF4"                  , rn_sae().vcvtss2sh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F5540F1DB4F400000010"        , k(k7).vcvtss2sh(xmm6, xmm5, dword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F554081D31"                  , vcvtss2sh(xmm6, xmm5, dword_ptr(ecx)));
  TEST_INSTRUCTION("62F554081D717F"                , vcvtss2sh(xmm6, xmm5, dword_ptr(ecx, 508)));
  TEST_INSTRUCTION("62F5548F1D7280"                , k(k7).z().vcvtss2sh(xmm6, xmm5, dword_ptr(edx, -512)));
  TEST_INSTRUCTION("62F57E485BF5"                  , vcvttph2dq(zmm6, ymm5));
  TEST_INSTRUCTION("62F57E185BF5"                  , sae().vcvttph2dq(zmm6, ymm5));
  TEST_INSTRUCTION("62F57E4F5BB4F400000010"        , k(k7).vcvttph2dq(zmm6, ymmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F57E585B31"                  , vcvttph2dq(zmm6, word_ptr(ecx)._1to16()));
  TEST_INSTRUCTION("62F57E485B717F"                , vcvttph2dq(zmm6, ymmword_ptr(ecx, 4064)));
  TEST_INSTRUCTION("62F57EDF5B7280"                , k(k7).z().vcvttph2dq(zmm6, word_ptr(edx, -256)._1to16()));
  TEST_INSTRUCTION("62F57D487AF5"                  , vcvttph2qq(zmm6, xmm5));
  TEST_INSTRUCTION("62F57D187AF5"                  , sae().vcvttph2qq(zmm6, xmm5));
  TEST_INSTRUCTION("62F57D4F7AB4F400000010"        , k(k7).vcvttph2qq(zmm6, xmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F57D587A31"                  , vcvttph2qq(zmm6, word_ptr(ecx)._1to8()));
  TEST_INSTRUCTION("62F57D487A717F"                , vcvttph2qq(zmm6, xmmword_ptr(ecx, 2032)));
  TEST_INSTRUCTION("62F57DDF7A7280"                , k(k7).z().vcvttph2qq(zmm6, word_ptr(edx, -256)._1to8()));
  TEST_INSTRUCTION("62F57C4878F5"                  , vcvttph2udq(zmm6, ymm5));
  TEST_INSTRUCTION("62F57C1878F5"                  , sae().vcvttph2udq(zmm6, ymm5));
  TEST_INSTRUCTION("62F57C4F78B4F400000010"        , k(k7).vcvttph2udq(zmm6, ymmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F57C587831"                  , vcvttph2udq(zmm6, word_ptr(ecx)._1to16()));
  TEST_INSTRUCTION("62F57C4878717F"                , vcvttph2udq(zmm6, ymmword_ptr(ecx, 4064)));
  TEST_INSTRUCTION("62F57CDF787280"                , k(k7).z().vcvttph2udq(zmm6, word_ptr(edx, -256)._1to16()));
  TEST_INSTRUCTION("62F57D4878F5"                  , vcvttph2uqq(zmm6, xmm5));
  TEST_INSTRUCTION("62F57D1878F5"                  , sae().vcvttph2uqq(zmm6, xmm5));
  TEST_INSTRUCTION("62F57D4F78B4F400000010"        , k(k7).vcvttph2uqq(zmm6, xmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F57D587831"                  , vcvttph2uqq(zmm6, word_ptr(ecx)._1to8()));
  TEST_INSTRUCTION("62F57D4878717F"                , vcvttph2uqq(zmm6, xmmword_ptr(ecx, 2032)));
  TEST_INSTRUCTION("62F57DDF787280"                , k(k7).z().vcvttph2uqq(zmm6, word_ptr(edx, -256)._1to8()));
  TEST_INSTRUCTION("62F57C487CF5"                  , vcvttph2uw(zmm6, zmm5));
  TEST_INSTRUCTION("62F57C187CF5"                  , sae().vcvttph2uw(zmm6, zmm5));
  TEST_INSTRUCTION("62F57C4F7CB4F400000010"        , k(k7).vcvttph2uw(zmm6, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F57C587C31"                  , vcvttph2uw(zmm6, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F57C487C717F"                , vcvttph2uw(zmm6, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F57CDF7C7280"                , k(k7).z().vcvttph2uw(zmm6, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F57D487CF5"                  , vcvttph2w(zmm6, zmm5));
  TEST_INSTRUCTION("62F57D187CF5"                  , sae().vcvttph2w(zmm6, zmm5));
  TEST_INSTRUCTION("62F57D4F7CB4F400000010"        , k(k7).vcvttph2w(zmm6, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F57D587C31"                  , vcvttph2w(zmm6, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F57D487C717F"                , vcvttph2w(zmm6, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F57DDF7C7280"                , k(k7).z().vcvttph2w(zmm6, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F57E082CD6"                  , vcvttsh2si(edx, xmm6));
  TEST_INSTRUCTION("62F57E182CD6"                  , sae().vcvttsh2si(edx, xmm6));
  TEST_INSTRUCTION("62F57E082C94F400000010"        , vcvttsh2si(edx, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F57E082C11"                  , vcvttsh2si(edx, word_ptr(ecx)));
  TEST_INSTRUCTION("62F57E082C517F"                , vcvttsh2si(edx, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F57E082C5280"                , vcvttsh2si(edx, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F57E0878D6"                  , vcvttsh2usi(edx, xmm6));
  TEST_INSTRUCTION("62F57E1878D6"                  , sae().vcvttsh2usi(edx, xmm6));
  TEST_INSTRUCTION("62F57E087894F400000010"        , vcvttsh2usi(edx, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F57E087811"                  , vcvttsh2usi(edx, word_ptr(ecx)));
  TEST_INSTRUCTION("62F57E0878517F"                , vcvttsh2usi(edx, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F57E08785280"                , vcvttsh2usi(edx, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F57F487AF5"                  , vcvtudq2ph(ymm6, zmm5));
  TEST_INSTRUCTION("62F57F187AF5"                  , rn_sae().vcvtudq2ph(ymm6, zmm5));
  TEST_INSTRUCTION("62F57F4F7AB4F400000010"        , k(k7).vcvtudq2ph(ymm6, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F57F587A31"                  , vcvtudq2ph(ymm6, dword_ptr(ecx)._1to16()));
  TEST_INSTRUCTION("62F57F487A717F"                , vcvtudq2ph(ymm6, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F57FDF7A7280"                , k(k7).z().vcvtudq2ph(ymm6, dword_ptr(edx, -512)._1to16()));
  TEST_INSTRUCTION("62F5FF487AF5"                  , vcvtuqq2ph(xmm6, zmm5));
  TEST_INSTRUCTION("62F5FF187AF5"                  , rn_sae().vcvtuqq2ph(xmm6, zmm5));
  TEST_INSTRUCTION("62F5FF4F7AB4F400000010"        , k(k7).vcvtuqq2ph(xmm6, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F5FF587A31"                  , vcvtuqq2ph(xmm6, qword_ptr(ecx)._1to8()));
  TEST_INSTRUCTION("62F5FF487A717F"                , vcvtuqq2ph(xmm6, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F5FFDF7A7280"                , k(k7).z().vcvtuqq2ph(xmm6, qword_ptr(edx, -1024)._1to8()));
  TEST_INSTRUCTION("62F556087BF2"                  , vcvtusi2sh(xmm6, xmm5, edx));
  TEST_INSTRUCTION("62F556187BF2"                  , rn_sae().vcvtusi2sh(xmm6, xmm5, edx));
  TEST_INSTRUCTION("62F556087BB4F400000010"        , vcvtusi2sh(xmm6, xmm5, dword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F556087B31"                  , vcvtusi2sh(xmm6, xmm5, dword_ptr(ecx)));
  TEST_INSTRUCTION("62F556087B717F"                , vcvtusi2sh(xmm6, xmm5, dword_ptr(ecx, 508)));
  TEST_INSTRUCTION("62F556087B7280"                , vcvtusi2sh(xmm6, xmm5, dword_ptr(edx, -512)));
  TEST_INSTRUCTION("62F57F487DF5"                  , vcvtuw2ph(zmm6, zmm5));
  TEST_INSTRUCTION("62F57F187DF5"                  , rn_sae().vcvtuw2ph(zmm6, zmm5));
  TEST_INSTRUCTION("62F57F4F7DB4F400000010"        , k(k7).vcvtuw2ph(zmm6, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F57F587D31"                  , vcvtuw2ph(zmm6, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F57F487D717F"                , vcvtuw2ph(zmm6, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F57FDF7D7280"                , k(k7).z().vcvtuw2ph(zmm6, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F57E487DF5"                  , vcvtw2ph(zmm6, zmm5));
  TEST_INSTRUCTION("62F57E187DF5"                  , rn_sae().vcvtw2ph(zmm6, zmm5));
  TEST_INSTRUCTION("62F57E4F7DB4F400000010"        , k(k7).vcvtw2ph(zmm6, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F57E587D31"                  , vcvtw2ph(zmm6, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F57E487D717F"                , vcvtw2ph(zmm6, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F57EDF7D7280"                , k(k7).z().vcvtw2ph(zmm6, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F554485EF4"                  , vdivph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F554185EF4"                  , rn_sae().vdivph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F5544F5EB4F400000010"        , k(k7).vdivph(zmm6, zmm5, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F554585E31"                  , vdivph(zmm6, zmm5, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F554485E717F"                , vdivph(zmm6, zmm5, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F554DF5E7280"                , k(k7).z().vdivph(zmm6, zmm5, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F556085EF4"                  , vdivsh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F556185EF4"                  , rn_sae().vdivsh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F5560F5EB4F400000010"        , k(k7).vdivsh(xmm6, xmm5, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F556085E31"                  , vdivsh(xmm6, xmm5, word_ptr(ecx)));
  TEST_INSTRUCTION("62F556085E717F"                , vdivsh(xmm6, xmm5, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F5568F5E7280"                , k(k7).z().vdivsh(xmm6, xmm5, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F554485FF4"                  , vmaxph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F554185FF4"                  , sae().vmaxph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F5544F5FB4F400000010"        , k(k7).vmaxph(zmm6, zmm5, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F554585F31"                  , vmaxph(zmm6, zmm5, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F554485F717F"                , vmaxph(zmm6, zmm5, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F554DF5F7280"                , k(k7).z().vmaxph(zmm6, zmm5, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F556085FF4"                  , vmaxsh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F556185FF4"                  , sae().vmaxsh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F5560F5FB4F400000010"        , k(k7).vmaxsh(xmm6, xmm5, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F556085F31"                  , vmaxsh(xmm6, xmm5, word_ptr(ecx)));
  TEST_INSTRUCTION("62F556085F717F"                , vmaxsh(xmm6, xmm5, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F5568F5F7280"                , k(k7).z().vmaxsh(xmm6, xmm5, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F554485DF4"                  , vminph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F554185DF4"                  , sae().vminph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F5544F5DB4F400000010"        , k(k7).vminph(zmm6, zmm5, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F554585D31"                  , vminph(zmm6, zmm5, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F554485D717F"                , vminph(zmm6, zmm5, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F554DF5D7280"                , k(k7).z().vminph(zmm6, zmm5, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F556085DF4"                  , vminsh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F556185DF4"                  , sae().vminsh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F5560F5DB4F400000010"        , k(k7).vminsh(xmm6, xmm5, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F556085D31"                  , vminsh(xmm6, xmm5, word_ptr(ecx)));
  TEST_INSTRUCTION("62F556085D717F"                , vminsh(xmm6, xmm5, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F5568F5D7280"                , k(k7).z().vminsh(xmm6, xmm5, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F5544859F4"                  , vmulph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F5541859F4"                  , rn_sae().vmulph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F5544F59B4F400000010"        , k(k7).vmulph(zmm6, zmm5, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F554585931"                  , vmulph(zmm6, zmm5, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F5544859717F"                , vmulph(zmm6, zmm5, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F554DF597280"                , k(k7).z().vmulph(zmm6, zmm5, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F5560859F4"                  , vmulsh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F5561859F4"                  , rn_sae().vmulsh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F5560F59B4F400000010"        , k(k7).vmulsh(xmm6, xmm5, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F556085931"                  , vmulsh(xmm6, xmm5, word_ptr(ecx)));
  TEST_INSTRUCTION("62F5560859717F"                , vmulsh(xmm6, xmm5, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F5568F597280"                , k(k7).z().vmulsh(xmm6, xmm5, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F554485CF4"                  , vsubph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F554185CF4"                  , rn_sae().vsubph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F5544F5CB4F400000010"        , k(k7).vsubph(zmm6, zmm5, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F554585C31"                  , vsubph(zmm6, zmm5, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F554485C717F"                , vsubph(zmm6, zmm5, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F554DF5C7280"                , k(k7).z().vsubph(zmm6, zmm5, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F556085CF4"                  , vsubsh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F556185CF4"                  , rn_sae().vsubsh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F5560F5CB4F400000010"        , k(k7).vsubsh(xmm6, xmm5, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F556085C31"                  , vsubsh(xmm6, xmm5, word_ptr(ecx)));
  TEST_INSTRUCTION("62F556085C717F"                , vsubsh(xmm6, xmm5, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F5568F5C7280"                , k(k7).z().vsubsh(xmm6, xmm5, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F57C082EF5"                  , vucomish(xmm6, xmm5));
  TEST_INSTRUCTION("62F57C182EF5"                  , sae().vucomish(xmm6, xmm5));
  TEST_INSTRUCTION("62F57C082EB4F400000010"        , vucomish(xmm6, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F57C082E31"                  , vucomish(xmm6, word_ptr(ecx)));
  TEST_INSTRUCTION("62F57C082E717F"                , vucomish(xmm6, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F57C082E7280"                , vucomish(xmm6, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F37C4866EE7B"                , vfpclassph(k5, zmm6, 123));
  TEST_INSTRUCTION("62F37C4F66ACF4000000107B"      , k(k7).vfpclassph(k5, zmmword_ptr(esp, esi, 3, 268435456), 123));
  TEST_INSTRUCTION("62F37C5866297B"                , vfpclassph(k5, word_ptr(ecx)._1to32(), 123));
  TEST_INSTRUCTION("62F37C4866697F7B"              , vfpclassph(k5, zmmword_ptr(ecx, 8128), 123));
  TEST_INSTRUCTION("62F37C5F666A807B"              , k(k7).vfpclassph(k5, word_ptr(edx, -256)._1to32(), 123));
  TEST_INSTRUCTION("62F37C0867EE7B"                , vfpclasssh(k5, xmm6, 123));
  TEST_INSTRUCTION("62F37C0F67ACF4000000107B"      , k(k7).vfpclasssh(k5, word_ptr(esp, esi, 3, 268435456), 123));
  TEST_INSTRUCTION("62F37C0867297B"                , vfpclasssh(k5, word_ptr(ecx), 123));
  TEST_INSTRUCTION("62F37C0867697F7B"              , vfpclasssh(k5, word_ptr(ecx, 254), 123));
  TEST_INSTRUCTION("62F37C0F676A807B"              , k(k7).vfpclasssh(k5, word_ptr(edx, -256), 123));
  TEST_INSTRUCTION("62F67D4842F5"                  , vgetexpph(zmm6, zmm5));
  TEST_INSTRUCTION("62F67D1842F5"                  , sae().vgetexpph(zmm6, zmm5));
  TEST_INSTRUCTION("62F67D4F42B4F400000010"        , k(k7).vgetexpph(zmm6, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F67D584231"                  , vgetexpph(zmm6, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F67D4842717F"                , vgetexpph(zmm6, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F67DDF427280"                , k(k7).z().vgetexpph(zmm6, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F6550843F4"                  , vgetexpsh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F6551843F4"                  , sae().vgetexpsh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F6550F43B4F400000010"        , k(k7).vgetexpsh(xmm6, xmm5, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F655084331"                  , vgetexpsh(xmm6, xmm5, word_ptr(ecx)));
  TEST_INSTRUCTION("62F6550843717F"                , vgetexpsh(xmm6, xmm5, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F6558F437280"                , k(k7).z().vgetexpsh(xmm6, xmm5, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F37C4826F57B"                , vgetmantph(zmm6, zmm5, 123));
  TEST_INSTRUCTION("62F37C1826F57B"                , sae().vgetmantph(zmm6, zmm5, 123));
  TEST_INSTRUCTION("62F37C4F26B4F4000000107B"      , k(k7).vgetmantph(zmm6, zmmword_ptr(esp, esi, 3, 268435456), 123));
  TEST_INSTRUCTION("62F37C5826317B"                , vgetmantph(zmm6, word_ptr(ecx)._1to32(), 123));
  TEST_INSTRUCTION("62F37C4826717F7B"              , vgetmantph(zmm6, zmmword_ptr(ecx, 8128), 123));
  TEST_INSTRUCTION("62F37CDF2672807B"              , k(k7).z().vgetmantph(zmm6, word_ptr(edx, -256)._1to32(), 123));
  TEST_INSTRUCTION("62F3540827F47B"                , vgetmantsh(xmm6, xmm5, xmm4, 123));
  TEST_INSTRUCTION("62F3541827F47B"                , sae().vgetmantsh(xmm6, xmm5, xmm4, 123));
  TEST_INSTRUCTION("62F3540F27B4F4000000107B"      , k(k7).vgetmantsh(xmm6, xmm5, word_ptr(esp, esi, 3, 268435456), 123));
  TEST_INSTRUCTION("62F3540827317B"                , vgetmantsh(xmm6, xmm5, word_ptr(ecx), 123));
  TEST_INSTRUCTION("62F3540827717F7B"              , vgetmantsh(xmm6, xmm5, word_ptr(ecx, 254), 123));
  TEST_INSTRUCTION("62F3548F2772807B"              , k(k7).z().vgetmantsh(xmm6, xmm5, word_ptr(edx, -256), 123));
  TEST_INSTRUCTION("62F67D484CF5"                  , vrcpph(zmm6, zmm5));
  TEST_INSTRUCTION("62F67D4F4CB4F400000010"        , k(k7).vrcpph(zmm6, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F67D584C31"                  , vrcpph(zmm6, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F67D484C717F"                , vrcpph(zmm6, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F67DDF4C7280"                , k(k7).z().vrcpph(zmm6, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F655084DF4"                  , vrcpsh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F6550F4DB4F400000010"        , k(k7).vrcpsh(xmm6, xmm5, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F655084D31"                  , vrcpsh(xmm6, xmm5, word_ptr(ecx)));
  TEST_INSTRUCTION("62F655084D717F"                , vrcpsh(xmm6, xmm5, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F6558F4D7280"                , k(k7).z().vrcpsh(xmm6, xmm5, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F37C4856F57B"                , vreduceph(zmm6, zmm5, 123));
  TEST_INSTRUCTION("62F37C1856F57B"                , sae().vreduceph(zmm6, zmm5, 123));
  TEST_INSTRUCTION("62F37C4F56B4F4000000107B"      , k(k7).vreduceph(zmm6, zmmword_ptr(esp, esi, 3, 268435456), 123));
  TEST_INSTRUCTION("62F37C5856317B"                , vreduceph(zmm6, word_ptr(ecx)._1to32(), 123));
  TEST_INSTRUCTION("62F37C4856717F7B"              , vreduceph(zmm6, zmmword_ptr(ecx, 8128), 123));
  TEST_INSTRUCTION("62F37CDF5672807B"              , k(k7).z().vreduceph(zmm6, word_ptr(edx, -256)._1to32(), 123));
  TEST_INSTRUCTION("62F3540857F47B"                , vreducesh(xmm6, xmm5, xmm4, 123));
  TEST_INSTRUCTION("62F3541857F47B"                , sae().vreducesh(xmm6, xmm5, xmm4, 123));
  TEST_INSTRUCTION("62F3540F57B4F4000000107B"      , k(k7).vreducesh(xmm6, xmm5, word_ptr(esp, esi, 3, 268435456), 123));
  TEST_INSTRUCTION("62F3540857317B"                , vreducesh(xmm6, xmm5, word_ptr(ecx), 123));
  TEST_INSTRUCTION("62F3540857717F7B"              , vreducesh(xmm6, xmm5, word_ptr(ecx, 254), 123));
  TEST_INSTRUCTION("62F3548F5772807B"              , k(k7).z().vreducesh(xmm6, xmm5, word_ptr(edx, -256), 123));
  TEST_INSTRUCTION("62F37C4808F57B"                , vrndscaleph(zmm6, zmm5, 123));
  TEST_INSTRUCTION("62F37C1808F57B"                , sae().vrndscaleph(zmm6, zmm5, 123));
  TEST_INSTRUCTION("62F37C4F08B4F4000000107B"      , k(k7).vrndscaleph(zmm6, zmmword_ptr(esp, esi, 3, 268435456), 123));
  TEST_INSTRUCTION("62F37C5808317B"                , vrndscaleph(zmm6, word_ptr(ecx)._1to32(), 123));
  TEST_INSTRUCTION("62F37C4808717F7B"              , vrndscaleph(zmm6, zmmword_ptr(ecx, 8128), 123));
  TEST_INSTRUCTION("62F37CDF0872807B"              , k(k7).z().vrndscaleph(zmm6, word_ptr(edx, -256)._1to32(), 123));
  TEST_INSTRUCTION("62F354080AF47B"                , vrndscalesh(xmm6, xmm5, xmm4, 123));
  TEST_INSTRUCTION("62F354180AF47B"                , sae().vrndscalesh(xmm6, xmm5, xmm4, 123));
  TEST_INSTRUCTION("62F3540F0AB4F4000000107B"      , k(k7).vrndscalesh(xmm6, xmm5, word_ptr(esp, esi, 3, 268435456), 123));
  TEST_INSTRUCTION("62F354080A317B"                , vrndscalesh(xmm6, xmm5, word_ptr(ecx), 123));
  TEST_INSTRUCTION("62F354080A717F7B"              , vrndscalesh(xmm6, xmm5, word_ptr(ecx, 254), 123));
  TEST_INSTRUCTION("62F3548F0A72807B"              , k(k7).z().vrndscalesh(xmm6, xmm5, word_ptr(edx, -256), 123));
  TEST_INSTRUCTION("62F67D484EF5"                  , vrsqrtph(zmm6, zmm5));
  TEST_INSTRUCTION("62F67D4F4EB4F400000010"        , k(k7).vrsqrtph(zmm6, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F67D584E31"                  , vrsqrtph(zmm6, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F67D484E717F"                , vrsqrtph(zmm6, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F67DDF4E7280"                , k(k7).z().vrsqrtph(zmm6, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F655084FF4"                  , vrsqrtsh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F6550F4FB4F400000010"        , k(k7).vrsqrtsh(xmm6, xmm5, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F655084F31"                  , vrsqrtsh(xmm6, xmm5, word_ptr(ecx)));
  TEST_INSTRUCTION("62F655084F717F"                , vrsqrtsh(xmm6, xmm5, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F6558F4F7280"                , k(k7).z().vrsqrtsh(xmm6, xmm5, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F655482CF4"                  , vscalefph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F655182CF4"                  , rn_sae().vscalefph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F6554F2CB4F400000010"        , k(k7).vscalefph(zmm6, zmm5, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F655582C31"                  , vscalefph(zmm6, zmm5, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F655482C717F"                , vscalefph(zmm6, zmm5, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F655DF2C7280"                , k(k7).z().vscalefph(zmm6, zmm5, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F655082DF4"                  , vscalefsh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F655182DF4"                  , rn_sae().vscalefsh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F6550F2DB4F400000010"        , k(k7).vscalefsh(xmm6, xmm5, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F655082D31"                  , vscalefsh(xmm6, xmm5, word_ptr(ecx)));
  TEST_INSTRUCTION("62F655082D717F"                , vscalefsh(xmm6, xmm5, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F6558F2D7280"                , k(k7).z().vscalefsh(xmm6, xmm5, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F57C4851F5"                  , vsqrtph(zmm6, zmm5));
  TEST_INSTRUCTION("62F57C1851F5"                  , rn_sae().vsqrtph(zmm6, zmm5));
  TEST_INSTRUCTION("62F57C4F51B4F400000010"        , k(k7).vsqrtph(zmm6, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F57C585131"                  , vsqrtph(zmm6, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F57C4851717F"                , vsqrtph(zmm6, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F57CDF517280"                , k(k7).z().vsqrtph(zmm6, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F5560851F4"                  , vsqrtsh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F5561851F4"                  , rn_sae().vsqrtsh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F5560F51B4F400000010"        , k(k7).vsqrtsh(xmm6, xmm5, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F556085131"                  , vsqrtsh(xmm6, xmm5, word_ptr(ecx)));
  TEST_INSTRUCTION("62F5560851717F"                , vsqrtsh(xmm6, xmm5, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F5568F517280"                , k(k7).z().vsqrtsh(xmm6, xmm5, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F6554898F4"                  , vfmadd132ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F6551898F4"                  , rn_sae().vfmadd132ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F6554F98B4F400000010"        , k(k7).vfmadd132ph(zmm6, zmm5, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F655589831"                  , vfmadd132ph(zmm6, zmm5, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F6554898717F"                , vfmadd132ph(zmm6, zmm5, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F655DF987280"                , k(k7).z().vfmadd132ph(zmm6, zmm5, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F6550899F4"                  , vfmadd132sh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F6551899F4"                  , rn_sae().vfmadd132sh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F6550F99B4F400000010"        , k(k7).vfmadd132sh(xmm6, xmm5, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F655089931"                  , vfmadd132sh(xmm6, xmm5, word_ptr(ecx)));
  TEST_INSTRUCTION("62F6550899717F"                , vfmadd132sh(xmm6, xmm5, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F6558F997280"                , k(k7).z().vfmadd132sh(xmm6, xmm5, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F65548A8F4"                  , vfmadd213ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F65518A8F4"                  , rn_sae().vfmadd213ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F6554FA8B4F400000010"        , k(k7).vfmadd213ph(zmm6, zmm5, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F65558A831"                  , vfmadd213ph(zmm6, zmm5, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F65548A8717F"                , vfmadd213ph(zmm6, zmm5, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F655DFA87280"                , k(k7).z().vfmadd213ph(zmm6, zmm5, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F65508A9F4"                  , vfmadd213sh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F65518A9F4"                  , rn_sae().vfmadd213sh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F6550FA9B4F400000010"        , k(k7).vfmadd213sh(xmm6, xmm5, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F65508A931"                  , vfmadd213sh(xmm6, xmm5, word_ptr(ecx)));
  TEST_INSTRUCTION("62F65508A9717F"                , vfmadd213sh(xmm6, xmm5, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F6558FA97280"                , k(k7).z().vfmadd213sh(xmm6, xmm5, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F65548B8F4"                  , vfmadd231ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F65518B8F4"                  , rn_sae().vfmadd231ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F6554FB8B4F400000010"        , k(k7).vfmadd231ph(zmm6, zmm5, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F65558B831"                  , vfmadd231ph(zmm6, zmm5, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F65548B8717F"                , vfmadd231ph(zmm6, zmm5, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F655DFB87280"                , k(k7).z().vfmadd231ph(zmm6, zmm5, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F65508B9F4"                  , vfmadd231sh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F65518B9F4"                  , rn_sae().vfmadd231sh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F6550FB9B4F400000010"        , k(k7).vfmadd231sh(xmm6, xmm5, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F65508B931"                  , vfmadd231sh(xmm6, xmm5, word_ptr(ecx)));
  TEST_INSTRUCTION("62F65508B9717F"                , vfmadd231sh(xmm6, xmm5, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F6558FB97280"                , k(k7).z().vfmadd231sh(xmm6, xmm5, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F6554896F4"                  , vfmaddsub132ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F6551896F4"                  , rn_sae().vfmaddsub132ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F6554F96B4F400000010"        , k(k7).vfmaddsub132ph(zmm6, zmm5, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F655589631"                  , vfmaddsub132ph(zmm6, zmm5, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F6554896717F"                , vfmaddsub132ph(zmm6, zmm5, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F655DF967280"                , k(k7).z().vfmaddsub132ph(zmm6, zmm5, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F65548A6F4"                  , vfmaddsub213ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F65518A6F4"                  , rn_sae().vfmaddsub213ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F6554FA6B4F400000010"        , k(k7).vfmaddsub213ph(zmm6, zmm5, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F65558A631"                  , vfmaddsub213ph(zmm6, zmm5, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F65548A6717F"                , vfmaddsub213ph(zmm6, zmm5, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F655DFA67280"                , k(k7).z().vfmaddsub213ph(zmm6, zmm5, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F65548B6F4"                  , vfmaddsub231ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F65518B6F4"                  , rn_sae().vfmaddsub231ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F6554FB6B4F400000010"        , k(k7).vfmaddsub231ph(zmm6, zmm5, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F65558B631"                  , vfmaddsub231ph(zmm6, zmm5, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F65548B6717F"                , vfmaddsub231ph(zmm6, zmm5, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F655DFB67280"                , k(k7).z().vfmaddsub231ph(zmm6, zmm5, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F655489AF4"                  , vfmsub132ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F655189AF4"                  , rn_sae().vfmsub132ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F6554F9AB4F400000010"        , k(k7).vfmsub132ph(zmm6, zmm5, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F655589A31"                  , vfmsub132ph(zmm6, zmm5, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F655489A717F"                , vfmsub132ph(zmm6, zmm5, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F655DF9A7280"                , k(k7).z().vfmsub132ph(zmm6, zmm5, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F655089BF4"                  , vfmsub132sh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F655189BF4"                  , rn_sae().vfmsub132sh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F6550F9BB4F400000010"        , k(k7).vfmsub132sh(xmm6, xmm5, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F655089B31"                  , vfmsub132sh(xmm6, xmm5, word_ptr(ecx)));
  TEST_INSTRUCTION("62F655089B717F"                , vfmsub132sh(xmm6, xmm5, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F6558F9B7280"                , k(k7).z().vfmsub132sh(xmm6, xmm5, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F65548AAF4"                  , vfmsub213ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F65518AAF4"                  , rn_sae().vfmsub213ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F6554FAAB4F400000010"        , k(k7).vfmsub213ph(zmm6, zmm5, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F65558AA31"                  , vfmsub213ph(zmm6, zmm5, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F65548AA717F"                , vfmsub213ph(zmm6, zmm5, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F655DFAA7280"                , k(k7).z().vfmsub213ph(zmm6, zmm5, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F65508ABF4"                  , vfmsub213sh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F65518ABF4"                  , rn_sae().vfmsub213sh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F6550FABB4F400000010"        , k(k7).vfmsub213sh(xmm6, xmm5, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F65508AB31"                  , vfmsub213sh(xmm6, xmm5, word_ptr(ecx)));
  TEST_INSTRUCTION("62F65508AB717F"                , vfmsub213sh(xmm6, xmm5, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F6558FAB7280"                , k(k7).z().vfmsub213sh(xmm6, xmm5, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F65548BAF4"                  , vfmsub231ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F65518BAF4"                  , rn_sae().vfmsub231ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F6554FBAB4F400000010"        , k(k7).vfmsub231ph(zmm6, zmm5, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F65558BA31"                  , vfmsub231ph(zmm6, zmm5, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F65548BA717F"                , vfmsub231ph(zmm6, zmm5, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F655DFBA7280"                , k(k7).z().vfmsub231ph(zmm6, zmm5, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F65508BBF4"                  , vfmsub231sh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F65518BBF4"                  , rn_sae().vfmsub231sh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F6550FBBB4F400000010"        , k(k7).vfmsub231sh(xmm6, xmm5, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F65508BB31"                  , vfmsub231sh(xmm6, xmm5, word_ptr(ecx)));
  TEST_INSTRUCTION("62F65508BB717F"                , vfmsub231sh(xmm6, xmm5, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F6558FBB7280"                , k(k7).z().vfmsub231sh(xmm6, xmm5, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F6554897F4"                  , vfmsubadd132ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F6551897F4"                  , rn_sae().vfmsubadd132ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F6554F97B4F400000010"        , k(k7).vfmsubadd132ph(zmm6, zmm5, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F655589731"                  , vfmsubadd132ph(zmm6, zmm5, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F6554897717F"                , vfmsubadd132ph(zmm6, zmm5, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F655DF977280"                , k(k7).z().vfmsubadd132ph(zmm6, zmm5, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F65548A7F4"                  , vfmsubadd213ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F65518A7F4"                  , rn_sae().vfmsubadd213ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F6554FA7B4F400000010"        , k(k7).vfmsubadd213ph(zmm6, zmm5, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F65558A731"                  , vfmsubadd213ph(zmm6, zmm5, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F65548A7717F"                , vfmsubadd213ph(zmm6, zmm5, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F655DFA77280"                , k(k7).z().vfmsubadd213ph(zmm6, zmm5, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F65548B7F4"                  , vfmsubadd231ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F65518B7F4"                  , rn_sae().vfmsubadd231ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F6554FB7B4F400000010"        , k(k7).vfmsubadd231ph(zmm6, zmm5, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F65558B731"                  , vfmsubadd231ph(zmm6, zmm5, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F65548B7717F"                , vfmsubadd231ph(zmm6, zmm5, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F655DFB77280"                , k(k7).z().vfmsubadd231ph(zmm6, zmm5, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F655489CF4"                  , vfnmadd132ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F655189CF4"                  , rn_sae().vfnmadd132ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F6554F9CB4F400000010"        , k(k7).vfnmadd132ph(zmm6, zmm5, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F655589C31"                  , vfnmadd132ph(zmm6, zmm5, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F655489C717F"                , vfnmadd132ph(zmm6, zmm5, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F655DF9C7280"                , k(k7).z().vfnmadd132ph(zmm6, zmm5, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F655089DF4"                  , vfnmadd132sh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F655189DF4"                  , rn_sae().vfnmadd132sh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F6550F9DB4F400000010"        , k(k7).vfnmadd132sh(xmm6, xmm5, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F655089D31"                  , vfnmadd132sh(xmm6, xmm5, word_ptr(ecx)));
  TEST_INSTRUCTION("62F655089D717F"                , vfnmadd132sh(xmm6, xmm5, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F6558F9D7280"                , k(k7).z().vfnmadd132sh(xmm6, xmm5, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F65548ACF4"                  , vfnmadd213ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F65518ACF4"                  , rn_sae().vfnmadd213ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F6554FACB4F400000010"        , k(k7).vfnmadd213ph(zmm6, zmm5, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F65558AC31"                  , vfnmadd213ph(zmm6, zmm5, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F65548AC717F"                , vfnmadd213ph(zmm6, zmm5, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F655DFAC7280"                , k(k7).z().vfnmadd213ph(zmm6, zmm5, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F65508ADF4"                  , vfnmadd213sh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F65518ADF4"                  , rn_sae().vfnmadd213sh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F6550FADB4F400000010"        , k(k7).vfnmadd213sh(xmm6, xmm5, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F65508AD31"                  , vfnmadd213sh(xmm6, xmm5, word_ptr(ecx)));
  TEST_INSTRUCTION("62F65508AD717F"                , vfnmadd213sh(xmm6, xmm5, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F6558FAD7280"                , k(k7).z().vfnmadd213sh(xmm6, xmm5, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F65548BCF4"                  , vfnmadd231ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F65518BCF4"                  , rn_sae().vfnmadd231ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F6554FBCB4F400000010"        , k(k7).vfnmadd231ph(zmm6, zmm5, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F65558BC31"                  , vfnmadd231ph(zmm6, zmm5, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F65548BC717F"                , vfnmadd231ph(zmm6, zmm5, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F655DFBC7280"                , k(k7).z().vfnmadd231ph(zmm6, zmm5, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F65508BDF4"                  , vfnmadd231sh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F65518BDF4"                  , rn_sae().vfnmadd231sh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F6550FBDB4F400000010"        , k(k7).vfnmadd231sh(xmm6, xmm5, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F65508BD31"                  , vfnmadd231sh(xmm6, xmm5, word_ptr(ecx)));
  TEST_INSTRUCTION("62F65508BD717F"                , vfnmadd231sh(xmm6, xmm5, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F6558FBD7280"                , k(k7).z().vfnmadd231sh(xmm6, xmm5, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F655489EF4"                  , vfnmsub132ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F655189EF4"                  , rn_sae().vfnmsub132ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F6554F9EB4F400000010"        , k(k7).vfnmsub132ph(zmm6, zmm5, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F655589E31"                  , vfnmsub132ph(zmm6, zmm5, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F655489E717F"                , vfnmsub132ph(zmm6, zmm5, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F655DF9E7280"                , k(k7).z().vfnmsub132ph(zmm6, zmm5, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F655089FF4"                  , vfnmsub132sh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F655189FF4"                  , rn_sae().vfnmsub132sh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F6550F9FB4F400000010"        , k(k7).vfnmsub132sh(xmm6, xmm5, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F655089F31"                  , vfnmsub132sh(xmm6, xmm5, word_ptr(ecx)));
  TEST_INSTRUCTION("62F655089F717F"                , vfnmsub132sh(xmm6, xmm5, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F6558F9F7280"                , k(k7).z().vfnmsub132sh(xmm6, xmm5, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F65548AEF4"                  , vfnmsub213ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F65518AEF4"                  , rn_sae().vfnmsub213ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F6554FAEB4F400000010"        , k(k7).vfnmsub213ph(zmm6, zmm5, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F65558AE31"                  , vfnmsub213ph(zmm6, zmm5, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F65548AE717F"                , vfnmsub213ph(zmm6, zmm5, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F655DFAE7280"                , k(k7).z().vfnmsub213ph(zmm6, zmm5, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F65508AFF4"                  , vfnmsub213sh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F65518AFF4"                  , rn_sae().vfnmsub213sh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F6550FAFB4F400000010"        , k(k7).vfnmsub213sh(xmm6, xmm5, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F65508AF31"                  , vfnmsub213sh(xmm6, xmm5, word_ptr(ecx)));
  TEST_INSTRUCTION("62F65508AF717F"                , vfnmsub213sh(xmm6, xmm5, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F6558FAF7280"                , k(k7).z().vfnmsub213sh(xmm6, xmm5, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F65548BEF4"                  , vfnmsub231ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F65518BEF4"                  , rn_sae().vfnmsub231ph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F6554FBEB4F400000010"        , k(k7).vfnmsub231ph(zmm6, zmm5, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F65558BE31"                  , vfnmsub231ph(zmm6, zmm5, word_ptr(ecx)._1to32()));
  TEST_INSTRUCTION("62F65548BE717F"                , vfnmsub231ph(zmm6, zmm5, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F655DFBE7280"                , k(k7).z().vfnmsub231ph(zmm6, zmm5, word_ptr(edx, -256)._1to32()));
  TEST_INSTRUCTION("62F65508BFF4"                  , vfnmsub231sh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F65518BFF4"                  , rn_sae().vfnmsub231sh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F6550FBFB4F400000010"        , k(k7).vfnmsub231sh(xmm6, xmm5, word_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F65508BF31"                  , vfnmsub231sh(xmm6, xmm5, word_ptr(ecx)));
  TEST_INSTRUCTION("62F65508BF717F"                , vfnmsub231sh(xmm6, xmm5, word_ptr(ecx, 254)));
  TEST_INSTRUCTION("62F6558FBF7280"                , k(k7).z().vfnmsub231sh(xmm6, xmm5, word_ptr(edx, -256)));
  TEST_INSTRUCTION("62F6574856F4"                  , vfcmaddcph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F6571856F4"                  , rn_sae().vfcmaddcph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F6574F56B4F400000010"        , k(k7).vfcmaddcph(zmm6, zmm5, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F657585631"                  , vfcmaddcph(zmm6, zmm5, dword_ptr(ecx)._1to16()));
  TEST_INSTRUCTION("62F6574856717F"                , vfcmaddcph(zmm6, zmm5, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F657DF567280"                , k(k7).z().vfcmaddcph(zmm6, zmm5, dword_ptr(edx, -512)._1to16()));
  TEST_INSTRUCTION("62F6570857F4"                  , vfcmaddcsh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F6571857F4"                  , rn_sae().vfcmaddcsh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F6570F57B4F400000010"        , k(k7).vfcmaddcsh(xmm6, xmm5, dword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F657085731"                  , vfcmaddcsh(xmm6, xmm5, dword_ptr(ecx)));
  TEST_INSTRUCTION("62F6570857717F"                , vfcmaddcsh(xmm6, xmm5, dword_ptr(ecx, 508)));
  TEST_INSTRUCTION("62F6578F577280"                , k(k7).z().vfcmaddcsh(xmm6, xmm5, dword_ptr(edx, -512)));
  TEST_INSTRUCTION("62F65748D6F4"                  , vfcmulcph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F65718D6F4"                  , rn_sae().vfcmulcph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F6574FD6B4F400000010"        , k(k7).vfcmulcph(zmm6, zmm5, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F65758D631"                  , vfcmulcph(zmm6, zmm5, dword_ptr(ecx)._1to16()));
  TEST_INSTRUCTION("62F65748D6717F"                , vfcmulcph(zmm6, zmm5, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F657DFD67280"                , k(k7).z().vfcmulcph(zmm6, zmm5, dword_ptr(edx, -512)._1to16()));
  TEST_INSTRUCTION("62F65708D7F4"                  , vfcmulcsh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F65718D7F4"                  , rn_sae().vfcmulcsh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F6570FD7B4F400000010"        , k(k7).vfcmulcsh(xmm6, xmm5, dword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F65708D731"                  , vfcmulcsh(xmm6, xmm5, dword_ptr(ecx)));
  TEST_INSTRUCTION("62F65708D7717F"                , vfcmulcsh(xmm6, xmm5, dword_ptr(ecx, 508)));
  TEST_INSTRUCTION("62F6578FD77280"                , k(k7).z().vfcmulcsh(xmm6, xmm5, dword_ptr(edx, -512)));
  TEST_INSTRUCTION("62F6564856F4"                  , vfmaddcph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F6561856F4"                  , rn_sae().vfmaddcph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F6564F56B4F400000010"        , k(k7).vfmaddcph(zmm6, zmm5, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F656585631"                  , vfmaddcph(zmm6, zmm5, dword_ptr(ecx)._1to16()));
  TEST_INSTRUCTION("62F6564856717F"                , vfmaddcph(zmm6, zmm5, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F656DF567280"                , k(k7).z().vfmaddcph(zmm6, zmm5, dword_ptr(edx, -512)._1to16()));
  TEST_INSTRUCTION("62F6560857F4"                  , vfmaddcsh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F6561857F4"                  , rn_sae().vfmaddcsh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F6560F57B4F400000010"        , k(k7).vfmaddcsh(xmm6, xmm5, dword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F656085731"                  , vfmaddcsh(xmm6, xmm5, dword_ptr(ecx)));
  TEST_INSTRUCTION("62F6560857717F"                , vfmaddcsh(xmm6, xmm5, dword_ptr(ecx, 508)));
  TEST_INSTRUCTION("62F6568F577280"                , k(k7).z().vfmaddcsh(xmm6, xmm5, dword_ptr(edx, -512)));
  TEST_INSTRUCTION("62F65648D6F4"                  , vfmulcph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F65618D6F4"                  , rn_sae().vfmulcph(zmm6, zmm5, zmm4));
  TEST_INSTRUCTION("62F6564FD6B4F400000010"        , k(k7).vfmulcph(zmm6, zmm5, zmmword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F65658D631"                  , vfmulcph(zmm6, zmm5, dword_ptr(ecx)._1to16()));
  TEST_INSTRUCTION("62F65648D6717F"                , vfmulcph(zmm6, zmm5, zmmword_ptr(ecx, 8128)));
  TEST_INSTRUCTION("62F656DFD67280"                , k(k7).z().vfmulcph(zmm6, zmm5, dword_ptr(edx, -512)._1to16()));
  TEST_INSTRUCTION("62F65608D7F4"                  , vfmulcsh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F65618D7F4"                  , rn_sae().vfmulcsh(xmm6, xmm5, xmm4));
  TEST_INSTRUCTION("62F6560FD7B4F400000010"        , k(k7).vfmulcsh(xmm6, xmm5, dword_ptr(esp, esi, 3, 268435456)));
  TEST_INSTRUCTION("62F65608D731"                  , vfmulcsh(xmm6, xmm5, dword_ptr(ecx)));
  TEST_INSTRUCTION("62F65608D7717F"                , vfmulcsh(xmm6, xmm5, dword_ptr(ecx, 508)));
  TEST_INSTRUCTION("62F6568FD77280"                , k(k7).z().vfmulcsh(xmm6, xmm5, dword_ptr(edx, -512)));
}

static void ASMJIT_NOINLINE testX86AssemblerExtras(AssemblerTester<x86::Assembler>& tester) noexcept {
  using namespace x86;

  // Extended X86 tests.
  TEST_INSTRUCTION("66050201"                      , add(ax, 0x0102));
  TEST_INSTRUCTION("6603849004030201"              , add(ax, ptr(eax, edx, 2, 0x01020304)));
  TEST_INSTRUCTION("0504030201"                    , add(eax, 0x01020304));
  TEST_INSTRUCTION("67668D00"                      , lea(ax, ptr(bx, si)));
  TEST_INSTRUCTION("67668D01"                      , lea(ax, ptr(bx, di)));
  TEST_INSTRUCTION("67668D02"                      , lea(ax, ptr(bp, si)));
  TEST_INSTRUCTION("67668D03"                      , lea(ax, ptr(bp, di)));
  TEST_INSTRUCTION("67668D04"                      , lea(ax, ptr(si)));
  TEST_INSTRUCTION("67668D05"                      , lea(ax, ptr(di)));
  TEST_INSTRUCTION("67668D4600"                    , lea(ax, ptr(bp)));
  TEST_INSTRUCTION("67668D07"                      , lea(ax, ptr(bx)));
  TEST_INSTRUCTION("67668D4010"                    , lea(ax, ptr(bx, si, 0, 0x10)));
  TEST_INSTRUCTION("67668D4120"                    , lea(ax, ptr(bx, di, 0, 0x20)));
  TEST_INSTRUCTION("67668D4240"                    , lea(ax, ptr(bp, si, 0, 0x40)));
  TEST_INSTRUCTION("67668D4360"                    , lea(ax, ptr(bp, di, 0, 0x60)));
  TEST_INSTRUCTION("67668D848000"                  , lea(ax, ptr(si, 0x80)));
  TEST_INSTRUCTION("67668D85A000"                  , lea(ax, ptr(di, 0xA0)));
  TEST_INSTRUCTION("67668D86C000"                  , lea(ax, ptr(bp, 0xC0)));
  TEST_INSTRUCTION("67668D87FF01"                  , lea(ax, ptr(bx, 0x01FF)));
  TEST_INSTRUCTION("678D00"                        , lea(eax, ptr(bx, si)));
  TEST_INSTRUCTION("678D01"                        , lea(eax, ptr(bx, di)));
  TEST_INSTRUCTION("8D0433"                        , lea(eax, ptr(ebx, esi)));
  TEST_INSTRUCTION("8D043B"                        , lea(eax, ptr(ebx, edi)));
  TEST_INSTRUCTION("8D0500000000"                  , lea(eax, ptr(0)));
  TEST_INSTRUCTION("B8FFFFFFFF"                    , mov(eax, 0xFFFFFFFF));
  TEST_INSTRUCTION("8B10"                          , mov(edx, ptr(eax)));
  TEST_INSTRUCTION("8B10"                          , mov(edx, ptr(eax, 0)));
  TEST_INSTRUCTION("8B9080000000"                  , mov(edx, ptr(eax, 128)));
  TEST_INSTRUCTION("8B1408"                        , mov(edx, ptr(eax, ecx)));
  TEST_INSTRUCTION("8B940880000000"                , mov(edx, ptr(eax, ecx, 0, 128)));
  TEST_INSTRUCTION("8B1408"                        , mov(edx, ptr(eax, ecx)));
  TEST_INSTRUCTION("8B544820"                      , mov(edx, ptr(eax, ecx, 1, 32)));
  TEST_INSTRUCTION("8B548840"                      , mov(edx, ptr(eax, ecx, 2, 64)));
  TEST_INSTRUCTION("8B94C800010000"                , mov(edx, ptr(eax, ecx, 3, 128 + 128)));
  TEST_INSTRUCTION("8B1408"                        , mov(edx, ptr(eax, ecx)));
  TEST_INSTRUCTION("8B940880000000"                , mov(edx, ptr(eax, ecx, 0, 128)));
  TEST_INSTRUCTION("8B1408"                        , mov(edx, ptr(eax, ecx)));
  TEST_INSTRUCTION("8B544820"                      , mov(edx, ptr(eax, ecx, 1, 32)));
  TEST_INSTRUCTION("8B54C802"                      , mov(edx, ptr(eax, ecx, 3, 2)));
  TEST_INSTRUCTION("0F20C0"                        , mov(eax, cr0));
  TEST_INSTRUCTION("F00F20C0"                      , mov(eax, cr8));
  TEST_INSTRUCTION("A344332211"                    , mov(ptr(0x11223344), eax));
  TEST_INSTRUCTION("890544332211"                  , mod_mr().mov(ptr(0x11223344), eax));
  TEST_INSTRUCTION("891D44332211"                  , mov(ptr(0x11223344), ebx));
  TEST_INSTRUCTION("0FBE07"                        , movsx(eax, byte_ptr(edi)));
  TEST_INSTRUCTION("0FBF07"                        , movsx(eax, word_ptr(edi)));
  TEST_INSTRUCTION("0FB607"                        , movzx(eax, byte_ptr(edi)));
  TEST_INSTRUCTION("0FB6C6"                        , movzx(eax, dh));
  TEST_INSTRUCTION("0FB707"                        , movzx(eax, word_ptr(edi)));
  TEST_INSTRUCTION("0F95C3"                        , setnz(bl));
  TEST_INSTRUCTION("0F94C7"                        , setz(bh));
  TEST_INSTRUCTION("F600FF"                        , test(byte_ptr(eax), 0xFF));
  TEST_INSTRUCTION("66F700FF00"                    , test(word_ptr(eax), 0xFF));
  TEST_INSTRUCTION("F700FF000000"                  , test(dword_ptr(eax), 0xFF));
  TEST_INSTRUCTION("A836"                          , test(al, 0x36));
  TEST_INSTRUCTION("F6C436"                        , test(ah, 0x36));
  TEST_INSTRUCTION("50"                            , push(eax));
  TEST_INSTRUCTION("51"                            , push(ecx));
  TEST_INSTRUCTION("52"                            , push(edx));
  TEST_INSTRUCTION("53"                            , push(ebx));
  TEST_INSTRUCTION("54"                            , push(esp));
  TEST_INSTRUCTION("55"                            , push(ebp));
  TEST_INSTRUCTION("56"                            , push(esi));
  TEST_INSTRUCTION("57"                            , push(edi));
  TEST_INSTRUCTION("0E"                            , push(cs));
  TEST_INSTRUCTION("16"                            , push(ss));
  TEST_INSTRUCTION("1E"                            , push(ds));
  TEST_INSTRUCTION("06"                            , push(es));
  TEST_INSTRUCTION("0FA0"                          , push(fs));
  TEST_INSTRUCTION("0FA8"                          , push(gs));
  TEST_INSTRUCTION("C8010002"                      , enter(1, 2));
  TEST_INSTRUCTION("FF10"                          , call(ptr(eax)));
  TEST_INSTRUCTION("FF10"                          , call(dword_ptr(eax)));
  TEST_INSTRUCTION("6690"                          , xchg(ax, ax));
  TEST_INSTRUCTION("90"                            , xchg(eax, eax));

  TEST_INSTRUCTION("F00118"                        , lock().add(ptr(eax), ebx));
  TEST_INSTRUCTION("F00FC138"                      , lock().xadd(ptr(eax), edi));
  TEST_INSTRUCTION("F2F00108"                      , xacquire().lock().add(dword_ptr(eax), ecx));
  TEST_INSTRUCTION("F3F00108"                      , xrelease().lock().add(dword_ptr(eax), ecx));

  // MOD RM & MR tests.
  TEST_INSTRUCTION("01CB"                          , mod_mr().add(ebx, ecx));
  TEST_INSTRUCTION("03D9"                          , mod_rm().add(ebx, ecx));
  TEST_INSTRUCTION("88C4"                          , mod_mr().mov(ah, al));
  TEST_INSTRUCTION("88C6"                          , mod_mr().mov(dh, al));
  TEST_INSTRUCTION("89D8"                          , mod_mr().mov(eax, ebx));
  TEST_INSTRUCTION("8AE0"                          , mod_rm().mov(ah, al));
  TEST_INSTRUCTION("8AF0"                          , mod_rm().mov(dh, al));
  TEST_INSTRUCTION("8BC3"                          , mod_rm().mov(eax, ebx));

  // FPU.
  TEST_INSTRUCTION("9B"                            , fwait());
  TEST_INSTRUCTION("D800"                          , fadd(dword_ptr(eax)));
  TEST_INSTRUCTION("DC00"                          , fadd(qword_ptr(eax)));

  // AVX & AVX512.
  TEST_INSTRUCTION("62F17D086EC0"                  , evex().vmovd(xmm0, eax));
  TEST_INSTRUCTION("62F17D087EC0"                  , evex().vmovd(eax, xmm0));
  TEST_INSTRUCTION("62F1F5D95800"                  , k(k1).z().vaddpd(zmm0, zmm1, ptr(eax)._1to8()));
  TEST_INSTRUCTION("62F1748858C2"                  , z().vaddps(xmm0, xmm1, xmm2));
  TEST_INSTRUCTION("62F1748958C2"                  , k(k1).z().vaddps(xmm0, xmm1, xmm2));
  TEST_INSTRUCTION("62F16C4FC25498040F"            , k(k7).vcmpps(k2, zmm2, zmmword_ptr(eax, ebx, 2, 256), 15));
  TEST_INSTRUCTION("62F16C5FC25498400F"            , k(k7).vcmpps(k2, zmm2, dword_ptr(eax, ebx, 2, 256)._1to16(), 15));
  TEST_INSTRUCTION("C4E2F990040500000000"          , vpgatherdq(xmm0, ptr(0, xmm0), xmm0));
  TEST_INSTRUCTION("C4E2FD91040500000000"          , vpgatherqq(ymm0, ptr(0, ymm0), ymm0));
  TEST_INSTRUCTION("C4E2E9920C00"                  , vgatherdpd(xmm1, ptr(eax, xmm0), xmm2));
  TEST_INSTRUCTION("62F26D48CF4C1101"              , vgf2p8mulb(zmm1, zmm2, zmmword_ptr(ecx, edx, 0, 64)));
  TEST_INSTRUCTION("62F3ED48CE4C11010F"            , vgf2p8affineqb(zmm1, zmm2, zmmword_ptr(ecx, edx, 0, 64), 15));
  TEST_INSTRUCTION("62F3ED48CF4C11010F"            , vgf2p8affineinvqb(zmm1, zmm2, zmmword_ptr(ecx, edx, 0, 64), 15));
  TEST_INSTRUCTION("62F2674868246D00F8FFFF"        , vp2intersectd(k4, k5, zmm3, zmmword_ptr(0xFFFFF800, ebp, 1)));

  // AVX_VNNI & AVX512_VNNI.
  TEST_INSTRUCTION("C4E25550F4"                    , vex().vpdpbusd(ymm6, ymm5, ymm4));
  TEST_INSTRUCTION("62F2552850F4"                  , vpdpbusd(ymm6, ymm5, ymm4));
}

/*
// TODO: Failures can be enabled once the assembler always validates.
static void ASMJIT_NOINLINE testX86AssemblerFailures(AssemblerTester<x86::Assembler>& tester) noexcept {
  using namespace x86;

  FAIL_INSTRUCTION(kErrorInvalidInstruction        , movs(byte_ptr(eax), byte_ptr(esi)));
  FAIL_INSTRUCTION(kErrorInvalidInstruction        , movs(word_ptr(eax), word_ptr(esi)));
  FAIL_INSTRUCTION(kErrorInvalidInstruction        , movs(dword_ptr(eax), dword_ptr(esi)));
}
*/

bool testX86Assembler(const TestSettings& settings) noexcept {
  using namespace x86;

  AssemblerTester<Assembler> tester(Arch::kX86, settings);
  tester.printHeader("X86");

  testX86AssemblerBase(tester);
  testX86AssemblerBaseExt(tester);
  testX86AssemblerMMX_SSE(tester);
  testX86AssemblerAVX(tester);
  testX86AssemblerAVX_NE_CONVERT(tester);
  testX86AssemblerAVX_VNNI(tester);
  testX86AssemblerAVX_VNNI_INT8(tester);
  testX86AssemblerAVX_VNNI_INT16(tester);
  testX86AssemblerAVX_SHA512(tester);
  testX86AssemblerAVX_SM3(tester);
  testX86AssemblerAVX_SM4(tester);
  testX86AssemblerFMA(tester);
  testX86AssemblerFMA4(tester);
  testX86AssemblerXOP(tester);
  testX86AssemblerAVX512(tester);
  testX86AssemblerAVX512_FP16(tester);
  testX86AssemblerExtras(tester);

  // testX86AssemblerFailures(tester);

  tester.printSummary();
  return tester.didPass();
}

#undef FAIL_INSTRUCTION
#undef TEST_INSTRUCTION

#endif // !ASMJIT_NO_X86
