// Copyright Earl Warren <contact@earl-warren.org>
// Copyright Loïc Dachary <loic@dachary.org>
// SPDX-License-Identifier: MIT

package forgejo

import (
	"context"
	"fmt"
	"strings"

	"code.forgejo.org/f3/gof3/v3/f3"
	"code.forgejo.org/f3/gof3/v3/id"
	"code.forgejo.org/f3/gof3/v3/kind"
	f3_tree "code.forgejo.org/f3/gof3/v3/tree/f3"
	"code.forgejo.org/f3/gof3/v3/tree/generic"
	"code.forgejo.org/f3/gof3/v3/util"
)

type ownerInfo struct {
	kind kind.Kind
	name string
}

type forge struct {
	common

	ownersInfo map[string]ownerInfo
}

func newForge() generic.NodeDriverInterface {
	return &forge{
		ownersInfo: make(map[string]ownerInfo),
	}
}

func (o *forge) getOwnersKind(ctx context.Context, id string) kind.Kind {
	return o.getOwnersInfo(ctx, id).kind
}

func (o *forge) getOwnersName(ctx context.Context, id string) string {
	return o.getOwnersInfo(ctx, id).name
}

func (o *forge) getOwnersInfo(ctx context.Context, id string) ownerInfo {
	info, ok := o.ownersInfo[id]
	if !ok {
		user, _, err := o.getClient().GetUserByID(util.ParseInt(id))
		if err != nil {
			if strings.Contains(err.Error(), "user not found") {
				organizations := o.getChildDriver(f3_tree.KindOrganizations).(*organizations)
				info.name = organizations.getNameFromID(ctx, util.ParseInt(id))
				if info.name != "" {
					info.kind = f3_tree.KindOrganizations
				} else {
					panic(fmt.Errorf("%v does not match any user or organization", id))
				}
			} else {
				panic(fmt.Errorf("GetUserByID(%s) failed %w", id, err))
			}
		} else {
			info.kind = f3_tree.KindUsers
			info.name = user.UserName
		}
		o.ownersInfo[id] = info
	}
	return info
}

func (o *forge) getOwnersPath(ctx context.Context, id string) f3_tree.Path {
	return f3_tree.NewPathFromString("/").SetForge().SetOwners(o.getOwnersKind(ctx, id))
}

func (o *forge) Equals(context.Context, generic.NodeInterface) bool { return true }
func (o *forge) Get(context.Context) bool                           { return true }
func (o *forge) Put(context.Context) id.NodeID                      { return id.NewNodeID("forge") }
func (o *forge) Patch(context.Context)                              {}
func (o *forge) Delete(context.Context)                             {}
func (o *forge) NewFormat() f3.Interface                            { return &f3.Forge{} }
func (o *forge) FromFormat(f3.Interface)                            {}

func (o *forge) ToFormat() f3.Interface {
	return &f3.Forge{
		Common: f3.NewCommon("forge"),
		URL:    o.String(),
	}
}

func (o *forge) String() string {
	options := o.GetTreeDriver().(*treeDriver).options
	return options.GetURL()
}
