/*
 This file is part of GNU Taler
 (C) 2021 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  AccessToken,
  HttpStatusCode,
  LoginTokenScope,
  MerchantAuthMethod,
  succeedOrThrow,
  TalerMerchantInstanceHttpClient,
  TalerMerchantManagementHttpClient,
  URL,
} from "@gnu-taler/taler-util";
import {
  ExchangeService,
  getTestHarnessPaytoForLabel,
  GlobalTestState,
  harnessHttpLib,
  MERCHANT_DEFAULT_AUTH,
  MERCHANT_DEFAULT_LOGIN_SCOPE,
  MerchantService,
  setupDb,
} from "../harness/harness.js";

/**
 * Do basic checks on instance management and authentication.
 */
export async function runMerchantInstancesTest(t: GlobalTestState) {
  // Set up test environment

  const db = await setupDb(t);

  const exchange = ExchangeService.create(t, {
    name: "testexchange-1",
    currency: "TESTKUDOS",
    httpPort: 8081,
    database: db.connStr,
  });

  const merchant = await MerchantService.create(t, {
    name: "testmerchant-1",
    httpPort: 8083,
    database: db.connStr,
  });

  // We add the exchange to the config, but note that the exchange won't be started.
  merchant.addExchange(exchange);

  await merchant.start();
  await merchant.pingUntilAvailable();

  // Base URL for the default instance.
  const baseUrl = merchant.makeInstanceBaseUrl();

  {
    const r = await harnessHttpLib.fetch(new URL("config", baseUrl).href);
    const data = await r.json();
    console.log(data);
    t.assertDeepEqual(data.currency, "TESTKUDOS");
  }

  // Instances should initially be empty
  {
    const r = await harnessHttpLib.fetch(
      new URL("management/instances", baseUrl).href,
    );
    const data = await r.json();
    t.assertDeepEqual(data.instances, []);
  }

  // Add an instance, no auth!
  const { accessToken: adminAccessToken } = await merchant.addInstanceWithWireAccount({
    id: "admin",
    name: "Default Instance",
    paytoUris: [getTestHarnessPaytoForLabel("merchant-default")],
    auth: MERCHANT_DEFAULT_AUTH,
  });

  // Add it again, should be idempotent
  // FIXME: this fails, means we lost idempotentcy with password auth?
  // await merchant.addInstanceWithWireAccount({
  //   id: "admin",
  //   name: "Default Instance",
  //   paytoUris: [getTestHarnessPaytoForLabel("merchant-default")],
  //   auth: MERCHANT_DEFAULT_AUTH,
  // });

  // Add an instance, no auth!
  const {accessToken: myInstAccessToken} = await merchant.addInstanceWithWireAccount({
    id: "myinst",
    name: "Second Instance",
    paytoUris: [getTestHarnessPaytoForLabel("merchant-default")],
    auth: MERCHANT_DEFAULT_AUTH,
  }, {adminAccessToken});

  const merchantAdminAccessToken = adminAccessToken;

  let merchantClient = new TalerMerchantManagementHttpClient(
    merchant.makeInstanceBaseUrl(),
  );

  {
    const r = succeedOrThrow(await merchantClient.listInstances(merchantAdminAccessToken));
    t.assertDeepEqual(r.instances.length, 2);
  }

  {
    const fullDetails = succeedOrThrow(
      await merchantClient.getInstanceDetails(merchantAdminAccessToken, "admin"),
    );
    t.assertDeepEqual(fullDetails.auth.method, "token");
  }

  await merchantClient.updateCurrentInstanceAuthentication(merchantAdminAccessToken, {
    method: MerchantAuthMethod.TOKEN,
    password: "foobar",
  });

  const { access_token: auth } = succeedOrThrow(
    await merchantClient.createAccessToken("admin", "foobar", {
      scope: LoginTokenScope.All,
    }),
  );

  console.log("requesting instances with no auth");
  const exc = await merchantClient.listInstances(undefined);
  t.assertTrue(exc.type === "fail");
  t.assertTrue(exc.case === HttpStatusCode.Unauthorized);

  // With the new client auth settings, request should work again.
  succeedOrThrow(await merchantClient.listInstances(auth));

  // Now, try some variations.
  {
    const url = merchant.makeInstanceBaseUrl();
    const resp = await harnessHttpLib.fetch(
      new URL("management/instances", url).href,
      {
        headers: {
          // Note the spaces
          Authorization: `Bearer     ${auth}`,
        },
      },
    );
    t.assertDeepEqual(resp.status, 200);
  }

  // Check that auth is reported properly
  {
    const fullDetails = succeedOrThrow(
      await merchantClient.getInstanceDetails(auth, "admin"),
    );
    t.assertDeepEqual(fullDetails.auth.method, "token");
    // Token should *not* be reported back.
    t.assertDeepEqual((fullDetails.auth as any).token, undefined);
  }

  // Check that deleting an instance checks the auth
  // of the default instance.
  {
    const res = await merchantClient.deleteInstance(myInstAccessToken, "myinst");
    t.assertTrue(res.type === "fail");
    t.assertTrue(res.case === HttpStatusCode.Unauthorized);
  }
}

runMerchantInstancesTest.suites = ["merchant"];
