/*
  This file is part of TALER
  Copyright (C) 2021-2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU Affero General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU Affero General Public License for more details.

  You should have received a copy of the GNU Affero General Public License along with
  TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
*/
/**
 * @file taler-merchant-httpd_auth.h
 * @brief request authentication logic
 * @author Florian Dold
 * @author Martin Schanzenbach
 * @author Christian Grothoff
 */
#ifndef TALER_MERCHANT_HTTPD_AUTH_H
#define TALER_MERCHANT_HTTPD_AUTH_H

#include "taler-merchant-httpd.h"

/**
 * Check that @a token hashes to @a hash under @a salt for
 * merchant instance authentication.
 *
 * @param token the token to check
 * @param salt the salt to use when hashing
 * @param hash the hash to check against
 * @return #GNUNET_OK if the @a token matches
 */
enum GNUNET_GenericReturnValue
TMH_check_auth (const char *token,
                struct TALER_MerchantAuthenticationSaltP *salt,
                struct TALER_MerchantAuthenticationHashP *hash);


/**
 * Compute a @a hash from @a token hashes for
 * merchant instance authentication.
 *
 * @param password the password to check
 * @param[out] salt set to a fresh random salt
 * @param[out] hash set to the hash of @a token under @a salt
 */
void
TMH_compute_auth (const char *password,
                  struct TALER_MerchantAuthenticationSaltP *salt,
                  struct TALER_MerchantAuthenticationHashP *hash);


/**
 * Check if @a candidate permissions are a subset of @a as permissions
 *
 * @param as scope to check against
 * @param candidate scope to check if its permissions are a subset of @a as permissions.
 * @return true if it was a subset, false otherwise.
 */
bool
TMH_scope_is_subset (enum TMH_AuthScope as,
                     enum TMH_AuthScope candidate);


/**
 * Return the TMH_AuthScope corresponding to @a name.
 *
 * @param name the name to look for
 * @return the scope corresponding to the name, or TMH_AS_NONE.
 */
enum TMH_AuthScope
TMH_get_scope_by_name (const char *name);


/**
 * Return the name corresponding to @a scop.
 *
 * @param scope the scope to look for
 * @param[out] refreshable outputs if scope value was refreshable
 * @return the name corresponding to the scope, or NULL.
 */
const char *
TMH_get_name_by_scope (enum TMH_AuthScope scope,
                       bool *refreshable);


/**
 * Check if the client has provided the necessary credentials
 * to access the selected endpoint of the selected instance.
 *
 * @param[in,out] hc handler context
 * @return #GNUNET_OK on success,
 *         #GNUNET_NO if an error was queued (return #MHD_YES)
 *         #GNUNET_SYSERR to close the connection (return #MHD_NO)
 */
enum GNUNET_GenericReturnValue
TMH_perform_access_control (struct TMH_HandlerContext *hc);

#endif
